import { Tuple4d } from './Tuple4d';
/**
 * A 4 element vector that is represented by double precision floating point
 * x,y,z,w coordinates.
 * @version specification 1.1, implementation $Revision: 1.9 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Vector4d extends Tuple4d {
    /**
     * Sets the x,y,z components of this point to the corresponding
     * components of tuple t1. The w component of this point is set to 1.
     * @param t1 the tuple to be copied
     * @since Java3D 1.2
     */
    set_tuple3(t1) {
        this.set_xyzw(t1.x, t1.y, t1.z, 0);
    }
    /**
     * Returns the squared length of this vector.
     * @return the squared length of this vector
     */
    lengthSquared() {
        return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
    }
    /**
     * Returns the length of this vector.
     * @return the length of this vector
     */
    length() {
        return Math.sqrt(this.lengthSquared());
    }
    /**
     * Computes the dot product of the this vector and vector v1.
     * @param v1 the other vector
     * @return the dot product of this vector and vector v1
     */
    dot(v1) {
        return this.x * v1.x + this.y * v1.y + this.z * v1.z + this.w * v1.w;
    }
    /**
     * Sets the value of this vector to the normalization of vector v1.
     * @param v1 the un-normalized vector
     * Normalizes this vector in place.
     */
    normalize(v1) {
        if (v1)
            this.set_tuple(v1);
        let d = this.length();
        this.x /= d;
        this.y /= d;
        this.z /= d;
        this.w /= d;
    }
    /**
     * Returns the (4-space) angle in radians between this vector and
     * the vector parameter; the return value is constrained to the
     * range [0,PI].
     * @param v1  the other vector
     * @return the angle in radians in the range [0,PI]
     */
    angle(v1) {
        let d = this.dot(v1);
        let v1Length = v1.length();
        let vLength = this.length();
        return Math.acos(d / v1Length / vLength);
    }
}
//# sourceMappingURL=Vector4d.js.map