/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { Tuple3d } from './Tuple3d';
/**
 * A 3 element vector that is represented by double precision floating point
 * x,y,z coordinates. If this value represents a normal, then it should be
 * normalized.
 * @version specification 1.1, implementation $Revision: 1.8 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Vector3d extends Tuple3d {
    /**
     * Sets this vector to be the vector cross product of vectors v1 and v2.
     * @param v1 the first vector
     * @param v2 the second vector
     */
    cross(v1, v2) {
        this.set_xyz(v1.y * v2.z - v1.z * v2.y, v1.z * v2.x - v1.x * v2.z, v1.x * v2.y - v1.y * v2.x);
    }
    /**
     * Sets the value of this vector to the normalization of vector v1.
     * @param v1 the un-normalized vector
     * Normalizes this vector in place.
     */
    normalize(v1) {
        if (v1)
            this.set_tuple(v1);
        let d = this.length();
        this.x /= d;
        this.y /= d;
        this.z /= d;
    }
    /**
     * Computes the dot product of the this vector and vector v1.
     * @param v1 the other vector
     */
    dot(v1) {
        return this.x * v1.x + this.y * v1.y + this.z * v1.z;
    }
    /**
     * Returns the squared length of this vector.
     * @return the squared length of this vector
     */
    lengthSquared() {
        return this.x * this.x + this.y * this.y + this.z * this.z;
    }
    /**
     * Returns the length of this vector.
     * @return the length of this vector
     */
    length() {
        return Math.sqrt(this.lengthSquared());
    }
    /**
     * Returns the angle in radians between this vector and
     * the vector parameter; the return value is constrained to the
     * range [0,PI].
     * @param v1  the other vector
     * @return the angle in radians in the range [0,PI]
     */
    angle(v1) {
        let xx = this.y * v1.z - this.z * v1.y;
        let yy = this.z * v1.x - this.x * v1.z;
        let zz = this.x * v1.y - this.y * v1.x;
        let cross = Math.sqrt(xx * xx + yy * yy + zz * zz);
        return Math.abs(Math.atan2(cross, this.dot(v1)));
    }
}
//# sourceMappingURL=Vector3d.js.map