/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { GMatrix } from './GMatrix';
import { GVector } from './GVector';
import { Vector3d } from './Vector3d';
import { Point3d } from './Point3d';
import { Quat4d } from './Quat4d';
import { AxisAngle4d } from './AxisAngle4d';
import { Vector4d } from './Vector4d';
import { Matrix3d } from './Matrix3d';
import { Matrix4d } from './Matrix4d';
//module jp.kitec.vecmath
/**
 * java.vecmath Test class.
 * <pre>
 * % java java.vecmath.VecmathTest
 * </pre>
 * to run this test.
 * If you find any bugs, please add a test method to reproduce the bug,
 * and insert the method call to this main driver.
 */
const NL = "\n";
const epsilon = 1.0e-5;
function equals(m1, m2) {
    return Math.abs(m1 - m2) < epsilon;
}
function equalsM3(m1, m2) {
    return m1.epsilonEquals(m2, epsilon);
}
function equalsM4(m1, m2) {
    return m1.epsilonEquals(m2, epsilon);
}
function equalsT4(m1, m2) {
    return m1.epsilonEquals(m2, epsilon);
}
function equalsT3(m1, m2) {
    return m1.epsilonEquals(m2, epsilon);
}
function equalsGM(m1, m2) {
    return m1.epsilonEquals(m2, epsilon);
}
function equalsGV(v1, v2) {
    return v1.epsilonEquals(v2, epsilon);
}
function equalsA4(a1, a2) {
    if (0 < a1.x * a2.x + a1.y * a2.y + a1.z * a2.z) {
        return equals(a1.y * a2.z - a1.z * a2.y, 0) && equals(a1.z * a2.x - a1.x * a2.z, 0) &&
            equals(a1.x * a2.y - a1.y * a2.x, 0) && equals(a1.angle, a2.angle);
    }
    else {
        return equals(a1.y * a2.z - a1.z * a2.y, 0) && equals(a1.z * a2.x - a1.x * a2.z, 0) &&
            equals(a1.x * a2.y - a1.y * a2.x, 0) &&
            (equals(a1.angle, -a2.angle) || equals(a1.angle + a2.angle, 2 * Math.PI) || equals(a1.angle + a2.angle, -2 * Math.PI));
    }
}
QUnit.test("Vector3d ...", () => Vector3dTest());
QUnit.test("Matrix3d with Quat4d, AxisAngle4d, Point/Vector3d interaction ...", () => Matrix3dTest());
QUnit.test("Matrix4d with Quat4d, AxisAngle4d, Point/Vector3d interaction ...", () => Matrix4dTest());
QUnit.test("GMatrix with GVector interaction ...", () => GMatrixTest());
QUnit.test("SVD test ...", () => SVDTest());
function Vector3dTest() {
    let zeroVector = new Vector3d();
    let v1 = new Vector3d(2, 3, 4);
    let v2 = new Vector3d(2, 5, -8);
    let v3 = new Vector3d();
    v3.cross(v1, v2);
    QUnit.ok(equals(v3.dot(v1), 0));
    QUnit.ok(equals(v3.dot(v2), 0));
    v1.cross(v1, v2);
    QUnit.ok(equalsT3(v1, new Vector3d(-44, 24, 4)));
    QUnit.ok(equals(v2.lengthSquared(), 93));
    QUnit.ok(equals(v2.length(), Math.sqrt(93)));
    v1.set_tuple(v2);
    v2.normalize();
    QUnit.ok(equals(v2.length(), 1));
    v1.cross(v2, v1);
    QUnit.ok(equalsT3(v1, zeroVector));
    v1.set_xyz(1, 2, 3);
    v2.set_xyz(-1, -6, -3);
    let ang = v1.angle(v2);
    QUnit.ok(equals(v1.length() * v2.length() * Math.cos(ang), v1.dot(v2)));
    v1.set_tuple(v2);
    ang = v1.angle(v2);
    QUnit.ok(equals(ang, 0));
    QUnit.ok(equals(v1.length() * v2.length() * Math.cos(ang), v1.dot(v2)));
    v1.set_xyz(1, 2, 3);
    v2.set_xyz(1, 2, 3.00001);
    ang = v1.angle(v2);
    QUnit.ok(equals(v1.length() * v2.length() * Math.cos(ang), v1.dot(v2)));
    v1.set_xyz(1, 2, 3);
    v2.set_xyz(-1, -2, -3.00001);
    ang = v1.angle(v2);
    QUnit.ok(equals(v1.length() * v2.length() * Math.cos(ang), v1.dot(v2)));
}
function Matrix3dTest() {
    let O = new Matrix3d();
    let I = new Matrix3d();
    I.setIdentity();
    let m1 = new Matrix3d();
    let m2 = new Matrix3d();
    let v = [2, 1, 4, 1, -2, 3, -3, -1, 1];
    for (let i__0 = 0; i__0 < 3; i__0++) {
        for (let j__0 = 0; j__0 < 3; j__0++)
            m1.setElement(i__0, j__0, i__0 * 2 * j__0 + 3);
    }
    for (let i__1 = 0; i__1 < 3; i__1++) {
        for (let j__1 = 0; j__1 < 3; j__1++)
            QUnit.ok(equals(m1.getElement(i__1, j__1), i__1 * 2 * j__1 + 3));
    }
    m1.set_array(v);
    m2 = new Matrix3d(m1);
    m2.mul_matrix(O);
    QUnit.ok(equalsM3(m2, O));
    m2.mul_matrix(m1, I);
    QUnit.ok(equalsM3(m2, m1));
    QUnit.ok(equals(m1.determinant(), -36));
    m2.negate(m1);
    m2.add(m1);
    QUnit.ok(equalsM3(m2, O));
    m2.negate(m1);
    let m3 = new Matrix3d(m1);
    m3.sub(m2);
    m3.mul_scalar(0.5);
    QUnit.ok(equalsM3(m1, m3));
    m3.invert(m2);
    m3.mul_matrix(m2);
    QUnit.ok(equalsM3(m3, I));
    let p1 = new Point3d(1, 2, 3);
    let v1 = new Vector3d(2, -1, -4);
    p1.set_xyz(1, 0, 0);
    m1.rotZ(Math.PI / 6);
    m1.transform(p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 6), Math.sin(Math.PI / 6), 0)));
    p1.set_xyz(1, 0, 0);
    m1.rotY(Math.PI / 3);
    m1.transform(p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    let a1 = new AxisAngle4d(0, 1, 0, Math.PI / 3);
    p1.set_xyz(1, 0, 0);
    m1.set_axisAngle(a1);
    m1.transform(p1, p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    let q1 = new Quat4d();
    p1.set_xyz(1, 0, 0);
    q1.set_axisAngle(a1);
    m2.set_quat(q1);
    QUnit.ok(equalsM3(m1, m2));
    m2.transform(p1, p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    a1.set_xyz_angle(1, 2, -3, Math.PI / 3);
    Mat3dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, 2, 3, Math.PI);
    Mat3dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, .1, .1, Math.PI);
    Mat3dQuatAxisAngle(a1);
    a1.set_xyz_angle(.1, 1, .1, Math.PI);
    Mat3dQuatAxisAngle(a1);
    a1.set_xyz_angle(.1, .1, 1, Math.PI);
    Mat3dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, 1, 1, 2 * Math.PI / 3);
    m1.set_axisAngle(a1);
    p1.set_xyz(1, 0, 0);
    m1.transform(p1);
    QUnit.ok(equalsT3(p1, new Point3d(0, 1, 0)));
    m1.transform(p1);
    QUnit.ok(equalsT3(p1, new Point3d(0, 0, 1)));
    m1.transform(p1);
    QUnit.ok(equalsT3(p1, new Point3d(1, 0, 0)));
    m1.set_axisAngle(a1);
    QUnit.ok(equals(m1.determinant(), 1));
    QUnit.ok(equals(m1.getScale(), 1));
    m2.set_axisAngle(a1);
    m2.normalize();
    QUnit.ok(equalsM3(m1, m2));
    m2.set_axisAngle(a1);
    m2.normalizeCP();
    QUnit.ok(equalsM3(m1, m2));
    let scale = 3.0;
    m2.rotZ(-Math.PI / 4);
    m2.mul_scalar(scale);
    QUnit.ok(equals(m2.determinant(), scale * scale * scale));
    QUnit.ok(equals(m2.getScale(), scale));
    m2.normalize();
    QUnit.ok(equals(m2.determinant(), 1));
    QUnit.ok(equals(m2.getScale(), 1));
    m2.rotX(Math.PI / 3);
    m2.mul_scalar(scale);
    QUnit.ok(equals(m2.determinant(), scale * scale * scale));
    QUnit.ok(equals(m2.getScale(), scale));
    m2.normalizeCP();
    QUnit.ok(equals(m2.determinant(), 1));
    QUnit.ok(equals(m2.getScale(), 1));
    m1.set_axisAngle(a1);
    m2.invert(m1);
    m1.transpose();
    QUnit.ok(equalsM3(m1, m2));
}
function Mat3dQuatAxisAngle(a1) {
    let m1 = new Matrix3d();
    let m2 = new Matrix3d();
    let a2 = new AxisAngle4d();
    let q1 = new Quat4d();
    let q2 = new Quat4d();
    q1.set_axisAngle(a1);
    a2.set_quat(q1);
    QUnit.ok(equalsA4(a1, a2));
    q2 = new Quat4d();
    q2.set_axisAngle(a2);
    QUnit.ok(equalsT4(q1, q2));
    q1.set_axisAngle(a1);
    m1.set_quat(q1);
    q2.set_matrix3(m1);
    QUnit.ok(equalsT4(q1, q2));
    m2.set_quat(q2);
    QUnit.ok(equalsM3(m1, m2));
    m1.set_axisAngle(a1);
    a2.set_matrix3(m1);
    QUnit.ok(equalsA4(a1, a2));
    m2.set_axisAngle(a1);
    QUnit.ok(equalsM3(m1, m2));
    a1.x *= 2;
    a1.y *= 2;
    a1.z *= 2;
    m2.set_axisAngle(a1);
    a1.x = -a1.x;
    a1.y = -a1.y;
    a1.z = -a1.z;
    a1.angle = -a1.angle;
    m2.set_axisAngle(a1);
    QUnit.ok(equalsM3(m1, m2));
}
function Mat4dQuatAxisAngle(a1) {
    let m1 = new Matrix4d();
    let m2 = new Matrix4d();
    let a2 = new AxisAngle4d();
    let q1 = new Quat4d();
    let q2 = new Quat4d();
    q1.set_axisAngle(a1);
    a2.set_quat(q1);
    QUnit.ok(equalsA4(a1, a2));
    q2 = new Quat4d();
    q2.set_axisAngle(a2);
    QUnit.ok(equalsT4(q1, q2));
    q1.set_axisAngle(a1);
    m1.set_quat(q1);
    q2.set_matrix4(m1);
    QUnit.ok(equalsT4(q1, q2));
    m2.set_quat(q2);
    QUnit.ok(equalsM4(m1, m2));
    m1.set_axisAngle(a1);
    a2.set_matrix4(m1);
    QUnit.ok(equalsA4(a1, a2));
    m2.set_axisAngle(a1);
    QUnit.ok(equalsM4(m1, m2));
    a1.x *= 2;
    a1.y *= 2;
    a1.z *= 2;
    m2.set_axisAngle(a1);
    a1.x = -a1.x;
    a1.y = -a1.y;
    a1.z = -a1.z;
    a1.angle = -a1.angle;
    m2.set_axisAngle(a1);
    QUnit.ok(equalsM4(m1, m2));
}
function Matrix4dTest() {
    let O = new Matrix4d();
    let I = new Matrix4d();
    I.setIdentity();
    let m1 = new Matrix4d();
    let m2 = new Matrix4d();
    for (let i__0 = 0; i__0 < 4; i__0++) {
        for (let j__0 = 0; j__0 < 4; j__0++)
            m1.setElement(i__0, j__0, i__0 * 2 * j__0 + 3);
    }
    for (let i__1 = 0; i__1 < 4; i__1++) {
        for (let j__1 = 0; j__1 < 4; j__1++)
            QUnit.ok(equals(m1.getElement(i__1, j__1), i__1 * 2 * j__1 + 3));
    }
    m1 = new Matrix4d(2, 1, 4, 1, -2, 3, -3, 1, -1, 1, 2, 2, 0, 8, 1, -10);
    m2 = new Matrix4d(m1);
    m2.mul_matrix(O);
    QUnit.ok(equalsM4(m2, O), "O = m2 x O");
    m2.mul_matrix(m1, I);
    QUnit.ok(equalsM4(m2, m1), "m2 = m1 x I");
    m2.negate(m1);
    m2.add_matrix(m1);
    QUnit.ok(equalsM4(m2, O));
    let v = [5, 1, 4, 0, 2, 3, -4, -1, 2, 3, -4, -1, 1, 1, 1, 1];
    m2.set_array(v);
    m2.negate(m1);
    let m3 = new Matrix4d(m1);
    m3.sub(m2);
    m3.mul_scalar(0.5);
    QUnit.ok(equalsM4(m1, m3));
    m2 = new Matrix4d(.5, 1, 4, 1, -2, 3, -4, -1, 1, 9, 100, 2, -20, 2, 1, 9);
    m3.invert(m2);
    m3.mul_matrix(m2);
    QUnit.ok(equalsM4(m3, I));
    m1 = new Matrix4d(-1, 2, 0, 3, -1, 1, -3, -1, 1, 2, 1, 1, 0, 0, 0, 1);
    let p1 = new Point3d(1, 2, 3);
    let v0 = new Vector3d();
    let v1 = new Vector3d(1, 2, 3);
    let V2 = new Vector4d(2, -1, -4, 1);
    QUnit.equal(m1.toString(), ("[" + NL + "  [-1	2	0	3]" + NL + "  [-1	1	-3	-1]"
        + NL + "  [1	2	1	1]" + NL + "  [0	0	0	1] ]"));
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(6, -9, 9)));
    m1.transform(V2, V2);
    QUnit.ok(equalsT4(V2, new Vector4d(-1, 8, -3, 1)));
    p1.set_xyz(1, 0, 0);
    m1.rotZ(Math.PI / 6);
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 6), Math.sin(Math.PI / 6), 0)));
    p1.set_xyz(1, 0, 0);
    m1.rotY(Math.PI / 3);
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    let a1 = new AxisAngle4d(0, 1, 0, Math.PI / 3);
    p1.set_xyz(1, 0, 0);
    m1.set_axisAngle(a1);
    m1.transformPoint(p1, p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    let q1 = new Quat4d();
    p1.set_xyz(1, 0, 0);
    q1.set_axisAngle(a1);
    m2.set_quat(q1);
    QUnit.ok(equalsM4(m1, m2));
    m2.transformPoint(p1, p1);
    QUnit.ok(equalsT3(p1, new Point3d(Math.cos(Math.PI / 3), 0, -Math.sin(Math.PI / 3))));
    a1.set_xyz_angle(1, 2, -3, Math.PI / 3);
    Mat4dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, 2, 3, Math.PI);
    Mat4dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, .1, .1, Math.PI);
    Mat4dQuatAxisAngle(a1);
    a1.set_xyz_angle(.1, 1, .1, Math.PI);
    Mat4dQuatAxisAngle(a1);
    a1.set_xyz_angle(.1, .1, 1, Math.PI);
    Mat4dQuatAxisAngle(a1);
    a1.set_xyz_angle(1, 1, 1, 2 * Math.PI / 3);
    m1.set_axisAngle(a1);
    p1.set_xyz(1, 0, 0);
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(0, 1, 0)));
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(0, 0, 1)));
    m1.transformPoint(p1);
    QUnit.ok(equalsT3(p1, new Point3d(1, 0, 0)));
    m1.set_axisAngle(a1);
    QUnit.ok(equals(m1.determinant(), 1));
    QUnit.ok(equals(m1.getScale(), 1));
    m2.set_axisAngle(a1);
    m1.set_axisAngle(a1);
    m2.invert(m1);
    m1.transpose();
    QUnit.ok(equalsM4(m1, m2));
    let n1 = new Matrix3d();
    n1.set_axisAngle(a1);
    let n2 = new Matrix3d();
    v1.set_xyz(2, -1, -1);
    m1.set_rotaionTranslationScale(n1, v1, 0.4);
    m2.set_rotaionTranslationScale(n1, v1, 0.4);
    let v2 = new Vector3d();
    let s = m1.get_matrix3_vector(n2, v2);
    QUnit.ok(equalsM3(n1, n2));
    QUnit.ok(equals(s, 0.4));
    QUnit.ok(equalsT3(v1, v2));
    QUnit.ok(equalsM4(m1, m2));
}
function GMatrixTest() {
    let I44 = new GMatrix(4, 4);
    let O44 = new GMatrix(4, 4);
    O44.setZero();
    let O34 = new GMatrix(3, 4);
    O34.setZero();
    let m1 = new GMatrix(3, 4);
    let m2 = new GMatrix(3, 4);
    let mm1 = new Matrix3d();
    let mm2 = new Matrix3d();
    for (let i__0 = 0; i__0 < 3; i__0++)
        for (let j__0 = 0; j__0 < 4; j__0++) {
            m1.setElement(i__0, j__0, (i__0 + 1) * (j__0 + 2));
            if (j__0 < 3)
                mm1.setElement(i__0, j__0, (i__0 + 1) * (j__0 + 2));
        }
    for (let i__1 = 0; i__1 < 3; i__1++)
        for (let j__1 = 0; j__1 < 4; j__1++) {
            QUnit.ok(equals(m1.getElement(i__1, j__1), (i__1 + 1) * (j__1 + 2)));
        }
    m1.get_matrix3(mm2);
    QUnit.ok(equalsM3(mm1, mm2));
    m2.mul(m1, I44);
    QUnit.ok(equalsGM(m1, m2));
    m2.mul(m1, O44);
    QUnit.ok(equalsGM(O34, m2));
    let mm3 = new Matrix4d(1, 2, 3, 4, -2, 3, -1, 3, -1, -2, -4, 1, 1, 1, -1, -2);
    let mm4 = new Matrix4d();
    let mm5 = new Matrix4d();
    mm5.set_matrix4(mm3);
    m1.setSize(4, 4);
    m2.setSize(4, 4);
    m1.set_matrix4(mm3);
    QUnit.equal(m1.toString(), "[" + NL + "  [1	2	3	4]" + NL + "  [-2	3	-1	3]" + NL + "  [-1	-2	-4	1]" + NL + "  [1	1	-1	-2] ]");
    m2.set(m1);
    m1.invert();
    mm3.invert();
    mm5.mul_matrix(mm3);
    QUnit.ok(equalsM4(mm5, new Matrix4d(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)));
    m1.get_matrix4(mm4);
    QUnit.ok(equalsM4(mm3, mm4));
    m1.mul(m2);
    QUnit.ok(equalsGM(m1, I44));
    let mm6 = new Matrix4d(1, 2, 3, 4, -2, 3, -1, 3, -1, -2, -4, 1, 1, 1, -1, -2);
    let vv1 = new Vector4d(1, -1, -1, 2);
    let vv2 = new Vector4d();
    let vv3 = new Vector4d(4, 2, 7, -3);
    mm6.transform(vv1, vv2);
    QUnit.ok(equalsT4(vv2, vv3));
    m1.set_matrix4(mm6);
    let x = new GVector(4);
    let v2 = new GVector(4);
    let b = new GVector(4);
    x.set_tuple4(vv1);
    b.set_tuple4(vv3);
    let mx = new GVector(4);
    mx.mul_matrix_vector(m1, x);
    QUnit.ok(equalsGV(mx, b));
    let p = new GVector(4);
    m1.LUD(m2, p);
    QUnit.ok(checkLUD(m1, m2, p));
    let xx = new GVector(4);
    xx.LUDBackSolve(m2, b, p);
    QUnit.ok(equalsGV(xx, x));
    let u = new GMatrix(m1.getNumRow(), m1.getNumRow());
    let w = new GMatrix(m1.getNumRow(), m1.getNumCol());
    let v = new GMatrix(m1.getNumCol(), m1.getNumCol());
    let rank = m1.SVD(u, w, v);
    QUnit.equal(rank, 4);
    QUnit.ok(checkSVD(m1, u, w, v));
    xx.SVDBackSolve(u, w, v, b);
    QUnit.ok(equalsGV(xx, x));
}
function checkLUD(m, LU, permutation) {
    let n = m.getNumCol();
    let ok = true;
    for (let i = 0; i < n; i++) {
        for (let j = 0; j < n; j++) {
            let aij = 0.0;
            let min = i < j ? i : j;
            for (let k = 0; k <= min; k++) {
                if (i !== k)
                    aij += LU.getElement(i, k) * LU.getElement(k, j);
                else
                    aij += LU.getElement(k, j);
            }
            if (Math.abs(aij - m.getElement(permutation.getElement(i) | 0, j)) > epsilon) {
                console.log("a[" + i + "," + j + "] = " + aij + "(LU)ij ! = " + m.getElement(permutation.getElement(i) | 0, j));
                ok = false;
            }
        }
    }
    return ok;
}
function checkSVD(m, u, w, v) {
    let ok = true;
    let wsize = w.getNumRow() < w.getNumRow() ? w.getNumRow() : w.getNumCol();
    for (let i = 0; i < m.getNumRow(); i++) {
        for (let j = 0; j < m.getNumCol(); j++) {
            let sum = 0.0;
            for (let k = 0; k < m.getNumCol(); k++) {
                sum += u.getElement(i, k) * w.getElement(k, k) * v.getElement(j, k);
            }
            if (epsilon < Math.abs(m.getElement(i, j) - sum)) {
                console.log("(SVD)ij = " + sum + " !== a[" + i + "," + j + "] = " + m.getElement(i, j));
                ok = false;
            }
        }
    }
    if (!ok) {
        console.log("[W] = ");
        console.log(w);
        console.log("[U] = ");
        console.log(u);
        console.log("[V] = ");
        console.log(v);
    }
    return ok;
}
function SVDTest() {
    let val = [1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 8, 7, 6, 5, 4, 3, 2, 1, 0, 1];
    let m = 5;
    let n = 4;
    let matA = new GMatrix(m, n, val);
    let matU = new GMatrix(m, m);
    let matW = new GMatrix(m, n);
    let matV = new GMatrix(n, n);
    let rank = matA.SVD(matU, matW, matV);
    let matTEMP = new GMatrix(m, n);
    matTEMP.mul(matU, matW);
    matV.transpose();
    matTEMP.mul(matV);
    if (!equalsGM(matTEMP, matA)) {
        console.log("matU=" + matU);
        console.log("matW=" + matW);
        console.log("matV=" + matV);
        console.log("matA=" + matA);
        console.log("UWV=" + matTEMP);
    }
    QUnit.ok(equalsGM(matTEMP, matA));
}
//# sourceMappingURL=VecmathTest.js.map