/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
/**
 * A generic 2 element tuple that is represented by
 * double precision floating point x,y coordinates.
 * @version specification 1.1, implementation $Revision: 1.4 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Tuple2d {
    constructor(a, b) {
        if (Array.isArray(a)) {
            this.x = a[0] || 0;
            this.y = a[1] || 0;
        }
        else if (typeof a === "undefined") {
            this.x = 0.0;
            this.y = 0.0;
        }
        else if (typeof a === "number") {
            this.x = a || 0.0;
            this.y = b || 0.0;
        }
        else {
            this.x = a.x;
            this.y = a.y;
        }
    }
    /**
     * Sets the value of this tuple to the specified xy coordinates.
     * @param x the x coordinate
     * @param y the y coordinate
     */
    set_xy(x, y) {
        this.x = x;
        this.y = y;
    }
    /**
     * Sets the value of this tuple from the 2 values specified in the array.
     * @param t the array of length 2 containing xy in order
     */
    set_array(t) {
        this.x = t[0];
        this.y = t[1];
    }
    /**
     * Sets the value of this tuple to the value of the Tuple2d argument.
     * @param t1 the tuple to be copied
     */
    set_tuple(t1) {
        this.x = t1.x;
        this.y = t1.y;
    }
    /**
     * Copies the value of the elements of this tuple into the array t[].
     * @param t the array that will contain the values of the vector
     */
    get(t) {
        t[0] = this.x;
        t[1] = this.y;
    }
    /**
     * Sets the value of this tuple to the vector sum of tuples t1 and t2.
     * @param t1 the first tuple
     * @param t2 the second tuple
     *
     * Sets the value of this tuple to the vector sum of itself and tuple t1.
     * @param t1  the other tuple
     */
    add(t1, t2) {
        if (t2) {
            this.x = t1.x + t2.x;
            this.y = t1.y + t2.y;
        }
        else {
            this.x += t1.x;
            this.y += t1.y;
        }
    }
    /**
     * Sets the value of this tuple to the vector difference of tuple t1 and t2 (this = t1 - t2).
     * @param t1 the first tuple
     * @param t2 the second tuple
     *
     * Sets the value of this tuple to the vector difference of itself and tuple t1 (this = this - t1).
     * @param t1 the other tuple
     */
    sub(t1, t2) {
        if (t2) {
            this.x = t1.x - t2.x;
            this.y = t1.y - t2.y;
        }
        else {
            this.x -= t1.x;
            this.y -= t1.y;
        }
    }
    /**
     * Sets the value of this tuple to the negation of tuple t1.
     * @param t1 the source vector
     *
     * Negates the value of this vector in place.
     */
    negate(t1) {
        if (t1) {
            this.x = -t1.x;
            this.y = -t1.y;
        }
        else {
            this.x = -this.x;
            this.y = -this.y;
        }
    }
    /**
     * Sets the value of this tuple to the scalar multiplication of tuple t1.
     * @param s the scalar value
     * @param t1 the source tuple
     *
     * Sets the value of this tuple to the scalar multiplication of itself.
     * @param s the scalar value
     */
    scale(s, t1) {
        if (t1) {
            this.x = s * t1.x;
            this.y = s * t1.y;
        }
        else {
            this.x *= s;
            this.y *= s;
        }
    }
    /**
     * Sets the value of this tuple to the scalar multiplication of tuple t1 and then
     * adds tuple t2 (this = s*t1 + t2).
     * @param s the scalar value
     * @param t1 the tuple to be multipled
     * @param t2 the tuple to be added
     *
     * Sets the value of this tuple to the scalar multiplication of itself and then
     * adds tuple t1 (this = s*this + t1).
     * @param s the scalar value
     * @param t1 the tuple to be added
     */
    scaleAdd(s, t1, t2) {
        if (t2) {
            this.x = s * t1.x + t2.x;
            this.y = s * t1.y + t2.y;
        }
        else {
            this.x = s * this.x + t1.x;
            this.y = s * this.y + t1.y;
        }
    }
    /**
     * Returns true if all of the data members of Tuple2d t1 are equal to the corresponding
     * data members in this
     * @param t1 the vector with which the comparison is made.
     */
    equals(t1) {
        return t1 && this.x === t1.x && this.y === t1.y;
    }
    /**
     * Returns true if the L-infinite distance between this tuple and tuple t1 is
     * less than or equal to the epsilon parameter, otherwise returns false. The L-infinite
     * distance is equal to MAX[abs(x1-x2), abs(y1-y2)].
     * @param t1 the tuple to be compared to this tuple
     * @param epsilon the threshold value
     */
    epsilonEquals(t1, epsilon) {
        return (Math.abs(t1.x - this.x) <= epsilon) && (Math.abs(t1.y - this.y) <= epsilon);
    }
    /**
     * Returns a string that contains the values of this Tuple2d. The form is (x,y).
     * @return the String representation
     */
    toString() {
        return "(" + this.x + ", " + this.y + ")";
    }
    /**
     * Clamps the tuple parameter to the range [low, high] and places the values
     * into this tuple.
     * @param min the lowest value in the tuple after clamping
     * @param max the highest value in the tuple after clamping
     * @param t the source tuple, which will not be modified

     * Clamps this tuple to the range [low, high].
     * @param min the lowest value in this tuple after clamping
     * @param max the highest value in this tuple after clamping
     */
    clamp(min, max, t) {
        if (t)
            this.set_tuple(t);
        this.clampMin(min);
        this.clampMax(max);
    }
    /**
     * Clamps the minimum value of the tuple parameter to the min parameter
     * and places the values into this tuple.
     * @param min the lowest value in the tuple after clamping
     * @parm t the source tuple, which will not be modified
     * Clamps the minimum value of this tuple to the min parameter.
     * @param min the lowest value in this tuple after clamping
     */
    clampMin(min, t) {
        if (t)
            this.set_tuple(t);
        if (this.x < min)
            this.x = min;
        if (this.y < min)
            this.y = min;
    }
    /**
     * Clamps the maximum value of the tuple parameter to the max parameter and
     * places the values into this tuple.
     * @param max the highest value in the tuple after clamping
     * @param t the source tuple, which will not be modified
     * Clamps the maximum value of this tuple to the max parameter.
     * @param max the highest value in the tuple after clamping
     */
    clampMax(max, t) {
        if (t)
            this.set_tuple(t);
        if (this.x > max)
            this.x = max;
        if (this.y > max)
            this.y = max;
    }
    /**
     * Sets each component of the tuple parameter to its absolute value and
     * places the modified values into this tuple.
     * @param t the source tuple, which will not be modified
     * Sets each component of this tuple to its absolute value.
     */
    absolute(t) {
        if (t)
            this.set_tuple(t);
        if (this.x < 0.0)
            this.x = -this.x;
        if (this.y < 0.0)
            this.y = -this.y;
    }
    interpolate(t1, p2, alpha) {
        if (typeof p2 === "number") {
            alpha = p2;
            let beta = 1 - alpha;
            this.x = beta * this.x + alpha * t1.x;
            this.y = beta * this.y + alpha * t1.y;
        }
        else {
            let t2 = p2;
            this.set_tuple(t1);
            this.interpolate(t2, alpha);
        }
    }
}
//# sourceMappingURL=Tuple2d.js.map