/**
 * A generic 2 element tuple that is represented by
 * double precision floating point x,y coordinates.
 * @version specification 1.1, implementation $Revision: 1.4 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export declare abstract class Tuple2d {
    /**
     * The x coordinate.*/
    x: number;
    /**
     * The y coordinate.*/
    y: number;
    /**
     * Constructs and initializes a Tuple2d from the specified xy coordinates.
     * @param x the x coordinate
     * @param y the y coordinate
     */
    constructor(x: number, y: number);
    /**
     * Constructs and initializes a Tuple2d from the specified array.
     * @param t the array of length 2 containing xy in order
     */
    constructor(t: number[]);
    /**
     * Constructs and initializes a Tuple2d from the specified Tuple2d.
     * @param t1 the Tuple2d containing the initialization x y data
     */
    constructor(t1: Tuple2d);
    /**
     * Constructs and initializes a Tuple2d to (0,0).
     */
    constructor();
    /**
     * Sets the value of this tuple to the specified xy coordinates.
     * @param x the x coordinate
     * @param y the y coordinate
     */
    set_xy(x: number, y: number): void;
    /**
     * Sets the value of this tuple from the 2 values specified in the array.
     * @param t the array of length 2 containing xy in order
     */
    set_array(t: number[]): void;
    /**
     * Sets the value of this tuple to the value of the Tuple2d argument.
     * @param t1 the tuple to be copied
     */
    set_tuple(t1: Tuple2d): void;
    /**
     * Copies the value of the elements of this tuple into the array t[].
     * @param t the array that will contain the values of the vector
     */
    get(t: number[]): void;
    /**
     * Sets the value of this tuple to the vector sum of tuples t1 and t2.
     * @param t1 the first tuple
     * @param t2 the second tuple
     *
     * Sets the value of this tuple to the vector sum of itself and tuple t1.
     * @param t1  the other tuple
     */
    add(t1: Tuple2d, t2?: Tuple2d): void;
    /**
     * Sets the value of this tuple to the vector difference of tuple t1 and t2 (this = t1 - t2).
     * @param t1 the first tuple
     * @param t2 the second tuple
     *
     * Sets the value of this tuple to the vector difference of itself and tuple t1 (this = this - t1).
     * @param t1 the other tuple
     */
    sub(t1: Tuple2d, t2?: Tuple2d): void;
    /**
     * Sets the value of this tuple to the negation of tuple t1.
     * @param t1 the source vector
     *
     * Negates the value of this vector in place.
     */
    negate(t1?: Tuple2d): void;
    /**
     * Sets the value of this tuple to the scalar multiplication of tuple t1.
     * @param s the scalar value
     * @param t1 the source tuple
     *
     * Sets the value of this tuple to the scalar multiplication of itself.
     * @param s the scalar value
     */
    scale(s: number, t1?: Tuple2d): void;
    /**
     * Sets the value of this tuple to the scalar multiplication of tuple t1 and then
     * adds tuple t2 (this = s*t1 + t2).
     * @param s the scalar value
     * @param t1 the tuple to be multipled
     * @param t2 the tuple to be added
     *
     * Sets the value of this tuple to the scalar multiplication of itself and then
     * adds tuple t1 (this = s*this + t1).
     * @param s the scalar value
     * @param t1 the tuple to be added
     */
    scaleAdd(s: number, t1: Tuple2d, t2?: Tuple2d): void;
    /**
     * Returns true if all of the data members of Tuple2d t1 are equal to the corresponding
     * data members in this
     * @param t1 the vector with which the comparison is made.
     */
    equals(t1: Tuple2d): boolean;
    /**
     * Returns true if the L-infinite distance between this tuple and tuple t1 is
     * less than or equal to the epsilon parameter, otherwise returns false. The L-infinite
     * distance is equal to MAX[abs(x1-x2), abs(y1-y2)].
     * @param t1 the tuple to be compared to this tuple
     * @param epsilon the threshold value
     */
    epsilonEquals(t1: Tuple2d, epsilon: number): boolean;
    /**
     * Returns a string that contains the values of this Tuple2d. The form is (x,y).
     * @return the String representation
     */
    toString(): string;
    /**
     * Clamps the tuple parameter to the range [low, high] and places the values
     * into this tuple.
     * @param min the lowest value in the tuple after clamping
     * @param max the highest value in the tuple after clamping
     * @param t the source tuple, which will not be modified

     * Clamps this tuple to the range [low, high].
     * @param min the lowest value in this tuple after clamping
     * @param max the highest value in this tuple after clamping
     */
    clamp(min: number, max: number, t?: Tuple2d): void;
    /**
     * Clamps the minimum value of the tuple parameter to the min parameter
     * and places the values into this tuple.
     * @param min the lowest value in the tuple after clamping
     * @parm t the source tuple, which will not be modified
     * Clamps the minimum value of this tuple to the min parameter.
     * @param min the lowest value in this tuple after clamping
     */
    clampMin(min: number, t?: Tuple2d): void;
    /**
     * Clamps the maximum value of the tuple parameter to the max parameter and
     * places the values into this tuple.
     * @param max the highest value in the tuple after clamping
     * @param t the source tuple, which will not be modified
     * Clamps the maximum value of this tuple to the max parameter.
     * @param max the highest value in the tuple after clamping
     */
    clampMax(max: number, t?: Tuple2d): void;
    /**
     * Sets each component of the tuple parameter to its absolute value and
     * places the modified values into this tuple.
     * @param t the source tuple, which will not be modified
     * Sets each component of this tuple to its absolute value.
     */
    absolute(t?: Tuple2d): void;
    /**
     * Linearly interpolates between tuples t1 and t2 and places the
     * result into this tuple: this = (1-alpha)*t1 + alpha*t2.
     * @param t1 the first tuple
     * @param t2 the second tuple
     * @param alpha the alpha interpolation parameter
     */
    interpolate(t1: Tuple2d, t2: Tuple2d, alpha: number): void;
    /**
     * Linearly interpolates between this tuple and tuple t1 and places the
     * result into this tuple: this = (1-alpha)*this + alpha*t1.
     * @param t1 the first tuple
     * @param alpha the alpha interpolation parameter
     */
    interpolate(t1: Tuple2d, alpha: number): void;
}
