/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { Tuple4d } from './Tuple4d';
/**
 * A 4 element quaternion represented by double precision floating
 * point x,y,z,w coordinates.
 * @version specification 1.1, implementation $Revision: 1.8 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Quat4d extends Tuple4d {
    /**
     * Sets the value of this quaternion to the conjugate of quaternion q1.
     * @param q1 the source vector

     * Negate the value of of each of this quaternion's x,y,z coordinates
     * in place.
     */
    conjugate(q1) {
        if (q1) {
            this.x = -q1.x;
            this.y = -q1.y;
            this.z = -q1.z;
            this.w = q1.w;
        }
        else {
            this.x = -this.x;
            this.y = -this.y;
            this.z = -this.z;
        }
    }
    /**
     * Sets the value of this quaternion to the quaternion product of
     * quaternions q1 and q2 (this = q1 * q2).
     * Note that this is safe for aliasing (e.g. this can be q1 or q2).
     * @param q1 the first quaternion
     * @param q2 the second quaternion

     * Sets the value of this quaternion to the quaternion product of
     * itself and q1 (this = this * q1).
     * @param q1 the other quaternion
     */
    mul(q1, q2) {
        if (q2) {
            this.set_xyzw(q1.x * q2.w + q1.w * q2.x + q1.y * q2.z - q1.z * q2.y, q1.y * q2.w + q1.w * q2.y + q1.z * q2.x - q1.x * q2.z, q1.z * q2.w + q1.w * q2.z + q1.x * q2.y - q1.y * q2.x, q1.w * q2.w - q1.x * q2.x - q1.y * q2.y - q1.z * q2.z);
        }
        else {
            this.set_xyzw(this.x * q1.w + this.w * q1.x + this.y * q1.z - this.z * q1.y, this.y * q1.w + this.w * q1.y + this.z * q1.x - this.x * q1.z, this.z * q1.w + this.w * q1.z + this.x * q1.y - this.y * q1.x, this.w * q1.w - this.x * q1.x - this.y * q1.y - this.z * q1.z);
        }
    }
    /**
     * Multiplies quaternion q1 by the inverse of quaternion q2 and places
     * the value into this quaternion.  The value of both argument quaternions
     * is preservered (this = q1 * q2^-1).
     * @param q1 the left quaternion
     * @param q2 the right quaternion

     * Multiplies this quaternion by the inverse of quaternion q1 and places
     * the value into this quaternion.  The value of the argument quaternion
     * is preserved (this = this * q^-1).
     * @param q1 the other quaternion
     */
    mulInverse(q1, q2) {
        let n = this.norm();
        n = (n === 0.0 ? n : 1 / n);
        if (q2) {
            this.set_xyzw((q1.x * q2.w - q1.w * q2.x - q1.y * q2.z + q1.z * q2.y) * n, (q1.y * q2.w - q1.w * q2.y - q1.z * q2.x + q1.x * q2.z) * n, (q1.z * q2.w - q1.w * q2.z - q1.x * q2.y + q1.y * q2.x) * n, (q1.w * q2.w + q1.x * q2.x + q1.y * q2.y + q1.z * q2.z) * n);
        }
        else {
            this.set_xyzw((this.x * q1.w - this.w * q1.x - this.y * q1.z + this.z * q1.y) * n, (this.y * q1.w - this.w * q1.y - this.z * q1.x + this.x * q1.z) * n, (this.z * q1.w - this.w * q1.z - this.x * q1.y + this.y * q1.x) * n, (this.w * q1.w + this.x * q1.x + this.y * q1.y + this.z * q1.z) * n);
        }
    }
    norm() {
        return this.x * this.x + this.y * this.y + this.z * this.z + this.w * this.w;
    }
    /**
     * Sets the value of this quaternion to quaternion inverse of quaternion q1.
     * @param q1 the quaternion to be inverted
     *
     * Sets the value of this quaternion to the quaternion inverse of itself.
     */
    inverse(q1) {
        if (q1) {
            let n = q1.norm();
            this.x = -q1.x / n;
            this.y = -q1.y / n;
            this.z = -q1.z / n;
            this.w = q1.w / n;
        }
        else {
            let n = this.norm();
            this.x = -this.x / n;
            this.y = -this.y / n;
            this.z = -this.z / n;
            this.w /= n;
        }
    }
    /**
     * Sets the value of this quaternion to the normalized value
     * of quaternion q1.
     * @param q1 the quaternion to be normalized.
     *
     * Normalizes the value of this quaternion in place.
     */
    normalize(q1) {
        if (q1) {
            let n = Math.sqrt(q1.norm());
            this.x = q1.x / n;
            this.y = q1.y / n;
            this.z = q1.z / n;
            this.w = q1.w / n;
        }
        else {
            let n = Math.sqrt(this.norm());
            this.x /= n;
            this.y /= n;
            this.z /= n;
            this.w /= n;
        }
    }
    /**
     * Sets the value of this quaternion to the rotational component of
     * the passed matrix.
     * @param m1 the matrix4d
     */
    set_matrix4(m1) {
        this.setFromMat(m1.m00, m1.m01, m1.m02, m1.m10, m1.m11, m1.m12, m1.m20, m1.m21, m1.m22);
    }
    /**
     * Sets the value of this quaternion to the rotational component of
     * the passed matrix.
     * @param m1 the matrix3d
     */
    set_matrix3(m1) {
        this.setFromMat(m1.m00, m1.m01, m1.m02, m1.m10, m1.m11, m1.m12, m1.m20, m1.m21, m1.m22);
    }
    /**
     * Sets the value of this quaternion to the equivalent rotation of teh
     * AxisAngle argument.
     * @param a1 the axis-angle
     */
    set_axisAngle(a1) {
        this.x = a1.x;
        this.y = a1.y;
        this.z = a1.z;
        let n = Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
        let s = Math.sin(0.5 * a1.angle) / n;
        this.x *= s;
        this.y *= s;
        this.z *= s;
        this.w = Math.cos(0.5 * a1.angle);
    }
    interpolate(q1, p2, alpha) {
        if (p2 instanceof Quat4d) {
            this.set_tuple(q1);
            q1 = p2;
        }
        else {
            alpha = p2;
        }
        this.normalize();
        let n1 = Math.sqrt(q1.norm());
        let x1 = q1.x / n1;
        let y1 = q1.y / n1;
        let z1 = q1.z / n1;
        let w1 = q1.w / n1;
        let t = this.x * x1 + this.y * y1 + this.z * z1 + this.w * w1;
        if (1.0 <= Math.abs(t))
            return;
        t = Math.acos(t);
        let sin_t = Math.sin(t);
        if (sin_t === 0.0)
            return;
        let s = Math.sin((1.0 - alpha) * t) / sin_t;
        t = Math.sin(alpha * t) / sin_t;
        this.x = s * this.x + t * x1;
        this.y = s * this.y + t * y1;
        this.z = s * this.z + t * z1;
        this.w = s * this.w + t * w1;
    }
    setFromMat(m00, m01, m02, m10, m11, m12, m20, m21, m22) {
        let s;
        let tr = m00 + m11 + m22;
        if (tr >= 0.0) {
            s = Math.sqrt(tr + 1.0);
            this.w = s * 0.5;
            s = 0.5 / s;
            this.x = (m21 - m12) * s;
            this.y = (m02 - m20) * s;
            this.z = (m10 - m01) * s;
        }
        else {
            let max = Math.max(Math.max(m00, m11), m22);
            if (max === m00) {
                s = Math.sqrt(m00 - (m11 + m22) + 1.0);
                this.x = s * 0.5;
                s = 0.5 / s;
                this.y = (m01 + m10) * s;
                this.z = (m20 + m02) * s;
                this.w = (m21 - m12) * s;
            }
            else if (max === m11) {
                s = Math.sqrt(m11 - (m22 + m00) + 1.0);
                this.y = s * 0.5;
                s = 0.5 / s;
                this.z = (m12 + m21) * s;
                this.x = (m01 + m10) * s;
                this.w = (m02 - m20) * s;
            }
            else {
                s = Math.sqrt(m22 - (m00 + m11) + 1.0);
                this.z = s * 0.5;
                s = 0.5 / s;
                this.x = (m20 + m02) * s;
                this.y = (m12 + m21) * s;
                this.w = (m10 - m01) * s;
            }
        }
    }
}
//# sourceMappingURL=Quat4d.js.map