/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { Tuple4d } from './Tuple4d';
/**
 * A 4 element point that is represented by double precision
 * floating point x,y,z,w coordinates.
 * @version specification 1.1, implementation $Revision: 1.10 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Point4d extends Tuple4d {
    /**
     * Sets the x,y,z components of this point to the corresponding
     * components of tuple t1. The w component of this point is set to 1.
     * @param t1 the tuple to be copied
     * @since Java3D 1.2
     */
    set_tuple3(t1) {
        this.set_xyzw(t1.x, t1.y, t1.z, 1);
    }
    /**
     * Computes the square of the distance between this point and point p1.
     * @param p1 the other point
     * @return the square of distance between this point and p1
     */
    distanceSquared(p1) {
        let dx = this.x - p1.x;
        let dy = this.y - p1.y;
        let dz = this.z - p1.z;
        let dw = this.w - p1.w;
        return dx * dx + dy * dy + dz * dz + dw * dw;
    }
    /**
     * Returns the distance between this point and point p1.
     * @param p1 the other point
     * @return the distance between this point and point p1.
     */
    distance(p1) {
        return Math.sqrt(this.distanceSquared(p1));
    }
    /**
     * Computes the L-1 (Manhattan) distance between this point and point p1.
     * The L-1 distance is equal to abs(x1-x2) + abs(y1-y2)
     * + abs(z1-z2) + abs(w1-w2).
     * @param p1 the other point
     * @return L-1 distance
     */
    distanceL1(p1) {
        return Math.abs(this.x - p1.x) + Math.abs(this.y - p1.y) + Math.abs(this.z - p1.z) + Math.abs(this.w - p1.w);
    }
    /**
     * Computes the L-infinite distance between this point and point p1.
     * The L-infinite distance is equal to MAX[abs(x1-x2), abs(y1-y2), abs(z1-z2), abs(w1-w2)].
     * @param p1 the other point
     * @return L-infinite distance
     */
    distanceLinf(p1) {
        return Math.max(Math.max(Math.abs(this.x - p1.x), Math.abs(this.y - p1.y)), Math.max(Math.abs(this.z - p1.z), Math.abs(this.w - p1.w)));
    }
    /**
     * Multiplies each of the x,y,z components of the Point4d parameter by 1/w,
     * places the projected values into this point, and places a 1 as the w
     * parameter of this point.
     * @param p1 the source Point4d, which is not modified
     */
    project(p1) {
        this.x = p1.x / p1.w;
        this.y = p1.y / p1.w;
        this.z = p1.z / p1.w;
        this.w = 1.0;
    }
}
//# sourceMappingURL=Point4d.js.map