/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { Matrix3d } from './Matrix3d';
import { Quat4d } from './Quat4d';
//module jp.kitec.vecmath
/**
 * A double precision floating point 4 by 4 matrix.
 * @version specification 1.1, implementation $Revision: 1.15 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Matrix4d {
    constructor(p1, p2, p3, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33) {
        if (arguments.length === 0) {
            this.setZero();
        }
        else if (p1 instanceof Matrix3d) {
            this.set_rotaionTranslationScale(p1, p2, p3);
        }
        else if (p1 instanceof Matrix4d) {
            this.set_matrix4(p1);
        }
        else if (p1 instanceof Quat4d) {
            this.set_quat(p1, p2, p3);
        }
        else if (Array.isArray(p1)) {
            this.set_array(p1);
        }
        else if (arguments.length === 16) {
            this.set(p1, p2, p3, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33);
        }
    }
    /**
     * Returns a string that contains the values of this Matrix4d.
     * @return the String representation
     */
    toString() {
        let nl = "\n";
        return "[" + nl + "  [" + this.m00 + "\t" + this.m01 + "\t" + this.m02 + "\t" + this.m03 + "]" + nl + "  [" + this.m10 + "\t" + this.m11 + "\t" + this.m12 + "\t" + this.m13 + "]" + nl + "  [" + this.m20 + "\t" + this.m21 + "\t" + this.m22 + "\t" + this.m23 + "]" + nl + "  [" + this.m30 + "\t" + this.m31 + "\t" + this.m32 + "\t" + this.m33 + "] ]";
    }
    /**
     * Sets this Matrix4d to identity.*/
    setIdentity() {
        this.m00 = 1.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m03 = 0.0;
        this.m10 = 0.0;
        this.m11 = 1.0;
        this.m12 = 0.0;
        this.m13 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 1.0;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the specified element of this matrix4d to the value provided.
     * @param row  the row number to be modified (zero indexed)
     * @param column  the column number to be modified (zero indexed)
     * @param value the new value
     */
    setElement(row, column, value) {
        if (row === 0) {
            if (column === 0)
                this.m00 = value;
            else if (column === 1)
                this.m01 = value;
            else if (column === 2)
                this.m02 = value;
            else if (column === 3)
                this.m03 = value;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 1) {
            if (column === 0)
                this.m10 = value;
            else if (column === 1)
                this.m11 = value;
            else if (column === 2)
                this.m12 = value;
            else if (column === 3)
                this.m13 = value;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 2) {
            if (column === 0)
                this.m20 = value;
            else if (column === 1)
                this.m21 = value;
            else if (column === 2)
                this.m22 = value;
            else if (column === 3)
                this.m23 = value;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 3) {
            if (column === 0)
                this.m30 = value;
            else if (column === 1)
                this.m31 = value;
            else if (column === 2)
                this.m32 = value;
            else if (column === 3)
                this.m33 = value;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else
            throw new Error("row must be 0 to 2 and is " + row);
    }
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row  the row number to be retrieved (zero indexed)
     * @param column  the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row, column) {
        if (row === 0) {
            if (column === 0)
                return this.m00;
            else if (column === 1)
                return this.m01;
            else if (column === 2)
                return this.m02;
            else if (column === 3)
                return this.m03;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 1) {
            if (column === 0)
                return this.m10;
            else if (column === 1)
                return this.m11;
            else if (column === 2)
                return this.m12;
            else if (column === 3)
                return this.m13;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 2) {
            if (column === 0)
                return this.m20;
            else if (column === 1)
                return this.m21;
            else if (column === 2)
                return this.m22;
            else if (column === 3)
                return this.m23;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else if (row === 3) {
            if (column === 0)
                return this.m30;
            else if (column === 1)
                return this.m31;
            else if (column === 2)
                return this.m32;
            else if (column === 3)
                return this.m33;
            else
                throw new Error("column must be 0 to 3 and is " + column);
        }
        else
            throw new Error("row must be 0 to 3 and is " + row);
    }
    /**
     * Performs an SVD normalization of this matrix in order to acquire the
     * normalized rotational component; the values are placed into the Matrix3d parameter.
     * @param m1 matrix into which the rotational component is placed
     */
    get_matrix3(m1) {
        this.SVD(m1, null);
    }
    /**
     * Performs an SVD normalization of this matrix to calculate the rotation
     * as a 3x3 matrix, the translation, and the scale. None of the matrix values are modified.
     * @param m1 The normalized matrix representing the rotation
     * @param t1 The translation component
     * @return The scale component of this transform
     */
    get_matrix3_vector(m1, t1) {
        this.get_translation(t1);
        return this.SVD(m1, null);
    }
    /**
     * Performs an SVD normalization of this matrix in order to acquire the
     * normalized rotational component; the values are placed into
     * the Quat4f parameter.
     * @param q1 quaternion into which the rotation component is placed
     */
    get_quat(q1) {
        q1.set_matrix4(this);
        q1.normalize();
    }
    /**
     * Retrieves the translational components of this matrix.
     * @param trans the vector that will receive the translational component
     */
    get_translation(trans) {
        trans.x = this.m03;
        trans.y = this.m13;
        trans.z = this.m23;
    }
    /**
     * Gets the upper 3x3 values of this matrix and places them into the matrix m1.
     * @param m1 The matrix that will hold the values
     */
    getRotationScale(m1) {
        m1.m00 = this.m00;
        m1.m01 = this.m01;
        m1.m02 = this.m02;
        m1.m10 = this.m10;
        m1.m11 = this.m11;
        m1.m12 = this.m12;
        m1.m20 = this.m20;
        m1.m21 = this.m21;
        m1.m22 = this.m22;
    }
    /**
     * Performs an SVD normalization of this matrix to calculate and return the
     * uniform scale factor. This matrix is not modified.
     * @return the scale factor of this matrix
     */
    getScale() {
        return this._SVD(null);
    }
    /**
     * Replaces the upper 3x3 matrix values of this matrix with the values in the matrix m1.
     * @param m1 The matrix that will be the new upper 3x3
     */
    setRotationScale(m1) {
        this.m00 = m1.m00;
        this.m01 = m1.m01;
        this.m02 = m1.m02;
        this.m10 = m1.m10;
        this.m11 = m1.m11;
        this.m12 = m1.m12;
        this.m20 = m1.m20;
        this.m21 = m1.m21;
        this.m22 = m1.m22;
    }
    /**
     * Sets the scale component of the current matrix by factoring out the
     * current scale (by doing an SVD) from the rotational component and
     * multiplying by the new scale.
     * @param scale the new scale amount
     */
    setScale(scale) {
        this.SVD(null, this);
        this.mulRotationScale(scale);
    }
    /**
     * Sets the specified row of this matrix4d to the four values provided.
     * @param row  the row number to be modified (zero indexed)
     * @param x the first column element
     * @param y the second column element
     * @param z the third column element
     * @param w the fourth column element
     */
    setRow_xyzw(row, x, y, z, w) {
        if (row === 0) {
            this.m00 = x;
            this.m01 = y;
            this.m02 = z;
            this.m03 = w;
        }
        else if (row === 1) {
            this.m10 = x;
            this.m11 = y;
            this.m12 = z;
            this.m13 = w;
        }
        else if (row === 2) {
            this.m20 = x;
            this.m21 = y;
            this.m22 = z;
            this.m23 = w;
        }
        else if (row === 3) {
            this.m30 = x;
            this.m31 = y;
            this.m32 = z;
            this.m33 = w;
        }
        else {
            throw new Error("row must be 0 to 3 and is " + row);
        }
    }
    /**
     * Sets the specified row of this matrix4d to the Vector provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_vector(row, v) {
        if (row === 0) {
            this.m00 = v.x;
            this.m01 = v.y;
            this.m02 = v.z;
            this.m03 = v.w;
        }
        else if (row === 1) {
            this.m10 = v.x;
            this.m11 = v.y;
            this.m12 = v.z;
            this.m13 = v.w;
        }
        else if (row === 2) {
            this.m20 = v.x;
            this.m21 = v.y;
            this.m22 = v.z;
            this.m23 = v.w;
        }
        else if (row === 3) {
            this.m30 = v.x;
            this.m31 = v.y;
            this.m32 = v.z;
            this.m33 = v.w;
        }
        else {
            throw new Error("row must be 0 to 3 and is " + row);
        }
    }
    /**
     * Sets the specified row of this matrix4d to the four values provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_array(row, v) {
        if (row === 0) {
            this.m00 = v[0];
            this.m01 = v[1];
            this.m02 = v[2];
            this.m03 = v[3];
        }
        else if (row === 1) {
            this.m10 = v[0];
            this.m11 = v[1];
            this.m12 = v[2];
            this.m13 = v[3];
        }
        else if (row === 2) {
            this.m20 = v[0];
            this.m21 = v[1];
            this.m22 = v[2];
            this.m23 = v[3];
        }
        else if (row === 3) {
            this.m30 = v[0];
            this.m31 = v[1];
            this.m32 = v[2];
            this.m33 = v[3];
        }
        else {
            throw new Error("row must be 0 to 3 and is " + row);
        }
    }
    /**
     * Copies the matrix values in the specified row into the
     * vector parameter.
     * @param row the matrix row
     * @param v The vector into which the matrix row values will be copied
     */
    getRow_vector(row, v) {
        if (row === 0) {
            v.x = this.m00;
            v.y = this.m01;
            v.z = this.m02;
            v.w = this.m03;
        }
        else if (row === 1) {
            v.x = this.m10;
            v.y = this.m11;
            v.z = this.m12;
            v.w = this.m13;
        }
        else if (row === 2) {
            v.x = this.m20;
            v.y = this.m21;
            v.z = this.m22;
            v.w = this.m23;
        }
        else if (row === 3) {
            v.x = this.m30;
            v.y = this.m31;
            v.z = this.m32;
            v.w = this.m33;
        }
        else {
            throw new Error("row must be 0 to 3 and is " + row);
        }
    }
    /**
     * Copies the matrix values in the specified row into the
     * array parameter.
     * @param row the matrix row
     * @param v The array into which the matrix row values will be copied
     */
    getRow_array(row, v) {
        if (row === 0) {
            v[0] = this.m00;
            v[1] = this.m01;
            v[2] = this.m02;
            v[3] = this.m03;
        }
        else if (row === 1) {
            v[0] = this.m10;
            v[1] = this.m11;
            v[2] = this.m12;
            v[3] = this.m13;
        }
        else if (row === 2) {
            v[0] = this.m20;
            v[1] = this.m21;
            v[2] = this.m22;
            v[3] = this.m23;
        }
        else if (row === 3) {
            v[0] = this.m30;
            v[1] = this.m31;
            v[2] = this.m32;
            v[3] = this.m33;
        }
        else {
            throw new Error("row must be 0 to 3 and is " + row);
        }
    }
    /**
     * Sets the specified column of this matrix4d to the four values provided.
     * @param column the column number to be modified (zero indexed)
     * @param x the first row element
     * @param y the second row element
     * @param z the third row element
     * @param w the fourth row element
     */
    setColumn_xyzw(column, x, y, z, w) {
        if (column === 0) {
            this.m00 = x;
            this.m10 = y;
            this.m20 = z;
            this.m30 = w;
        }
        else if (column === 1) {
            this.m01 = x;
            this.m11 = y;
            this.m21 = z;
            this.m31 = w;
        }
        else if (column === 2) {
            this.m02 = x;
            this.m12 = y;
            this.m22 = z;
            this.m32 = w;
        }
        else if (column === 3) {
            this.m03 = x;
            this.m13 = y;
            this.m23 = z;
            this.m33 = w;
        }
        else {
            throw new Error("column must be 0 to 3 and is " + column);
        }
    }
    /**
     * Sets the specified column of this matrix4d to the vector provided.
     * @param column the column number to be modified (zero indexed)
     * @param v the replacement column
     */
    setColumn_vector(column, v) {
        if (column === 0) {
            this.m00 = v.x;
            this.m10 = v.y;
            this.m20 = v.z;
            this.m30 = v.w;
        }
        else if (column === 1) {
            this.m01 = v.x;
            this.m11 = v.y;
            this.m21 = v.z;
            this.m31 = v.w;
        }
        else if (column === 2) {
            this.m02 = v.x;
            this.m12 = v.y;
            this.m22 = v.z;
            this.m32 = v.w;
        }
        else if (column === 3) {
            this.m03 = v.x;
            this.m13 = v.y;
            this.m23 = v.z;
            this.m33 = v.w;
        }
        else {
            throw new Error("column must be 0 to 3 and is " + column);
        }
    }
    /**
     * Sets the specified column of this matrix4d to the four values provided.
     * @param column  the column number to be modified (zero indexed)
     * @param v       the replacement column
     */
    setColumn_array(column, v) {
        if (column === 0) {
            this.m00 = v[0];
            this.m10 = v[1];
            this.m20 = v[2];
            this.m30 = v[3];
        }
        else if (column === 1) {
            this.m01 = v[0];
            this.m11 = v[1];
            this.m21 = v[2];
            this.m31 = v[3];
        }
        else if (column === 2) {
            this.m02 = v[0];
            this.m12 = v[1];
            this.m22 = v[2];
            this.m32 = v[3];
        }
        else if (column === 3) {
            this.m03 = v[0];
            this.m13 = v[1];
            this.m23 = v[2];
            this.m33 = v[3];
        }
        else {
            throw new Error("column must be 0 to 3 and is " + column);
        }
    }
    /**
     * Copies the matrix values in the specified column into the
     * vector parameter.
     * @param column the matrix column
     * @param v The vector into which the matrix column values will be copied
     */
    getColumn_vector(column, v) {
        if (column === 0) {
            v.x = this.m00;
            v.y = this.m10;
            v.z = this.m20;
            v.w = this.m30;
        }
        else if (column === 1) {
            v.x = this.m01;
            v.y = this.m11;
            v.z = this.m21;
            v.w = this.m31;
        }
        else if (column === 2) {
            v.x = this.m02;
            v.y = this.m12;
            v.z = this.m22;
            v.w = this.m32;
        }
        else if (column === 3) {
            v.x = this.m03;
            v.y = this.m13;
            v.z = this.m23;
            v.w = this.m33;
        }
        else {
            throw new Error("column must be 0 to 3 and is " + column);
        }
    }
    /**
     * Copies the matrix values in the specified column into the
     * array parameter.
     * @param column the matrix column
     * @param v The array into which the matrix column values will be copied
     */
    getColumn_array(column, v) {
        if (column === 0) {
            v[0] = this.m00;
            v[1] = this.m10;
            v[2] = this.m20;
            v[3] = this.m30;
        }
        else if (column === 1) {
            v[0] = this.m01;
            v[1] = this.m11;
            v[2] = this.m21;
            v[3] = this.m31;
        }
        else if (column === 2) {
            v[0] = this.m02;
            v[1] = this.m12;
            v[2] = this.m22;
            v[3] = this.m32;
        }
        else if (column === 3) {
            v[0] = this.m03;
            v[1] = this.m13;
            v[2] = this.m23;
            v[3] = this.m33;
        }
        else {
            throw new Error("column must be 0 to 3 and is " + column);
        }
    }
    /**
     * Adds a scalar to each component of this matrix.
     * @param scalar The scalar adder.

     * Adds a scalar to each component of the matrix m1 and places
     * the result into this. Matrix m1 is not modified.
     * @param scalar The scalar adder.
     * @parm m1 The original matrix values.
     */
    add_scalar(scalar, m1) {
        if (m1)
            this.set_matrix4(m1);
        this.m00 += scalar;
        this.m01 += scalar;
        this.m02 += scalar;
        this.m03 += scalar;
        this.m10 += scalar;
        this.m11 += scalar;
        this.m12 += scalar;
        this.m13 += scalar;
        this.m20 += scalar;
        this.m21 += scalar;
        this.m22 += scalar;
        this.m23 += scalar;
        this.m30 += scalar;
        this.m31 += scalar;
        this.m32 += scalar;
        this.m33 += scalar;
    }
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix

     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add_matrix(m1, m2) {
        if (m2) {
            this.set(m1.m00 + m2.m00, m1.m01 + m2.m01, m1.m02 + m2.m02, m1.m03 + m2.m03, m1.m10 + m2.m10, m1.m11 + m2.m11, m1.m12 + m2.m12, m1.m13 + m2.m13, m1.m20 + m2.m20, m1.m21 + m2.m21, m1.m22 + m2.m22, m1.m23 + m2.m23, m1.m30 + m2.m30, m1.m31 + m2.m31, m1.m32 + m2.m32, m1.m33 + m2.m33);
        }
        else {
            this.m00 += m1.m00;
            this.m01 += m1.m01;
            this.m02 += m1.m02;
            this.m03 += m1.m03;
            this.m10 += m1.m10;
            this.m11 += m1.m11;
            this.m12 += m1.m12;
            this.m13 += m1.m13;
            this.m20 += m1.m20;
            this.m21 += m1.m21;
            this.m22 += m1.m22;
            this.m23 += m1.m23;
            this.m30 += m1.m30;
            this.m31 += m1.m31;
            this.m32 += m1.m32;
            this.m33 += m1.m33;
        }
    }
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix

     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1, m2) {
        if (m2) {
            this.set(m1.m00 - m2.m00, m1.m01 - m2.m01, m1.m02 - m2.m02, m1.m03 - m2.m03, m1.m10 - m2.m10, m1.m11 - m2.m11, m1.m12 - m2.m12, m1.m13 - m2.m13, m1.m20 - m2.m20, m1.m21 - m2.m21, m1.m22 - m2.m22, m1.m23 - m2.m23, m1.m30 - m2.m30, m1.m31 - m2.m31, m1.m32 - m2.m32, m1.m33 - m2.m33);
        }
        else {
            this.m00 -= m1.m00;
            this.m01 -= m1.m01;
            this.m02 -= m1.m02;
            this.m03 -= m1.m03;
            this.m10 -= m1.m10;
            this.m11 -= m1.m11;
            this.m12 -= m1.m12;
            this.m13 -= m1.m13;
            this.m20 -= m1.m20;
            this.m21 -= m1.m21;
            this.m22 -= m1.m22;
            this.m23 -= m1.m23;
            this.m30 -= m1.m30;
            this.m31 -= m1.m31;
            this.m32 -= m1.m32;
            this.m33 -= m1.m33;
        }
    }
    /**
     * Sets the value of this matrix to the transpose of the argument matrix
     * @param m1 the matrix to be transposed

     * Sets the value of this matrix to its transpose.
     */
    transpose(m1) {
        if (m1)
            this.set_matrix4(m1);
        let tmp = this.m01;
        this.m01 = this.m10;
        this.m10 = tmp;
        tmp = this.m02;
        this.m02 = this.m20;
        this.m20 = tmp;
        tmp = this.m03;
        this.m03 = this.m30;
        this.m30 = tmp;
        tmp = this.m12;
        this.m12 = this.m21;
        this.m21 = tmp;
        tmp = this.m13;
        this.m13 = this.m31;
        this.m31 = tmp;
        tmp = this.m23;
        this.m23 = this.m32;
        this.m32 = tmp;
    }
    /**
     * Sets the values in this Matrix4d equal to the row-major array parameter
     * (ie, the first four elements of the array will be copied into the first
     * row of this matrix, etc.).*/
    set_array(m) {
        this.m00 = m[0];
        this.m01 = m[1];
        this.m02 = m[2];
        this.m03 = m[3];
        this.m10 = m[4];
        this.m11 = m[5];
        this.m12 = m[6];
        this.m13 = m[7];
        this.m20 = m[8];
        this.m21 = m[9];
        this.m22 = m[10];
        this.m23 = m[11];
        this.m30 = m[12];
        this.m31 = m[13];
        this.m32 = m[14];
        this.m33 = m[15];
    }
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * values in the double precision Matrix3d argument; the other elements of
     * this matrix are initialized as if this were an identity matrix
     * (ie, affine matrix with no translational component).
     * @param m1 the 3x3 matrix
     */
    set_matrix3(m1) {
        this.m00 = m1.m00;
        this.m01 = m1.m01;
        this.m02 = m1.m02;
        this.m03 = 0.0;
        this.m10 = m1.m10;
        this.m11 = m1.m11;
        this.m12 = m1.m12;
        this.m13 = 0.0;
        this.m20 = m1.m20;
        this.m21 = m1.m21;
        this.m22 = m1.m22;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * double precision axis and angle argument.
     * @param a1 the axis and angle to be converted
     */
    set_axisAngle(a1) {
        this.setFromAxisAngle(a1.x, a1.y, a1.z, a1.angle);
    }
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * (double precision) quaternion argument.
     * @param q1 the quaternion to be converted
     * Sets the value of this matrix from the rotation expressed by the
     * quaternion q1, the translation t1, and the scale s.
     * @param q1  the rotation expressed as a quaternion
     * @param t1  the translation
     * @param s  the scale value
     */
    set_quat(q1, t1, s) {
        this.setFromQuat(q1.x, q1.y, q1.z, q1.w);
        if (arguments.length === 3) {
            this.mulRotationScale(s);
            this.m03 = t1.x;
            this.m13 = t1.y;
            this.m23 = t1.z;
        }
    }
    /**
     * Sets the value of this matrix to a copy of the
     * passed matrix m1.
     * @param m1 the matrix to be copied
     */
    set_matrix4(m1) {
        this.m00 = m1.m00;
        this.m01 = m1.m01;
        this.m02 = m1.m02;
        this.m03 = m1.m03;
        this.m10 = m1.m10;
        this.m11 = m1.m11;
        this.m12 = m1.m12;
        this.m13 = m1.m13;
        this.m20 = m1.m20;
        this.m21 = m1.m21;
        this.m22 = m1.m22;
        this.m23 = m1.m23;
        this.m30 = m1.m30;
        this.m31 = m1.m31;
        this.m32 = m1.m32;
        this.m33 = m1.m33;
    }
    /**
     * Sets the value of this matrix to its inverse.

     * Sets the value of this matrix to the matrix inverse
     * of the passed matrix m1.
     * @param m1 the matrix to be inverted
     */
    invert(m1) {
        if (m1)
            this.set_matrix4(m1);
        let s = this.determinant();
        if (s === 0.0)
            return;
        s = 1 / s;
        this.set(this.m11 * (this.m22 * this.m33 - this.m23 * this.m32) + this.m12 * (this.m23 * this.m31 - this.m21 * this.m33) + this.m13 * (this.m21 * this.m32 - this.m22 * this.m31), this.m21 * (this.m02 * this.m33 - this.m03 * this.m32) + this.m22 * (this.m03 * this.m31 - this.m01 * this.m33) + this.m23 * (this.m01 * this.m32 - this.m02 * this.m31), this.m31 * (this.m02 * this.m13 - this.m03 * this.m12) + this.m32 * (this.m03 * this.m11 - this.m01 * this.m13) + this.m33 * (this.m01 * this.m12 - this.m02 * this.m11), this.m01 * (this.m13 * this.m22 - this.m12 * this.m23) + this.m02 * (this.m11 * this.m23 - this.m13 * this.m21) + this.m03 * (this.m12 * this.m21 - this.m11 * this.m22), this.m12 * (this.m20 * this.m33 - this.m23 * this.m30) + this.m13 * (this.m22 * this.m30 - this.m20 * this.m32) + this.m10 * (this.m23 * this.m32 - this.m22 * this.m33), this.m22 * (this.m00 * this.m33 - this.m03 * this.m30) + this.m23 * (this.m02 * this.m30 - this.m00 * this.m32) + this.m20 * (this.m03 * this.m32 - this.m02 * this.m33), this.m32 * (this.m00 * this.m13 - this.m03 * this.m10) + this.m33 * (this.m02 * this.m10 - this.m00 * this.m12) + this.m30 * (this.m03 * this.m12 - this.m02 * this.m13), this.m02 * (this.m13 * this.m20 - this.m10 * this.m23) + this.m03 * (this.m10 * this.m22 - this.m12 * this.m20) + this.m00 * (this.m12 * this.m23 - this.m13 * this.m22), this.m13 * (this.m20 * this.m31 - this.m21 * this.m30) + this.m10 * (this.m21 * this.m33 - this.m23 * this.m31) + this.m11 * (this.m23 * this.m30 - this.m20 * this.m33), this.m23 * (this.m00 * this.m31 - this.m01 * this.m30) + this.m20 * (this.m01 * this.m33 - this.m03 * this.m31) + this.m21 * (this.m03 * this.m30 - this.m00 * this.m33), this.m33 * (this.m00 * this.m11 - this.m01 * this.m10) + this.m30 * (this.m01 * this.m13 - this.m03 * this.m11) + this.m31 * (this.m03 * this.m10 - this.m00 * this.m13), this.m03 * (this.m11 * this.m20 - this.m10 * this.m21) + this.m00 * (this.m13 * this.m21 - this.m11 * this.m23) + this.m01 * (this.m10 * this.m23 - this.m13 * this.m20), this.m10 * (this.m22 * this.m31 - this.m21 * this.m32) + this.m11 * (this.m20 * this.m32 - this.m22 * this.m30) + this.m12 * (this.m21 * this.m30 - this.m20 * this.m31), this.m20 * (this.m02 * this.m31 - this.m01 * this.m32) + this.m21 * (this.m00 * this.m32 - this.m02 * this.m30) + this.m22 * (this.m01 * this.m30 - this.m00 * this.m31), this.m30 * (this.m02 * this.m11 - this.m01 * this.m12) + this.m31 * (this.m00 * this.m12 - this.m02 * this.m10) + this.m32 * (this.m01 * this.m10 - this.m00 * this.m11), this.m00 * (this.m11 * this.m22 - this.m12 * this.m21) + this.m01 * (this.m12 * this.m20 - this.m10 * this.m22) + this.m02 * (this.m10 * this.m21 - this.m11 * this.m20));
        this.mul_scalar(s);
    }
    /**
     * Computes the determinant of this matrix.
     * @return the determinant of the matrix
     */
    determinant() {
        return (this.m00 * this.m11 - this.m01 * this.m10) * (this.m22 * this.m33 - this.m23 * this.m32)
            - (this.m00 * this.m12 - this.m02 * this.m10) * (this.m21 * this.m33 - this.m23 * this.m31)
            + (this.m00 * this.m13 - this.m03 * this.m10) * (this.m21 * this.m32 - this.m22 * this.m31)
            + (this.m01 * this.m12 - this.m02 * this.m11) * (this.m20 * this.m33 - this.m23 * this.m30)
            - (this.m01 * this.m13 - this.m03 * this.m11) * (this.m20 * this.m32 - this.m22 * this.m30)
            + (this.m02 * this.m13 - this.m03 * this.m12) * (this.m20 * this.m31 - this.m21 * this.m30);
    }
    /**
     * Sets the value of this matrix to a scale matrix with the
     * passed scale amount.
     * @param scale the scale factor for the matrix
     */
    set_scale(scale) {
        this.m00 = scale;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m03 = 0.0;
        this.m10 = 0.0;
        this.m11 = scale;
        this.m12 = 0.0;
        this.m13 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = scale;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the value of this matrix to a translate matrix by the
     * passed translation value.
     * @param v1 the translation amount
     */
    set_translation(v1) {
        this.setIdentity();
        this.setTranslation(v1);
    }
    /**
     * Sets the value of this matrix to a scale and translation matrix;
     * scale is not applied to the translation and all of the matrix
     * values are modified.
     * @param scale the scale factor for the matrix
     * @param v1 the translation amount
     */
    set_scaleTranslation(scale, v1) {
        this.set_scale(scale);
        this.setTranslation(v1);
    }
    /**
     * Sets the value of this matrix to a scale and translation matrix;
     * the translation is scaled by the scale factor and all of the
     * matrix values are modified.
     * @param v1 the translation amount
     * @param scale the scale factor for the matrix
     */
    set_translationScale(v1, scale) {
        this.m00 = scale;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m03 = scale * v1.x;
        this.m10 = 0.0;
        this.m11 = scale;
        this.m12 = 0.0;
        this.m13 = scale * v1.y;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = scale;
        this.m23 = scale * v1.z;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the value of this matrix from the rotation expressed by the
     * rotation matrix m1, the translation t1, and the scale s. The translation
     * is not modified by the scale.
     * @param m1 The rotation component
     * @param t1 The translation component
     * @param scale The scale component
     */
    set_rotaionTranslationScale(m1, t1, scale) {
        this.setRotationScale(m1);
        this.mulRotationScale(scale);
        this.setTranslation(t1);
        this.m33 = 1.0;
    }
    /**
     * Modifies the translational components of this matrix to the values of
     * the Vector3d argument; the other values of this matrix are not modified.
     * @param trans the translational component
     */
    setTranslation(trans) {
        this.m03 = trans.x;
        this.m13 = trans.y;
        this.m23 = trans.z;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the x axis
     * by the passed angle.
     * @param angle the angle to rotate about the X axis in radians
     */
    rotX(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = 1.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m03 = 0.0;
        this.m10 = 0.0;
        this.m11 = c;
        this.m12 = -s;
        this.m13 = 0.0;
        this.m20 = 0.0;
        this.m21 = s;
        this.m22 = c;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the y axis
     * by the passed angle.
     * @param angle the angle to rotate about the Y axis in radians
     */
    rotY(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = c;
        this.m01 = 0.0;
        this.m02 = s;
        this.m03 = 0.0;
        this.m10 = 0.0;
        this.m11 = 1.0;
        this.m12 = 0.0;
        this.m13 = 0.0;
        this.m20 = -s;
        this.m21 = 0.0;
        this.m22 = c;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the z axis
     * by the passed angle.
     * @param angle the angle to rotate about the Z axis in radians
     */
    rotZ(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = c;
        this.m01 = -s;
        this.m02 = 0.0;
        this.m03 = 0.0;
        this.m10 = s;
        this.m11 = c;
        this.m12 = 0.0;
        this.m13 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 1.0;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 1.0;
    }
    /**
     * Multiplies each element of this matrix by a scalar.
     * @param scalar The scalar multiplier.
     *
     * Multiplies each element of matrix m1 by a scalar and places the result
     * into this. Matrix m1 is not modified.
     * @param scalar The scalar multiplier.
     * @param m1 The original matrix.
     */
    mul_scalar(scalar, m1) {
        if (m1)
            this.set_matrix4(m1);
        this.m00 *= scalar;
        this.m01 *= scalar;
        this.m02 *= scalar;
        this.m03 *= scalar;
        this.m10 *= scalar;
        this.m11 *= scalar;
        this.m12 *= scalar;
        this.m13 *= scalar;
        this.m20 *= scalar;
        this.m21 *= scalar;
        this.m22 *= scalar;
        this.m23 *= scalar;
        this.m30 *= scalar;
        this.m31 *= scalar;
        this.m32 *= scalar;
        this.m33 *= scalar;
    }
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1.
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul_matrix(m1, m2) {
        if (!m2) {
            m2 = m1;
            m1 = this;
        }
        this.set(m1.m00 * m2.m00 + m1.m01 * m2.m10 + m1.m02 * m2.m20 + m1.m03 * m2.m30, m1.m00 * m2.m01 + m1.m01 * m2.m11 + m1.m02 * m2.m21 + m1.m03 * m2.m31, m1.m00 * m2.m02 + m1.m01 * m2.m12 + m1.m02 * m2.m22 + m1.m03 * m2.m32, m1.m00 * m2.m03 + m1.m01 * m2.m13 + m1.m02 * m2.m23 + m1.m03 * m2.m33, m1.m10 * m2.m00 + m1.m11 * m2.m10 + m1.m12 * m2.m20 + m1.m13 * m2.m30, m1.m10 * m2.m01 + m1.m11 * m2.m11 + m1.m12 * m2.m21 + m1.m13 * m2.m31, m1.m10 * m2.m02 + m1.m11 * m2.m12 + m1.m12 * m2.m22 + m1.m13 * m2.m32, m1.m10 * m2.m03 + m1.m11 * m2.m13 + m1.m12 * m2.m23 + m1.m13 * m2.m33, m1.m20 * m2.m00 + m1.m21 * m2.m10 + m1.m22 * m2.m20 + m1.m23 * m2.m30, m1.m20 * m2.m01 + m1.m21 * m2.m11 + m1.m22 * m2.m21 + m1.m23 * m2.m31, m1.m20 * m2.m02 + m1.m21 * m2.m12 + m1.m22 * m2.m22 + m1.m23 * m2.m32, m1.m20 * m2.m03 + m1.m21 * m2.m13 + m1.m22 * m2.m23 + m1.m23 * m2.m33, m1.m30 * m2.m00 + m1.m31 * m2.m10 + m1.m32 * m2.m20 + m1.m33 * m2.m30, m1.m30 * m2.m01 + m1.m31 * m2.m11 + m1.m32 * m2.m21 + m1.m33 * m2.m31, m1.m30 * m2.m02 + m1.m31 * m2.m12 + m1.m32 * m2.m22 + m1.m33 * m2.m32, m1.m30 * m2.m03 + m1.m31 * m2.m13 + m1.m32 * m2.m23 + m1.m33 * m2.m33);
    }
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2,
     * and places the result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1, m2) {
        this.mul_matrix(m2, m1);
        this.transpose();
    }
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeRight(m1, m2) {
        this.set(m1.m00 * m2.m00 + m1.m01 * m2.m01 + m1.m02 * m2.m02 + m1.m03 * m2.m03, m1.m00 * m2.m10 + m1.m01 * m2.m11 + m1.m02 * m2.m12 + m1.m03 * m2.m13, m1.m00 * m2.m20 + m1.m01 * m2.m21 + m1.m02 * m2.m22 + m1.m03 * m2.m23, m1.m00 * m2.m30 + m1.m01 * m2.m31 + m1.m02 * m2.m32 + m1.m03 * m2.m33, m1.m10 * m2.m00 + m1.m11 * m2.m01 + m1.m12 * m2.m02 + m1.m13 * m2.m03, m1.m10 * m2.m10 + m1.m11 * m2.m11 + m1.m12 * m2.m12 + m1.m13 * m2.m13, m1.m10 * m2.m20 + m1.m11 * m2.m21 + m1.m12 * m2.m22 + m1.m13 * m2.m23, m1.m10 * m2.m30 + m1.m11 * m2.m31 + m1.m12 * m2.m32 + m1.m13 * m2.m33, m1.m20 * m2.m00 + m1.m21 * m2.m01 + m1.m22 * m2.m02 + m1.m23 * m2.m03, m1.m20 * m2.m10 + m1.m21 * m2.m11 + m1.m22 * m2.m12 + m1.m23 * m2.m13, m1.m20 * m2.m20 + m1.m21 * m2.m21 + m1.m22 * m2.m22 + m1.m23 * m2.m23, m1.m20 * m2.m30 + m1.m21 * m2.m31 + m1.m22 * m2.m32 + m1.m23 * m2.m33, m1.m30 * m2.m00 + m1.m31 * m2.m01 + m1.m32 * m2.m02 + m1.m33 * m2.m03, m1.m30 * m2.m10 + m1.m31 * m2.m11 + m1.m32 * m2.m12 + m1.m33 * m2.m13, m1.m30 * m2.m20 + m1.m31 * m2.m21 + m1.m32 * m2.m22 + m1.m33 * m2.m23, m1.m30 * m2.m30 + m1.m31 * m2.m31 + m1.m32 * m2.m32 + m1.m33 * m2.m33);
    }
    /**
     * Multiplies the transpose of matrix m1 times matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1, m2) {
        this.set(m1.m00 * m2.m00 + m1.m10 * m2.m10 + m1.m20 * m2.m20 + m1.m30 * m2.m30, m1.m00 * m2.m01 + m1.m10 * m2.m11 + m1.m20 * m2.m21 + m1.m30 * m2.m31, m1.m00 * m2.m02 + m1.m10 * m2.m12 + m1.m20 * m2.m22 + m1.m30 * m2.m32, m1.m00 * m2.m03 + m1.m10 * m2.m13 + m1.m20 * m2.m23 + m1.m30 * m2.m33, m1.m01 * m2.m00 + m1.m11 * m2.m10 + m1.m21 * m2.m20 + m1.m31 * m2.m30, m1.m01 * m2.m01 + m1.m11 * m2.m11 + m1.m21 * m2.m21 + m1.m31 * m2.m31, m1.m01 * m2.m02 + m1.m11 * m2.m12 + m1.m21 * m2.m22 + m1.m31 * m2.m32, m1.m01 * m2.m03 + m1.m11 * m2.m13 + m1.m21 * m2.m23 + m1.m31 * m2.m33, m1.m02 * m2.m00 + m1.m12 * m2.m10 + m1.m22 * m2.m20 + m1.m32 * m2.m30, m1.m02 * m2.m01 + m1.m12 * m2.m11 + m1.m22 * m2.m21 + m1.m32 * m2.m31, m1.m02 * m2.m02 + m1.m12 * m2.m12 + m1.m22 * m2.m22 + m1.m32 * m2.m32, m1.m02 * m2.m03 + m1.m12 * m2.m13 + m1.m22 * m2.m23 + m1.m32 * m2.m33, m1.m03 * m2.m00 + m1.m13 * m2.m10 + m1.m23 * m2.m20 + m1.m33 * m2.m30, m1.m03 * m2.m01 + m1.m13 * m2.m11 + m1.m23 * m2.m21 + m1.m33 * m2.m31, m1.m03 * m2.m02 + m1.m13 * m2.m12 + m1.m23 * m2.m22 + m1.m33 * m2.m32, m1.m03 * m2.m03 + m1.m13 * m2.m13 + m1.m23 * m2.m23 + m1.m33 * m2.m33);
    }
    /**
     * Returns true if all of the data members of Matrix4d m1 are
     * equal to the corresponding data members in this Matrix4d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1) {
        return m1 && this.m00 === m1.m00 && this.m01 === m1.m01 && this.m02 === m1.m02 && this.m03 === m1.m03
            && this.m10 === m1.m10 && this.m11 === m1.m11 && this.m12 === m1.m12 && this.m13 === m1.m13
            && this.m20 === m1.m20 && this.m21 === m1.m21 && this.m22 === m1.m22 && this.m23 === m1.m23
            && this.m30 === m1.m30 && this.m31 === m1.m31 && this.m32 === m1.m32 && this.m33 === m1.m33;
    }
    /**
     * Returns true if the L-infinite distance between this matrix and matrix
     * m1 is less than or equal to the epsilon parameter, otherwise returns
     * false. The L-infinite distance is equal to MAX[i=0,1,2,3 ; j=0,1,2,3 ;
     * abs(this.m(i,j) - m1.m(i,j)]
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1, epsilon) {
        return Math.abs(this.m00 - m1.m00) <= epsilon
            && Math.abs(this.m01 - m1.m01) <= epsilon
            && Math.abs(this.m02 - m1.m02) <= epsilon
            && Math.abs(this.m03 - m1.m03) <= epsilon
            && Math.abs(this.m10 - m1.m10) <= epsilon
            && Math.abs(this.m11 - m1.m11) <= epsilon
            && Math.abs(this.m12 - m1.m12) <= epsilon
            && Math.abs(this.m13 - m1.m13) <= epsilon
            && Math.abs(this.m20 - m1.m20) <= epsilon
            && Math.abs(this.m21 - m1.m21) <= epsilon
            && Math.abs(this.m22 - m1.m22) <= epsilon
            && Math.abs(this.m23 - m1.m23) <= epsilon
            && Math.abs(this.m30 - m1.m30) <= epsilon
            && Math.abs(this.m31 - m1.m31) <= epsilon
            && Math.abs(this.m32 - m1.m32) <= epsilon
            && Math.abs(this.m33 - m1.m33) <= epsilon;
    }
    /**
     * Transform the vector vec using this Matrix4d and place the
     * result back into vec.
     * @param vec the double precision vector to be transformed
     * Transform the vector vec using this Matrix4d and place the
     * result into vecOut.
     * @param vec the double precision vector to be transformed
     * @param vecOut the vector into which the transformed values are placed
     */
    transform(vec, vecOut) {
        vecOut = vecOut || vec;
        vecOut.set_xyzw(this.m00 * vec.x + this.m01 * vec.y + this.m02 * vec.z + this.m03 * vec.w, this.m10 * vec.x + this.m11 * vec.y + this.m12 * vec.z + this.m13 * vec.w, this.m20 * vec.x + this.m21 * vec.y + this.m22 * vec.z + this.m23 * vec.w, this.m30 * vec.x + this.m31 * vec.y + this.m32 * vec.z + this.m33 * vec.w);
    }
    /**
     * Transforms the point parameter with this Matrix4d and
     * places the result back into point.  The fourth element of the
     * point input paramter is assumed to be one.
     * @param point the input point to be transformed.
     * Transforms the point parameter with this Matrix4d and places the result
     * into pointOut. The fourth element of the point input paramter is assumed
     * to be one.
     * @param point the input point to be transformed.
     * @param pointOut the transformed point
     */
    transformPoint(point, pointOut) {
        pointOut = pointOut || point;
        pointOut.set_xyz(this.m00 * point.x + this.m01 * point.y + this.m02 * point.z + this.m03, this.m10 * point.x + this.m11 * point.y + this.m12 * point.z + this.m13, this.m20 * point.x + this.m21 * point.y + this.m22 * point.z + this.m23);
    }
    /**
     * Transforms the normal parameter by this transform and places the value
     * back into normal.  The fourth element of the normal is assumed to be zero.
     * @param normal the input normal to be transformed.
     * Transforms the normal parameter by this Matrix4d and places the value
     * into normalOut.  The fourth element of the normal is assumed to be zero.
     * @param normal the input normal to be transformed.
     * @param normalOut the transformed normal
     */
    transformVector(normal, normalOut) {
        normalOut = normalOut || normal;
        normalOut.set_xyz(this.m00 * normal.x + this.m01 * normal.y + this.m02 * normal.z, this.m10 * normal.x + this.m11 * normal.y + this.m12 * normal.z, this.m20 * normal.x + this.m21 * normal.y + this.m22 * normal.z);
    }
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * values in the double precision Matrix3d argument; the other elements of
     * this matrix are unchanged; a singular value decomposition is performed
     * on this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the passed rotation
     * components, and then the scale is reapplied to the rotational
     * components.
     * @param m1 double precision 3x3 matrix
     */
    setRotation_matrix(m1) {
        let scale = this.SVD(null, null);
        this.setRotationScale(m1);
        this.mulRotationScale(scale);
    }
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * equivalent values of the quaternion argument; the other elements of this
     * matrix are unchanged; a singular value decomposition is performed on
     * this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the matrix
     * equivalent of the quaternion, and then the scale is reapplied to the
     * rotational components.
     * @param q1 the quaternion that specifies the rotation
     */
    setRotation_quat(q1) {
        let scale = this.SVD(null, null);
        let tx = this.m03;
        let ty = this.m13;
        let tz = this.m23;
        let w0 = this.m30;
        let w1 = this.m31;
        let w2 = this.m32;
        let w3 = this.m33;
        this.set_quat(q1);
        this.mulRotationScale(scale);
        this.m03 = tx;
        this.m13 = ty;
        this.m23 = tz;
        this.m30 = w0;
        this.m31 = w1;
        this.m32 = w2;
        this.m33 = w3;
    }
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * equivalent values of the axis-angle argument; the other elements of this
     * matrix are unchanged; a singular value decomposition is performed on
     * this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the matrix
     * equivalent of the axis-angle, and then the scale is reapplied to the
     * rotational components.
     * @param a1 the axis-angle to be converted (x, y, z, angle)
     */
    setRotation_axisAngle(a1) {
        let scale = this.SVD(null, null);
        let tx = this.m03;
        let ty = this.m13;
        let tz = this.m23;
        let w0 = this.m30;
        let w1 = this.m31;
        let w2 = this.m32;
        let w3 = this.m33;
        this.set_axisAngle(a1);
        this.mulRotationScale(scale);
        this.m03 = tx;
        this.m13 = ty;
        this.m23 = tz;
        this.m30 = w0;
        this.m31 = w1;
        this.m32 = w2;
        this.m33 = w3;
    }
    /**
     * Sets this matrix to all zeros.*/
    setZero() {
        this.m00 = 0.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m03 = 0.0;
        this.m10 = 0.0;
        this.m11 = 0.0;
        this.m12 = 0.0;
        this.m13 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 0.0;
        this.m23 = 0.0;
        this.m30 = 0.0;
        this.m31 = 0.0;
        this.m32 = 0.0;
        this.m33 = 0.0;
    }
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix equal to the negation of of the Matrix4d
     * parameter.
     * @param m1 The source matrix
     */
    negate(m1) {
        if (m1)
            this.set_matrix4(m1);
        this.m00 = -this.m00;
        this.m01 = -this.m01;
        this.m02 = -this.m02;
        this.m03 = -this.m03;
        this.m10 = -this.m10;
        this.m11 = -this.m11;
        this.m12 = -this.m12;
        this.m13 = -this.m13;
        this.m20 = -this.m20;
        this.m21 = -this.m21;
        this.m22 = -this.m22;
        this.m23 = -this.m23;
        this.m30 = -this.m30;
        this.m31 = -this.m31;
        this.m32 = -this.m32;
        this.m33 = -this.m33;
    }
    /**
     * Sets 16 values
     */
    set(m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33) {
        this.m00 = m00;
        this.m01 = m01;
        this.m02 = m02;
        this.m03 = m03;
        this.m10 = m10;
        this.m11 = m11;
        this.m12 = m12;
        this.m13 = m13;
        this.m20 = m20;
        this.m21 = m21;
        this.m22 = m22;
        this.m23 = m23;
        this.m30 = m30;
        this.m31 = m31;
        this.m32 = m32;
        this.m33 = m33;
    }
    /**
     * Performs SVD on this matrix and gets scale and rotation.
     * Rotation is placed into rot.
     * @param rot3 the rotation factor(Matrix3d).
     * @param rot4 the rotation factor(Matrix4d) only upper 3x3 elements are changed.
     * @return scale factor
     */
    SVD(rot3, rot4) {
        let s = Math.sqrt((this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20 + this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21 + this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22) / 3.0);
        if (rot3) {
            this.getRotationScale(rot3);
            let n = 1 / Math.sqrt(this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20);
            rot3.m00 *= n;
            rot3.m10 *= n;
            rot3.m20 *= n;
            n = 1 / Math.sqrt(this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21);
            rot3.m01 *= n;
            rot3.m11 *= n;
            rot3.m21 *= n;
            n = 1 / Math.sqrt(this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22);
            rot3.m02 *= n;
            rot3.m12 *= n;
            rot3.m22 *= n;
        }
        if (rot4) {
            if (rot4)
                rot4.setRotationScale_4(this);
            let n = 1 / Math.sqrt(this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20);
            rot4.m00 *= n;
            rot4.m10 *= n;
            rot4.m20 *= n;
            n = 1 / Math.sqrt(this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21);
            rot4.m01 *= n;
            rot4.m11 *= n;
            rot4.m21 *= n;
            n = 1 / Math.sqrt(this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22);
            rot4.m02 *= n;
            rot4.m12 *= n;
            rot4.m22 *= n;
        }
        return s;
    }
    /**
     * Performs SVD on this matrix and gets the scale and the pure rotation.
     * The pure rotation is placed into rot.
     * @param rot the rotation factor.
     * @return scale factor
     */
    _SVD(rot) {
        let s = Math.sqrt((this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20 + this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21 + this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22) / 3.0);
        let t = (s === 0.0 ? 0.0 : 1.0 / s);
        if (rot) {
            this.getRotationScale(rot);
            rot.mul_scalar(t);
        }
        return (s);
    }
    /**
     * Multiplies 3x3 upper elements of this matrix by a scalar.
     * The other elements are unchanged.*/
    mulRotationScale(scale) {
        this.m00 *= scale;
        this.m01 *= scale;
        this.m02 *= scale;
        this.m10 *= scale;
        this.m11 *= scale;
        this.m12 *= scale;
        this.m20 *= scale;
        this.m21 *= scale;
        this.m22 *= scale;
    }
    /**
     * Sets only 3x3 upper elements of this matrix to that of m1.
     * The other elements are unchanged.*/
    setRotationScale_4(m1) {
        this.m00 = m1.m00;
        this.m01 = m1.m01;
        this.m02 = m1.m02;
        this.m10 = m1.m10;
        this.m11 = m1.m11;
        this.m12 = m1.m12;
        this.m20 = m1.m20;
        this.m21 = m1.m21;
        this.m22 = m1.m22;
    }
    setFromQuat(x, y, z, w) {
        let n = x * x + y * y + z * z + w * w;
        let s = (n > 0.0) ? (2.0 / n) : 0.0;
        let xs = x * s;
        let ys = y * s;
        let zs = z * s;
        let wx = w * xs;
        let wy = w * ys;
        let wz = w * zs;
        let xx = x * xs;
        let xy = x * ys;
        let xz = x * zs;
        let yy = y * ys;
        let yz = y * zs;
        let zz = z * zs;
        this.setIdentity();
        this.m00 = 1.0 - (yy + zz);
        this.m01 = xy - wz;
        this.m02 = xz + wy;
        this.m10 = xy + wz;
        this.m11 = 1.0 - (xx + zz);
        this.m12 = yz - wx;
        this.m20 = xz - wy;
        this.m21 = yz + wx;
        this.m22 = 1.0 - (xx + yy);
    }
    setFromAxisAngle(x, y, z, angle) {
        let n = Math.sqrt(x * x + y * y + z * z);
        n = 1 / n;
        x *= n;
        y *= n;
        z *= n;
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        let omc = 1.0 - c;
        this.m00 = c + x * x * omc;
        this.m11 = c + y * y * omc;
        this.m22 = c + z * z * omc;
        let tmp1 = x * y * omc;
        let tmp2 = z * s;
        this.m01 = tmp1 - tmp2;
        this.m10 = tmp1 + tmp2;
        tmp1 = x * z * omc;
        tmp2 = y * s;
        this.m02 = tmp1 + tmp2;
        this.m20 = tmp1 - tmp2;
        tmp1 = y * z * omc;
        tmp2 = x * s;
        this.m12 = tmp1 - tmp2;
        this.m21 = tmp1 + tmp2;
    }
}
//# sourceMappingURL=Matrix4d.js.map