import { Point3d } from './Point3d';
import { Vector3d } from './Vector3d';
import { Matrix3d } from './Matrix3d';
import { Tuple4d } from './Tuple4d';
import { Vector4d } from './Vector4d';
import { Quat4d } from './Quat4d';
import { AxisAngle4d } from './AxisAngle4d';
/**
 * A double precision floating point 4 by 4 matrix.
 * @version specification 1.1, implementation $Revision: 1.15 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export declare class Matrix4d {
    /**
     * The first element of the first row.*/
    m00: number;
    /**
     * The second element of the first row.*/
    m01: number;
    /**
     * third element of the first row.*/
    m02: number;
    /**
     * The fourth element of the first row.*/
    m03: number;
    /**
     * The first element of the second row.*/
    m10: number;
    /**
     * The second element of the second row.*/
    m11: number;
    /**
     * The third element of the second row.*/
    m12: number;
    /**
     * The fourth element of the second row.*/
    m13: number;
    /**
     * The first element of the third row.*/
    m20: number;
    /**
     * The second element of the third row.*/
    m21: number;
    /**
     * The third element of the third row.*/
    m22: number;
    /**
     * The fourth element of the third row.*/
    m23: number;
    /**
     * The first element of the fourth row.*/
    m30: number;
    /**
     * The second element of the fourth row.*/
    m31: number;
    /**
     * The third element of the fourth row.*/
    m32: number;
    /**
     * The fourth element of the fourth row.*/
    m33: number;
    /**
     * Constructs and initializes a Matrix4d from the specified 16 values.
     * @param m00 the [0][0] element
     * @param m01 the [0][1] element
     * @param m02 the [0][2] element
     * @param m03 the [0][3] element
     * @param m10 the [1][0] element
     * @param m11 the [1][1] element
     * @param m12 the [1][2] element
     * @param m13 the [1][3] element
     * @param m20 the [2][0] element
     * @param m21 the [2][1] element
     * @param m22 the [2][2] element
     * @param m23 the [2][3] element
     * @param m30 the [3][0] element
     * @param m31 the [3][1] element
     * @param m32 the [3][2] element
     * @param m33 the [3][3] element
     */
    constructor(m00: number, m01: number, m02: number, m03: number, m10: number, m11: number, m12: number, m13: number, m20: number, m21: number, m22: number, m23: number, m30: number, m31: number, m32: number, m33: number);
    /**
     * Constructs and initializes a Matrix4d from the specified 16
     * element array.  this.m00 =v[0], this.m01=v[1], etc.
     * @param v the array of length 16 containing in order
     */
    constructor(v: number[]);
    /**
     * Constructs and initializes a Matrix4d from the quaternion,
     * translation, and scale values; the scale is applied only to the
     * rotational components of the matrix (upper 3x3) and not to the
     * translational components.
     * @param q1  The quaternion value representing the rotational component
     * @param t1  The translational component of the matrix
     * @param s  The scale value applied to the rotational components
     */
    constructor(q1: Quat4d, t1: Vector3d, s: number);
    /**
     * Constructs a new matrix with the same values as the Matrix4d parameter.
     * @param m1 The source matrix.
     */
    constructor(m1: Matrix4d);
    /**
     * Constructs and initializes a Matrix4d from the rotation matrix,
     * translation, and scale values; the scale is applied only to the
     * rotational components of the matrix (upper 3x3) and not to the
     * translational components.
     * @param m1  The rotation matrix representing the rotational components
     * @param t1  The translational components of the matrix
     * @param s  The scale value applied to the rotational components
     */
    constructor(m1: Matrix3d, t1: Vector3d, s: number);
    /**
     * Constructs and initializes a Matrix4d to all zeros.
     */
    constructor();
    /**
     * Returns a string that contains the values of this Matrix4d.
     * @return the String representation
     */
    toString(): string;
    /**
     * Sets this Matrix4d to identity.*/
    setIdentity(): void;
    /**
     * Sets the specified element of this matrix4d to the value provided.
     * @param row  the row number to be modified (zero indexed)
     * @param column  the column number to be modified (zero indexed)
     * @param value the new value
     */
    setElement(row: number, column: number, value: number): void;
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row  the row number to be retrieved (zero indexed)
     * @param column  the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row: number, column: number): number;
    /**
     * Performs an SVD normalization of this matrix in order to acquire the
     * normalized rotational component; the values are placed into the Matrix3d parameter.
     * @param m1 matrix into which the rotational component is placed
     */
    get_matrix3(m1: Matrix3d): void;
    /**
     * Performs an SVD normalization of this matrix to calculate the rotation
     * as a 3x3 matrix, the translation, and the scale. None of the matrix values are modified.
     * @param m1 The normalized matrix representing the rotation
     * @param t1 The translation component
     * @return The scale component of this transform
     */
    get_matrix3_vector(m1: Matrix3d, t1: Vector3d): number;
    /**
     * Performs an SVD normalization of this matrix in order to acquire the
     * normalized rotational component; the values are placed into
     * the Quat4f parameter.
     * @param q1 quaternion into which the rotation component is placed
     */
    get_quat(q1: Quat4d): void;
    /**
     * Retrieves the translational components of this matrix.
     * @param trans the vector that will receive the translational component
     */
    get_translation(trans: Vector3d): void;
    /**
     * Gets the upper 3x3 values of this matrix and places them into the matrix m1.
     * @param m1 The matrix that will hold the values
     */
    getRotationScale(m1: Matrix3d): void;
    /**
     * Performs an SVD normalization of this matrix to calculate and return the
     * uniform scale factor. This matrix is not modified.
     * @return the scale factor of this matrix
     */
    getScale(): number;
    /**
     * Replaces the upper 3x3 matrix values of this matrix with the values in the matrix m1.
     * @param m1 The matrix that will be the new upper 3x3
     */
    setRotationScale(m1: Matrix3d): void;
    /**
     * Sets the scale component of the current matrix by factoring out the
     * current scale (by doing an SVD) from the rotational component and
     * multiplying by the new scale.
     * @param scale the new scale amount
     */
    setScale(scale: number): void;
    /**
     * Sets the specified row of this matrix4d to the four values provided.
     * @param row  the row number to be modified (zero indexed)
     * @param x the first column element
     * @param y the second column element
     * @param z the third column element
     * @param w the fourth column element
     */
    setRow_xyzw(row: number, x: number, y: number, z: number, w: number): void;
    /**
     * Sets the specified row of this matrix4d to the Vector provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_vector(row: number, v: Vector4d): void;
    /**
     * Sets the specified row of this matrix4d to the four values provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_array(row: number, v: number[]): void;
    /**
     * Copies the matrix values in the specified row into the
     * vector parameter.
     * @param row the matrix row
     * @param v The vector into which the matrix row values will be copied
     */
    getRow_vector(row: number, v: Vector4d): void;
    /**
     * Copies the matrix values in the specified row into the
     * array parameter.
     * @param row the matrix row
     * @param v The array into which the matrix row values will be copied
     */
    getRow_array(row: number, v: number[]): void;
    /**
     * Sets the specified column of this matrix4d to the four values provided.
     * @param column the column number to be modified (zero indexed)
     * @param x the first row element
     * @param y the second row element
     * @param z the third row element
     * @param w the fourth row element
     */
    setColumn_xyzw(column: number, x: number, y: number, z: number, w: number): void;
    /**
     * Sets the specified column of this matrix4d to the vector provided.
     * @param column the column number to be modified (zero indexed)
     * @param v the replacement column
     */
    setColumn_vector(column: number, v: Vector4d): void;
    /**
     * Sets the specified column of this matrix4d to the four values provided.
     * @param column  the column number to be modified (zero indexed)
     * @param v       the replacement column
     */
    setColumn_array(column: number, v: number[]): void;
    /**
     * Copies the matrix values in the specified column into the
     * vector parameter.
     * @param column the matrix column
     * @param v The vector into which the matrix column values will be copied
     */
    getColumn_vector(column: number, v: Vector4d): void;
    /**
     * Copies the matrix values in the specified column into the
     * array parameter.
     * @param column the matrix column
     * @param v The array into which the matrix column values will be copied
     */
    getColumn_array(column: number, v: number[]): void;
    /**
     * Adds a scalar to each component of this matrix.
     * @param scalar The scalar adder.

     * Adds a scalar to each component of the matrix m1 and places
     * the result into this. Matrix m1 is not modified.
     * @param scalar The scalar adder.
     * @parm m1 The original matrix values.
     */
    add_scalar(scalar: number, m1?: Matrix4d): void;
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix

     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add_matrix(m1: Matrix4d, m2?: Matrix4d): void;
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix

     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1: Matrix4d, m2?: Matrix4d): void;
    /**
     * Sets the value of this matrix to the transpose of the argument matrix
     * @param m1 the matrix to be transposed

     * Sets the value of this matrix to its transpose.
     */
    transpose(m1?: Matrix4d): void;
    /**
     * Sets the values in this Matrix4d equal to the row-major array parameter
     * (ie, the first four elements of the array will be copied into the first
     * row of this matrix, etc.).*/
    set_array(m: number[]): void;
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * values in the double precision Matrix3d argument; the other elements of
     * this matrix are initialized as if this were an identity matrix
     * (ie, affine matrix with no translational component).
     * @param m1 the 3x3 matrix
     */
    set_matrix3(m1: Matrix3d): void;
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * double precision axis and angle argument.
     * @param a1 the axis and angle to be converted
     */
    set_axisAngle(a1: AxisAngle4d): void;
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * (double precision) quaternion argument.
     * @param q1 the quaternion to be converted
     * Sets the value of this matrix from the rotation expressed by the
     * quaternion q1, the translation t1, and the scale s.
     * @param q1  the rotation expressed as a quaternion
     * @param t1  the translation
     * @param s  the scale value
     */
    set_quat(q1: Quat4d, t1?: Vector3d, s?: number): void;
    /**
     * Sets the value of this matrix to a copy of the
     * passed matrix m1.
     * @param m1 the matrix to be copied
     */
    set_matrix4(m1: Matrix4d): void;
    /**
     * Sets the value of this matrix to its inverse.

     * Sets the value of this matrix to the matrix inverse
     * of the passed matrix m1.
     * @param m1 the matrix to be inverted
     */
    invert(m1?: Matrix4d): void;
    /**
     * Computes the determinant of this matrix.
     * @return the determinant of the matrix
     */
    determinant(): number;
    /**
     * Sets the value of this matrix to a scale matrix with the
     * passed scale amount.
     * @param scale the scale factor for the matrix
     */
    set_scale(scale: number): void;
    /**
     * Sets the value of this matrix to a translate matrix by the
     * passed translation value.
     * @param v1 the translation amount
     */
    set_translation(v1: Vector3d): void;
    /**
     * Sets the value of this matrix to a scale and translation matrix;
     * scale is not applied to the translation and all of the matrix
     * values are modified.
     * @param scale the scale factor for the matrix
     * @param v1 the translation amount
     */
    set_scaleTranslation(scale: number, v1: Vector3d): void;
    /**
     * Sets the value of this matrix to a scale and translation matrix;
     * the translation is scaled by the scale factor and all of the
     * matrix values are modified.
     * @param v1 the translation amount
     * @param scale the scale factor for the matrix
     */
    set_translationScale(v1: Vector3d, scale: number): void;
    /**
     * Sets the value of this matrix from the rotation expressed by the
     * rotation matrix m1, the translation t1, and the scale s. The translation
     * is not modified by the scale.
     * @param m1 The rotation component
     * @param t1 The translation component
     * @param scale The scale component
     */
    set_rotaionTranslationScale(m1: Matrix3d, t1: Vector3d, scale: number): void;
    /**
     * Modifies the translational components of this matrix to the values of
     * the Vector3d argument; the other values of this matrix are not modified.
     * @param trans the translational component
     */
    setTranslation(trans: Vector3d): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the x axis
     * by the passed angle.
     * @param angle the angle to rotate about the X axis in radians
     */
    rotX(angle: number): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the y axis
     * by the passed angle.
     * @param angle the angle to rotate about the Y axis in radians
     */
    rotY(angle: number): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the z axis
     * by the passed angle.
     * @param angle the angle to rotate about the Z axis in radians
     */
    rotZ(angle: number): void;
    /**
     * Multiplies each element of this matrix by a scalar.
     * @param scalar The scalar multiplier.
     *
     * Multiplies each element of matrix m1 by a scalar and places the result
     * into this. Matrix m1 is not modified.
     * @param scalar The scalar multiplier.
     * @param m1 The original matrix.
     */
    mul_scalar(scalar: number, m1?: Matrix4d): void;
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1.
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul_matrix(m1: Matrix4d, m2?: Matrix4d): void;
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2,
     * and places the result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1: Matrix4d, m2: Matrix4d): void;
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeRight(m1: Matrix4d, m2: Matrix4d): void;
    /**
     * Multiplies the transpose of matrix m1 times matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1: Matrix4d, m2: Matrix4d): void;
    /**
     * Returns true if all of the data members of Matrix4d m1 are
     * equal to the corresponding data members in this Matrix4d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1: Matrix4d): boolean;
    /**
     * Returns true if the L-infinite distance between this matrix and matrix
     * m1 is less than or equal to the epsilon parameter, otherwise returns
     * false. The L-infinite distance is equal to MAX[i=0,1,2,3 ; j=0,1,2,3 ;
     * abs(this.m(i,j) - m1.m(i,j)]
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1: Matrix4d, epsilon: number): boolean;
    /**
     * Transform the vector vec using this Matrix4d and place the
     * result back into vec.
     * @param vec the double precision vector to be transformed
     * Transform the vector vec using this Matrix4d and place the
     * result into vecOut.
     * @param vec the double precision vector to be transformed
     * @param vecOut the vector into which the transformed values are placed
     */
    transform(vec: Tuple4d, vecOut?: Tuple4d): void;
    /**
     * Transforms the point parameter with this Matrix4d and
     * places the result back into point.  The fourth element of the
     * point input paramter is assumed to be one.
     * @param point the input point to be transformed.
     * Transforms the point parameter with this Matrix4d and places the result
     * into pointOut. The fourth element of the point input paramter is assumed
     * to be one.
     * @param point the input point to be transformed.
     * @param pointOut the transformed point
     */
    transformPoint(point: Point3d, pointOut?: Point3d): void;
    /**
     * Transforms the normal parameter by this transform and places the value
     * back into normal.  The fourth element of the normal is assumed to be zero.
     * @param normal the input normal to be transformed.
     * Transforms the normal parameter by this Matrix4d and places the value
     * into normalOut.  The fourth element of the normal is assumed to be zero.
     * @param normal the input normal to be transformed.
     * @param normalOut the transformed normal
     */
    transformVector(normal: Vector3d, normalOut?: Vector3d): void;
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * values in the double precision Matrix3d argument; the other elements of
     * this matrix are unchanged; a singular value decomposition is performed
     * on this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the passed rotation
     * components, and then the scale is reapplied to the rotational
     * components.
     * @param m1 double precision 3x3 matrix
     */
    setRotation_matrix(m1: Matrix3d): void;
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * equivalent values of the quaternion argument; the other elements of this
     * matrix are unchanged; a singular value decomposition is performed on
     * this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the matrix
     * equivalent of the quaternion, and then the scale is reapplied to the
     * rotational components.
     * @param q1 the quaternion that specifies the rotation
     */
    setRotation_quat(q1: Quat4d): void;
    /**
     * Sets the rotational component (upper 3x3) of this matrix to the matrix
     * equivalent values of the axis-angle argument; the other elements of this
     * matrix are unchanged; a singular value decomposition is performed on
     * this object's upper 3x3 matrix to factor out the scale, then this
     * object's upper 3x3 matrix components are replaced by the matrix
     * equivalent of the axis-angle, and then the scale is reapplied to the
     * rotational components.
     * @param a1 the axis-angle to be converted (x, y, z, angle)
     */
    setRotation_axisAngle(a1: AxisAngle4d): void;
    /**
     * Sets this matrix to all zeros.*/
    setZero(): void;
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix equal to the negation of of the Matrix4d
     * parameter.
     * @param m1 The source matrix
     */
    negate(m1?: Matrix4d): void;
    /**
     * Sets 16 values
     */
    private set(m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33);
    /**
     * Performs SVD on this matrix and gets scale and rotation.
     * Rotation is placed into rot.
     * @param rot3 the rotation factor(Matrix3d).
     * @param rot4 the rotation factor(Matrix4d) only upper 3x3 elements are changed.
     * @return scale factor
     */
    private SVD(rot3, rot4);
    /**
     * Performs SVD on this matrix and gets the scale and the pure rotation.
     * The pure rotation is placed into rot.
     * @param rot the rotation factor.
     * @return scale factor
     */
    private _SVD(rot);
    /**
     * Multiplies 3x3 upper elements of this matrix by a scalar.
     * The other elements are unchanged.*/
    private mulRotationScale(scale);
    /**
     * Sets only 3x3 upper elements of this matrix to that of m1.
     * The other elements are unchanged.*/
    private setRotationScale_4(m1);
    private setFromQuat(x, y, z, w);
    private setFromAxisAngle(x, y, z, angle);
}
