/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
/**
 * A double precision floating point 3 by 3 matrix.
 * Primarily to support rotations
 * @version specification 1.1, implementation $Revision: 1.14 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class Matrix3d {
    constructor(p1, m01, m02, m10, m11, m12, m20, m21, m22) {
        if (arguments.length === 0) {
            this.setZero();
        }
        else {
            if (p1 instanceof Matrix3d) {
                this.set_matrix(p1);
            }
            else if (Array.isArray(p1)) {
                let v = p1;
                this.set_array(v);
            }
            else if (typeof p1 === "number") {
                this.setValues(p1, m01, m02, m10, m11, m12, m20, m21, m22);
            }
        }
    }
    /**
     * Returns a string that contains the values of this Matrix3d.
     * @return the String representation
     */
    toString() {
        let nl = "\n"; //System.getProperty("line.separator");
        return "[" + nl +
            "  [" + this.m00 + "\t" + this.m01 + "\t" + this.m02 + "]" + nl +
            "  [" + this.m10 + "\t" + this.m11 + "\t" + this.m12 + "]" + nl +
            "  [" + this.m20 + "\t" + this.m21 + "\t" + this.m22 + "] ]";
    }
    /**
     * Sets this Matrix3d to identity.*/
    setIdentity() {
        this.m00 = 1.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m10 = 0.0;
        this.m11 = 1.0;
        this.m12 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 1.0;
    }
    /**
     * Sets the scale component of the current matrix by factoring out the
     * current scale (by doing an SVD) from the rotational component and
     * multiplying by the new scale.
     * @param scale the new scale amount
     */
    setScale(scale) {
        this.SVD(this);
        this.m00 *= scale;
        this.m11 *= scale;
        this.m22 *= scale;
    }
    /**
     * Sets the specified element of this matrix3d to the value provided.
     * @param row  the row number to be modified (zero indexed)
     * @param column  the column number to be modified (zero indexed)
     * @param value the new value
     */
    setElement(row, column, value) {
        if (row === 0) {
            if (column === 0)
                this.m00 = value;
            else if (column === 1)
                this.m01 = value;
            else if (column === 2)
                this.m02 = value;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else if (row === 1) {
            if (column === 0)
                this.m10 = value;
            else if (column === 1)
                this.m11 = value;
            else if (column === 2)
                this.m12 = value;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else if (row === 2) {
            if (column === 0)
                this.m20 = value;
            else if (column === 1)
                this.m21 = value;
            else if (column === 2)
                this.m22 = value;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else
            throw new Error("row must be 0 to 2 and is " + row);
    }
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row  the row number to be retrieved (zero indexed)
     * @param column  the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row, column) {
        if (row === 0) {
            if (column === 0)
                return this.m00;
            else if (column === 1)
                return this.m01;
            else if (column === 2)
                return this.m02;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else if (row === 1) {
            if (column === 0)
                return this.m10;
            else if (column === 1)
                return this.m11;
            else if (column === 2)
                return this.m12;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else if (row === 2) {
            if (column === 0)
                return this.m20;
            else if (column === 1)
                return this.m21;
            else if (column === 2)
                return this.m22;
            else
                throw new Error("col must be 0 to 2 and is " + column);
        }
        else
            throw new Error("row must be 0 to 2 and is " + row);
    }
    /**
     * Sets the specified row of this matrix3d to the three values provided.
     * @param row  the row number to be modified (zero indexed)
     * @param x the first column element
     * @param y the second column element
     * @param z the third column element
     */
    setRow_xyz(row, x, y, z) {
        if (row === 0) {
            this.m00 = x;
            this.m01 = y;
            this.m02 = z;
        }
        else if (row === 1) {
            this.m10 = x;
            this.m11 = y;
            this.m12 = z;
        }
        else if (row === 2) {
            this.m20 = x;
            this.m21 = y;
            this.m22 = z;
        }
        else {
            throw new Error("row must be 0 to 2 and is " + row);
        }
    }
    /**
     * Sets the specified row of this matrix3d to the Vector provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_vector(row, v) {
        if (row === 0) {
            this.m00 = v.x;
            this.m01 = v.y;
            this.m02 = v.z;
        }
        else if (row === 1) {
            this.m10 = v.x;
            this.m11 = v.y;
            this.m12 = v.z;
        }
        else if (row === 2) {
            this.m20 = v.x;
            this.m21 = v.y;
            this.m22 = v.z;
        }
        else {
            throw new Error("row must be 0 to 2 and is " + row);
        }
    }
    /**
     * Sets the specified row of this matrix3d to the four values provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_array(row, v) {
        if (row === 0) {
            this.m00 = v[0];
            this.m01 = v[1];
            this.m02 = v[2];
        }
        else if (row === 1) {
            this.m10 = v[0];
            this.m11 = v[1];
            this.m12 = v[2];
        }
        else if (row === 2) {
            this.m20 = v[0];
            this.m21 = v[1];
            this.m22 = v[2];
        }
        else {
            throw new Error("row must be 0 to 2 and is " + row);
        }
    }
    /**
     * Copies the matrix values in the specified row into the
     * array parameter.
     * @param row the matrix row
     * @param v The array into which the matrix row values will be copied
     */
    getRow_array(row, v) {
        if (row === 0) {
            v[0] = this.m00;
            v[1] = this.m01;
            v[2] = this.m02;
        }
        else if (row === 1) {
            v[0] = this.m10;
            v[1] = this.m11;
            v[2] = this.m12;
        }
        else if (row === 2) {
            v[0] = this.m20;
            v[1] = this.m21;
            v[2] = this.m22;
        }
        else {
            throw new Error("row must be 0 to 2 and is " + row);
        }
    }
    /**
     * Copies the matrix values in the specified row into the
     * vector parameter.
     * @param row the matrix row
     * @param v The vector into which the matrix row values will be copied
     */
    getRow_vector(row, v) {
        if (row === 0) {
            v.x = this.m00;
            v.y = this.m01;
            v.z = this.m02;
        }
        else if (row === 1) {
            v.x = this.m10;
            v.y = this.m11;
            v.z = this.m12;
        }
        else if (row === 2) {
            v.x = this.m20;
            v.y = this.m21;
            v.z = this.m22;
        }
        else {
            throw new Error("row must be 0 to 2 and is " + row);
        }
    }
    /**
     * Sets the specified column of this matrix3d to the three values provided.
     * @param column the column number to be modified (zero indexed)
     * @param x the first row element
     * @param y the second row element
     * @param z the third row element
     */
    setColumn_xyz(column, x, y, z) {
        if (column === 0) {
            this.m00 = x;
            this.m10 = y;
            this.m20 = z;
        }
        else if (column === 1) {
            this.m01 = x;
            this.m11 = y;
            this.m21 = z;
        }
        else if (column === 2) {
            this.m02 = x;
            this.m12 = y;
            this.m22 = z;
        }
        else {
            throw new Error("col must be 0 to 2 and is " + column);
        }
    }
    /**
     * Sets the specified column of this matrix3d to the vector provided.
     * @param column the column number to be modified (zero indexed)
     * @param v the replacement column
     */
    setColumn_vector(column, v) {
        if (column === 0) {
            this.m00 = v.x;
            this.m10 = v.y;
            this.m20 = v.z;
        }
        else if (column === 1) {
            this.m01 = v.x;
            this.m11 = v.y;
            this.m21 = v.z;
        }
        else if (column === 2) {
            this.m02 = v.x;
            this.m12 = v.y;
            this.m22 = v.z;
        }
        else {
            throw new Error("col must be 0 to 2 and is " + column);
        }
    }
    /**
     * Sets the specified column of this matrix3d to the four values provided.
     * @param column  the column number to be modified (zero indexed)
     * @param v       the replacement column
     */
    setColumn_array(column, v) {
        if (column === 0) {
            this.m00 = v[0];
            this.m10 = v[1];
            this.m20 = v[2];
        }
        else if (column === 1) {
            this.m01 = v[0];
            this.m11 = v[1];
            this.m21 = v[2];
        }
        else if (column === 2) {
            this.m02 = v[0];
            this.m12 = v[1];
            this.m22 = v[2];
        }
        else {
            throw new Error("col must be 0 to 2 and is " + column);
        }
    }
    /**
     * Copies the matrix values in the specified column into the vector
     * parameter.
     * @param column the matrix column
     * @param v The vector into which the matrix row values will be copied
     */
    getColumn(column, v) {
        if (column === 0) {
            v.x = this.m00;
            v.y = this.m10;
            v.z = this.m20;
        }
        else if (column === 1) {
            v.x = this.m01;
            v.y = this.m11;
            v.z = this.m21;
        }
        else if (column === 2) {
            v.x = this.m02;
            v.y = this.m12;
            v.z = this.m22;
        }
        else {
            throw new Error("column must be 0 to 2 and is " + column);
        }
    }
    /**
     * Copies the matrix values in the specified column into the array
     * parameter.
     * @param column the matrix column
     * @param v The array into which the matrix row values will be copied
     */
    getColumn_array(column, v) {
        if (column === 0) {
            v[0] = this.m00;
            v[1] = this.m10;
            v[2] = this.m20;
        }
        else if (column === 1) {
            v[0] = this.m01;
            v[1] = this.m11;
            v[2] = this.m21;
        }
        else if (column === 2) {
            v[0] = this.m02;
            v[1] = this.m12;
            v[2] = this.m22;
        }
        else {
            throw new Error("column must be 0 to 2 and is " + column);
        }
    }
    /**
     * Performs an SVD normalization of this matrix to calculate and return the
     * uniform scale factor. This matrix is not modified.
     * @return the scale factor of this matrix
     */
    getScale() {
        return this.SVD(null);
    }
    /**
     * Adds a scalar to each component of this matrix.
     * @param scalar The scalar adder.
     * Adds a scalar to each component of the matrix m1 and places
     * the result into this. Matrix m1 is not modified.
     * @param scalar The scalar adder.
     * @parm m1 The original matrix values.
     */
    add_scalar(scalar, m1) {
        if (m1)
            this.set_matrix(m1);
        this.m00 += scalar;
        this.m01 += scalar;
        this.m02 += scalar;
        this.m10 += scalar;
        this.m11 += scalar;
        this.m12 += scalar;
        this.m20 += scalar;
        this.m21 += scalar;
        this.m22 += scalar;
    }
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix
     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add(m1, m2) {
        if (m2) {
            this.setValues(m1.m00 + m2.m00, m1.m01 + m2.m01, m1.m02 + m2.m02, m1.m10 + m2.m10, m1.m11 + m2.m11, m1.m12 + m2.m12, m1.m20 + m2.m20, m1.m21 + m2.m21, m1.m22 + m2.m22);
        }
        else {
            this.m00 += m1.m00;
            this.m01 += m1.m01;
            this.m02 += m1.m02;
            this.m10 += m1.m10;
            this.m11 += m1.m11;
            this.m12 += m1.m12;
            this.m20 += m1.m20;
            this.m21 += m1.m21;
            this.m22 += m1.m22;
        }
    }
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix
     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1, m2) {
        if (m2) {
            this.setValues(m1.m00 - m2.m00, m1.m01 - m2.m01, m1.m02 - m2.m02, m1.m10 - m2.m10, m1.m11 - m2.m11, m1.m12 - m2.m12, m1.m20 - m2.m20, m1.m21 - m2.m21, m1.m22 - m2.m22);
        }
        else {
            this.m00 -= m1.m00;
            this.m01 -= m1.m01;
            this.m02 -= m1.m02;
            this.m10 -= m1.m10;
            this.m11 -= m1.m11;
            this.m12 -= m1.m12;
            this.m20 -= m1.m20;
            this.m21 -= m1.m21;
            this.m22 -= m1.m22;
        }
    }
    /**
     * Sets the value of this matrix to its transpose.
     * Sets the value of this matrix to the transpose of the argument matrix
     * @param m1 the matrix to be transposed
     */
    transpose(m1) {
        if (m1)
            this.set_matrix(m1);
        let tmp = this.m01;
        this.m01 = this.m10;
        this.m10 = tmp;
        tmp = this.m02;
        this.m02 = this.m20;
        this.m20 = tmp;
        tmp = this.m12;
        this.m12 = this.m21;
        this.m21 = tmp;
    }
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * (double precision) quaternion argument.
     * @param q1 the quaternion to be converted
     */
    set_quat(q1) {
        this.setFromQuat(q1.x, q1.y, q1.z, q1.w);
    }
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * double precision axis and angle argument.
     * @param a1 the axis and angle to be converted
     */
    set_axisAngle(a1) {
        this.setFromAxisAngle(a1.x, a1.y, a1.z, a1.angle);
    }
    /**
     * Sets the value of this matrix to the value of the Matrix3d
     * argument.*/
    set_matrix(m1) {
        this.m00 = m1.m00;
        this.m01 = m1.m01;
        this.m02 = m1.m02;
        this.m10 = m1.m10;
        this.m11 = m1.m11;
        this.m12 = m1.m12;
        this.m20 = m1.m20;
        this.m21 = m1.m21;
        this.m22 = m1.m22;
    }
    /**
     * Sets the values in this Matrix3d equal to the row-major array parameter
     * (ie, the first four elements of the array will be copied into the first
     * row of this matrix, etc.).*/
    set_array(m) {
        this.m00 = m[0];
        this.m01 = m[1];
        this.m02 = m[2];
        this.m10 = m[3];
        this.m11 = m[4];
        this.m12 = m[5];
        this.m20 = m[6];
        this.m21 = m[7];
        this.m22 = m[8];
    }
    /**
     * Sets the value of this matrix to its inverse.
     * Sets the value of this matrix to the matrix inverse
     * of the passed matrix m1.
     * @param m1 the matrix to be inverted
     */
    invert(m1) {
        if (m1)
            this.set_matrix(m1);
        let s = this.determinant();
        if (s === 0.0)
            return;
        s = 1 / s;
        this.setValues(this.m11 * this.m22 - this.m12 * this.m21, this.m02 * this.m21 - this.m01 * this.m22, this.m01 * this.m12 - this.m02 * this.m11, this.m12 * this.m20 - this.m10 * this.m22, this.m00 * this.m22 - this.m02 * this.m20, this.m02 * this.m10 - this.m00 * this.m12, this.m10 * this.m21 - this.m11 * this.m20, this.m01 * this.m20 - this.m00 * this.m21, this.m00 * this.m11 - this.m01 * this.m10);
        this.mul_scalar(s);
    }
    /**
     * Computes the determinant of this matrix.
     * @return the determinant of the matrix
     */
    determinant() {
        return this.m00 * (this.m11 * this.m22 - this.m21 * this.m12)
            - this.m01 * (this.m10 * this.m22 - this.m20 * this.m12)
            + this.m02 * (this.m10 * this.m21 - this.m20 * this.m11);
    }
    /**
     * Sets the value of this matrix to a scale matrix with the
     * passed scale amount.
     * @param scale the scale factor for the matrix
     */
    set_scale(scale) {
        this.m00 = scale;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m10 = 0.0;
        this.m11 = scale;
        this.m12 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = scale;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the x axis
     * by the passed angle.
     * @param angle the angle to rotate about the X axis in radians
     */
    rotX(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = 1.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m10 = 0.0;
        this.m11 = c;
        this.m12 = -s;
        this.m20 = 0.0;
        this.m21 = s;
        this.m22 = c;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the y axis
     * by the passed angle.
     * @param angle the angle to rotate about the Y axis in radians
     */
    rotY(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = c;
        this.m01 = 0.0;
        this.m02 = s;
        this.m10 = 0.0;
        this.m11 = 1.0;
        this.m12 = 0.0;
        this.m20 = -s;
        this.m21 = 0.0;
        this.m22 = c;
    }
    /**
     * Sets the value of this matrix to a rotation matrix about the z axis
     * by the passed angle.
     * @param angle the angle to rotate about the Z axis in radians
     */
    rotZ(angle) {
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        this.m00 = c;
        this.m01 = -s;
        this.m02 = 0.0;
        this.m10 = s;
        this.m11 = c;
        this.m12 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 1.0;
    }
    /**
     * Multiplies each element of this matrix by a scalar.
     * @param scalar The scalar multiplier.
     * Multiplies each element of matrix m1 by a scalar and places the result
     * into this. Matrix m1 is not modified.
     * @param scalar The scalar multiplier.
     * @param m1 The original matrix.
     */
    mul_scalar(scalar, m1) {
        if (m1)
            this.set_matrix(m1);
        this.m00 *= scalar;
        this.m01 *= scalar;
        this.m02 *= scalar;
        this.m10 *= scalar;
        this.m11 *= scalar;
        this.m12 *= scalar;
        this.m20 *= scalar;
        this.m21 *= scalar;
        this.m22 *= scalar;
    }
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1.
     * @param m1 the other matrix
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul_matrix(m1, m2) {
        if (typeof m2 === "undefined") {
            m2 = m1;
            m1 = this;
        }
        this.setValues(m1.m00 * m2.m00 + m1.m01 * m2.m10 + m1.m02 * m2.m20, m1.m00 * m2.m01 + m1.m01 * m2.m11 + m1.m02 * m2.m21, m1.m00 * m2.m02 + m1.m01 * m2.m12 + m1.m02 * m2.m22, m1.m10 * m2.m00 + m1.m11 * m2.m10 + m1.m12 * m2.m20, m1.m10 * m2.m01 + m1.m11 * m2.m11 + m1.m12 * m2.m21, m1.m10 * m2.m02 + m1.m11 * m2.m12 + m1.m12 * m2.m22, m1.m20 * m2.m00 + m1.m21 * m2.m10 + m1.m22 * m2.m20, m1.m20 * m2.m01 + m1.m21 * m2.m11 + m1.m22 * m2.m21, m1.m20 * m2.m02 + m1.m21 * m2.m12 + m1.m22 * m2.m22);
    }
    /**
     * Multiplies this matrix by matrix m1, does an SVD normalization of the
     * result, and places the result back into this matrix this =
     * SVDnorm(this*m1).
     * @param m1 the matrix on the right hand side of the multiplication
     * Multiplies matrix m1 by matrix m2, does an SVD normalization of the
     * result, and places the result into this matrix this = SVDnorm(m1*m2).
     * @param m1  the matrix on the left hand side of the multiplication
     * @param m2  the matrix on the right hand side of the multiplication
     */
    mulNormalize(m1, m2) {
        if (m2) {
            this.mul_matrix(m1, m2);
            this.SVD(this);
        }
        else {
            this.mul_matrix(m1);
            this.SVD(this);
        }
    }
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2,
     * and places the result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1, m2) {
        this.mul_matrix(m2, m1);
        this.transpose();
    }
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeRight(m1, m2) {
        this.setValues(m1.m00 * m2.m00 + m1.m01 * m2.m01 + m1.m02 * m2.m02, m1.m00 * m2.m10 + m1.m01 * m2.m11 + m1.m02 * m2.m12, m1.m00 * m2.m20 + m1.m01 * m2.m21 + m1.m02 * m2.m22, m1.m10 * m2.m00 + m1.m11 * m2.m01 + m1.m12 * m2.m02, m1.m10 * m2.m10 + m1.m11 * m2.m11 + m1.m12 * m2.m12, m1.m10 * m2.m20 + m1.m11 * m2.m21 + m1.m12 * m2.m22, m1.m20 * m2.m00 + m1.m21 * m2.m01 + m1.m22 * m2.m02, m1.m20 * m2.m10 + m1.m21 * m2.m11 + m1.m22 * m2.m12, m1.m20 * m2.m20 + m1.m21 * m2.m21 + m1.m22 * m2.m22);
    }
    /**
     * Multiplies the transpose of matrix m1 times matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1, m2) {
        this.setValues(m1.m00 * m2.m00 + m1.m10 * m2.m10 + m1.m20 * m2.m20, m1.m00 * m2.m01 + m1.m10 * m2.m11 + m1.m20 * m2.m21, m1.m00 * m2.m02 + m1.m10 * m2.m12 + m1.m20 * m2.m22, m1.m01 * m2.m00 + m1.m11 * m2.m10 + m1.m21 * m2.m20, m1.m01 * m2.m01 + m1.m11 * m2.m11 + m1.m21 * m2.m21, m1.m01 * m2.m02 + m1.m11 * m2.m12 + m1.m21 * m2.m22, m1.m02 * m2.m00 + m1.m12 * m2.m10 + m1.m22 * m2.m20, m1.m02 * m2.m01 + m1.m12 * m2.m11 + m1.m22 * m2.m21, m1.m02 * m2.m02 + m1.m12 * m2.m12 + m1.m22 * m2.m22);
    }
    /**
     * Performs singular value decomposition normalization of this matrix.

     * Perform singular value decomposition normalization of matrix m1 and
     * place the normalized values into this.
     * @param m1 Provides the matrix values to be normalized
     */
    normalize(m1) {
        if (m1)
            this.set_matrix(m1);
        this.SVD(this);
    }
    /**
     * Perform cross product normalization of this matrix.
     *
     * Perform cross product normalization of matrix m1 and place the
     * normalized values into this.
     * @param m1 Provides the matrix values to be normalized
     */
    normalizeCP(m1) {
        if (m1)
            this.set_matrix(m1);
        let s = Math.pow(Math.abs(this.determinant()), -1.0 / 3.0);
        this.mul_scalar(s);
    }
    /**
     * Returns true if all of the data members of Matrix3d m1 are
     * equal to the corresponding data members in this Matrix3d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1) {
        return m1 && this.m00 === m1.m00 && this.m01 === m1.m01 && this.m02 === m1.m02 &&
            this.m10 === m1.m10 && this.m11 === m1.m11 && this.m12 === m1.m12 &&
            this.m20 === m1.m20 && this.m21 === m1.m21 && this.m22 === m1.m22;
    }
    /**
     * Returns true if the L-infinite distance between this matrix and matrix
     * m1 is less than or equal to the epsilon parameter, otherwise returns
     * false. The L-infinite distance is equal to MAX[i=0,1,2,3 ; j=0,1,2,3 ;
     * abs(this.m(i,j) - m1.m(i,j)]
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1, epsilon) {
        return Math.abs(this.m00 - m1.m00) <= epsilon &&
            Math.abs(this.m01 - m1.m01) <= epsilon &&
            Math.abs(this.m02 - m1.m02) <= epsilon &&
            Math.abs(this.m10 - m1.m10) <= epsilon &&
            Math.abs(this.m11 - m1.m11) <= epsilon &&
            Math.abs(this.m12 - m1.m12) <= epsilon &&
            Math.abs(this.m20 - m1.m20) <= epsilon &&
            Math.abs(this.m21 - m1.m21) <= epsilon &&
            Math.abs(this.m22 - m1.m22) <= epsilon;
    }
    /**
     * Sets this matrix to all zeros. */
    setZero() {
        this.m00 = 0.0;
        this.m01 = 0.0;
        this.m02 = 0.0;
        this.m10 = 0.0;
        this.m11 = 0.0;
        this.m12 = 0.0;
        this.m20 = 0.0;
        this.m21 = 0.0;
        this.m22 = 0.0;
    }
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix equal to the negation of of the Matrix3d
     * parameter.
     * @param m1 The source matrix
     */
    negate(m1) {
        if (m1)
            this.set_matrix(m1);
        this.m00 = -this.m00;
        this.m01 = -this.m01;
        this.m02 = -this.m02;
        this.m10 = -this.m10;
        this.m11 = -this.m11;
        this.m12 = -this.m12;
        this.m20 = -this.m20;
        this.m21 = -this.m21;
        this.m22 = -this.m22;
    }
    /**
     * Transform the vector vec using this Matrix3d and place the
     * result back into vec.
     * @param vec the double precision vector to be transformed
     *
     * Transform the vector vec using this Matrix3d and place the
     * result into vecOut.
     * @paramt the double precision vector to be transformed
     * @param result the vector into which the transformed values are placed
     */
    transform(t, result) {
        if (!result)
            result = t;
        result.set_xyz(this.m00 * t.x + this.m01 * t.y + this.m02 * t.z, this.m10 * t.x + this.m11 * t.y + this.m12 * t.z, this.m20 * t.x + this.m21 * t.y + this.m22 * t.z);
    }
    /**
     * Sets 9 values
     */
    setValues(m00, m01, m02, m10, m11, m12, m20, m21, m22) {
        this.m00 = m00;
        this.m01 = m01;
        this.m02 = m02;
        this.m10 = m10;
        this.m11 = m11;
        this.m12 = m12;
        this.m20 = m20;
        this.m21 = m21;
        this.m22 = m22;
    }
    /**
     * Performs SVD on this matrix and gets scale and rotation.
     * Rotation is placed into rot.
     * @param rot the rotation factor.
     * @return scale factor
     */
    SVD(rot) {
        let s = Math.sqrt((this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20 + this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21 + this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22) / 3.0);
        if (rot) {
            let n = 1 / Math.sqrt(this.m00 * this.m00 + this.m10 * this.m10 + this.m20 * this.m20);
            rot.m00 = this.m00 * n;
            rot.m10 = this.m10 * n;
            rot.m20 = this.m20 * n;
            n = 1 / Math.sqrt(this.m01 * this.m01 + this.m11 * this.m11 + this.m21 * this.m21);
            rot.m01 = this.m01 * n;
            rot.m11 = this.m11 * n;
            rot.m21 = this.m21 * n;
            n = 1 / Math.sqrt(this.m02 * this.m02 + this.m12 * this.m12 + this.m22 * this.m22);
            rot.m02 = this.m02 * n;
            rot.m12 = this.m12 * n;
            rot.m22 = this.m22 * n;
        }
        return s;
    }
    setFromQuat(x, y, z, w) {
        let n = x * x + y * y + z * z + w * w;
        let s = (n > 0.0) ? (2.0 / n) : 0.0;
        let xs = x * s;
        let ys = y * s;
        let zs = z * s;
        let wx = w * xs;
        let wy = w * ys;
        let wz = w * zs;
        let xx = x * xs;
        let xy = x * ys;
        let xz = x * zs;
        let yy = y * ys;
        let yz = y * zs;
        let zz = z * zs;
        this.m00 = 1.0 - (yy + zz);
        this.m01 = xy - wz;
        this.m02 = xz + wy;
        this.m10 = xy + wz;
        this.m11 = 1.0 - (xx + zz);
        this.m12 = yz - wx;
        this.m20 = xz - wy;
        this.m21 = yz + wx;
        this.m22 = 1.0 - (xx + yy);
    }
    setFromAxisAngle(x, y, z, angle) {
        let n = Math.sqrt(x * x + y * y + z * z);
        n = 1 / n;
        x *= n;
        y *= n;
        z *= n;
        let c = Math.cos(angle);
        let s = Math.sin(angle);
        let omc = 1.0 - c;
        this.m00 = c + x * x * omc;
        this.m11 = c + y * y * omc;
        this.m22 = c + z * z * omc;
        let tmp1 = x * y * omc;
        let tmp2 = z * s;
        this.m01 = tmp1 - tmp2;
        this.m10 = tmp1 + tmp2;
        tmp1 = x * z * omc;
        tmp2 = y * s;
        this.m02 = tmp1 + tmp2;
        this.m20 = tmp1 - tmp2;
        tmp1 = y * z * omc;
        tmp2 = x * s;
        this.m12 = tmp1 - tmp2;
        this.m21 = tmp1 + tmp2;
    }
}
//# sourceMappingURL=Matrix3d.js.map