import { AxisAngle4d } from './AxisAngle4d';
import { Quat4d } from './Quat4d';
import { Vector3d } from './Vector3d';
import { Tuple3d } from './Tuple3d';
/**
 * A double precision floating point 3 by 3 matrix.
 * Primarily to support rotations
 * @version specification 1.1, implementation $Revision: 1.14 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export declare class Matrix3d {
    /**
     * The first element of the first row.*/
    m00: number;
    /**
     * The second element of the first row.*/
    m01: number;
    /**
     * third element of the first row.*/
    m02: number;
    /**
     * The first element of the second row.*/
    m10: number;
    /**
     * The second element of the second row.*/
    m11: number;
    /**
     * The third element of the second row.*/
    m12: number;
    /**
     * The first element of the third row.*/
    m20: number;
    /**
     * The second element of the third row.*/
    m21: number;
    /**
     * The third element of the third row.*/
    m22: number;
    /**
     * Constructs and initializes a Matrix3d from the specified nine values.
     * @param m00 the [0][0] element
     * @param m01 the [0][1] element
     * @param m02 the [0][2] element
     * @param m10 the [1][0] element
     * @param m11 the [1][1] element
     * @param m12 the [1][2] element
     * @param m20 the [2][0] element
     * @param m21 the [2][1] element
     * @param m22 the [2][2] element
     */
    constructor(m00: number, m01: number, m02: number, m10: number, m11: number, m12: number, m20: number, m21: number, m22: number);
    /**
     * Constructs and initializes a Matrix3d from the specified 9
     * element array.  this.m00 =v[0], this.m01=v[1], etc.
     * @param v the array of length 9 containing in order
     */
    constructor(v: number[]);
    /**
     * Constructs a new matrix with the same values as the Matrix3d parameter.
     * @param m1 The source matrix.
     */
    constructor(m1: Matrix3d);
    /**
     * Constructs and initializes a Matrix3d to all zeros.*/
    constructor();
    /**
     * Returns a string that contains the values of this Matrix3d.
     * @return the String representation
     */
    toString(): string;
    /**
     * Sets this Matrix3d to identity.*/
    setIdentity(): void;
    /**
     * Sets the scale component of the current matrix by factoring out the
     * current scale (by doing an SVD) from the rotational component and
     * multiplying by the new scale.
     * @param scale the new scale amount
     */
    setScale(scale: number): void;
    /**
     * Sets the specified element of this matrix3d to the value provided.
     * @param row  the row number to be modified (zero indexed)
     * @param column  the column number to be modified (zero indexed)
     * @param value the new value
     */
    setElement(row: number, column: number, value: number): void;
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row  the row number to be retrieved (zero indexed)
     * @param column  the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row: number, column: number): number;
    /**
     * Sets the specified row of this matrix3d to the three values provided.
     * @param row  the row number to be modified (zero indexed)
     * @param x the first column element
     * @param y the second column element
     * @param z the third column element
     */
    setRow_xyz(row: number, x: number, y: number, z: number): void;
    /**
     * Sets the specified row of this matrix3d to the Vector provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_vector(row: number, v: Vector3d): void;
    /**
     * Sets the specified row of this matrix3d to the four values provided.
     * @param row the row number to be modified (zero indexed)
     * @param v the replacement row
     */
    setRow_array(row: number, v: number[]): void;
    /**
     * Copies the matrix values in the specified row into the
     * array parameter.
     * @param row the matrix row
     * @param v The array into which the matrix row values will be copied
     */
    getRow_array(row: number, v: number[]): void;
    /**
     * Copies the matrix values in the specified row into the
     * vector parameter.
     * @param row the matrix row
     * @param v The vector into which the matrix row values will be copied
     */
    getRow_vector(row: number, v: Vector3d): void;
    /**
     * Sets the specified column of this matrix3d to the three values provided.
     * @param column the column number to be modified (zero indexed)
     * @param x the first row element
     * @param y the second row element
     * @param z the third row element
     */
    setColumn_xyz(column: number, x: number, y: number, z: number): void;
    /**
     * Sets the specified column of this matrix3d to the vector provided.
     * @param column the column number to be modified (zero indexed)
     * @param v the replacement column
     */
    setColumn_vector(column: number, v: Vector3d): void;
    /**
     * Sets the specified column of this matrix3d to the four values provided.
     * @param column  the column number to be modified (zero indexed)
     * @param v       the replacement column
     */
    setColumn_array(column: number, v: number[]): void;
    /**
     * Copies the matrix values in the specified column into the vector
     * parameter.
     * @param column the matrix column
     * @param v The vector into which the matrix row values will be copied
     */
    getColumn(column: number, v: Vector3d): void;
    /**
     * Copies the matrix values in the specified column into the array
     * parameter.
     * @param column the matrix column
     * @param v The array into which the matrix row values will be copied
     */
    getColumn_array(column: number, v: number[]): void;
    /**
     * Performs an SVD normalization of this matrix to calculate and return the
     * uniform scale factor. This matrix is not modified.
     * @return the scale factor of this matrix
     */
    getScale(): number;
    /**
     * Adds a scalar to each component of this matrix.
     * @param scalar The scalar adder.
     * Adds a scalar to each component of the matrix m1 and places
     * the result into this. Matrix m1 is not modified.
     * @param scalar The scalar adder.
     * @parm m1 The original matrix values.
     */
    add_scalar(scalar: number, m1?: Matrix3d): void;
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix
     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add(m1: Matrix3d, m2?: Matrix3d): void;
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix
     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1: Matrix3d, m2?: Matrix3d): void;
    /**
     * Sets the value of this matrix to its transpose.
     * Sets the value of this matrix to the transpose of the argument matrix
     * @param m1 the matrix to be transposed
     */
    transpose(m1?: Matrix3d): void;
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * (double precision) quaternion argument.
     * @param q1 the quaternion to be converted
     */
    set_quat(q1: Quat4d): void;
    /**
     * Sets the value of this matrix to the matrix conversion of the
     * double precision axis and angle argument.
     * @param a1 the axis and angle to be converted
     */
    set_axisAngle(a1: AxisAngle4d): void;
    /**
     * Sets the value of this matrix to the value of the Matrix3d
     * argument.*/
    set_matrix(m1: Matrix3d): void;
    /**
     * Sets the values in this Matrix3d equal to the row-major array parameter
     * (ie, the first four elements of the array will be copied into the first
     * row of this matrix, etc.).*/
    set_array(m: number[]): void;
    /**
     * Sets the value of this matrix to its inverse.
     * Sets the value of this matrix to the matrix inverse
     * of the passed matrix m1.
     * @param m1 the matrix to be inverted
     */
    invert(m1?: Matrix3d): void;
    /**
     * Computes the determinant of this matrix.
     * @return the determinant of the matrix
     */
    determinant(): number;
    /**
     * Sets the value of this matrix to a scale matrix with the
     * passed scale amount.
     * @param scale the scale factor for the matrix
     */
    set_scale(scale: number): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the x axis
     * by the passed angle.
     * @param angle the angle to rotate about the X axis in radians
     */
    rotX(angle: number): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the y axis
     * by the passed angle.
     * @param angle the angle to rotate about the Y axis in radians
     */
    rotY(angle: number): void;
    /**
     * Sets the value of this matrix to a rotation matrix about the z axis
     * by the passed angle.
     * @param angle the angle to rotate about the Z axis in radians
     */
    rotZ(angle: number): void;
    /**
     * Multiplies each element of this matrix by a scalar.
     * @param scalar The scalar multiplier.
     * Multiplies each element of matrix m1 by a scalar and places the result
     * into this. Matrix m1 is not modified.
     * @param scalar The scalar multiplier.
     * @param m1 The original matrix.
     */
    mul_scalar(scalar: number, m1?: Matrix3d): void;
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1.
     * @param m1 the other matrix
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul_matrix(m1: Matrix3d, m2?: Matrix3d): void;
    /**
     * Multiplies this matrix by matrix m1, does an SVD normalization of the
     * result, and places the result back into this matrix this =
     * SVDnorm(this*m1).
     * @param m1 the matrix on the right hand side of the multiplication
     * Multiplies matrix m1 by matrix m2, does an SVD normalization of the
     * result, and places the result into this matrix this = SVDnorm(m1*m2).
     * @param m1  the matrix on the left hand side of the multiplication
     * @param m2  the matrix on the right hand side of the multiplication
     */
    mulNormalize(m1: Matrix3d, m2?: Matrix3d): void;
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2,
     * and places the result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1: Matrix3d, m2: Matrix3d): void;
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeRight(m1: Matrix3d, m2: Matrix3d): void;
    /**
     * Multiplies the transpose of matrix m1 times matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1: Matrix3d, m2: Matrix3d): void;
    /**
     * Performs singular value decomposition normalization of this matrix.

     * Perform singular value decomposition normalization of matrix m1 and
     * place the normalized values into this.
     * @param m1 Provides the matrix values to be normalized
     */
    normalize(m1?: Matrix3d): void;
    /**
     * Perform cross product normalization of this matrix.
     *
     * Perform cross product normalization of matrix m1 and place the
     * normalized values into this.
     * @param m1 Provides the matrix values to be normalized
     */
    normalizeCP(m1?: Matrix3d): void;
    /**
     * Returns true if all of the data members of Matrix3d m1 are
     * equal to the corresponding data members in this Matrix3d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1: Matrix3d): boolean;
    /**
     * Returns true if the L-infinite distance between this matrix and matrix
     * m1 is less than or equal to the epsilon parameter, otherwise returns
     * false. The L-infinite distance is equal to MAX[i=0,1,2,3 ; j=0,1,2,3 ;
     * abs(this.m(i,j) - m1.m(i,j)]
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1: Matrix3d, epsilon: number): boolean;
    /**
     * Sets this matrix to all zeros. */
    setZero(): void;
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix equal to the negation of of the Matrix3d
     * parameter.
     * @param m1 The source matrix
     */
    negate(m1?: Matrix3d): void;
    /**
     * Transform the vector vec using this Matrix3d and place the
     * result back into vec.
     * @param vec the double precision vector to be transformed
     *
     * Transform the vector vec using this Matrix3d and place the
     * result into vecOut.
     * @paramt the double precision vector to be transformed
     * @param result the vector into which the transformed values are placed
     */
    transform(t: Tuple3d, result?: Tuple3d): void;
    /**
     * Sets 9 values
     */
    private setValues(m00, m01, m02, m10, m11, m12, m20, m21, m22);
    /**
     * Performs SVD on this matrix and gets scale and rotation.
     * Rotation is placed into rot.
     * @param rot the rotation factor.
     * @return scale factor
     */
    private SVD(rot);
    private setFromQuat(x, y, z, w);
    private setFromAxisAngle(x, y, z, angle);
}
