/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { Tuple2d } from './Tuple2d';
import { Tuple3d } from './Tuple3d';
import { Tuple4d } from './Tuple4d';
import { arraycopy } from './util';
/**
 * A double precision, general, and dynamically resizeable one
 * dimensional vector class. Index numbering begins with zero.
 * @version specification 1.1, implementation $Revision: 1.10 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class GVector {
    constructor(p1, p2) {
        let data = null;
        if (typeof p1 === "number") {
            this._setLength(p1);
        }
        else if (Array.isArray(p1)) {
            if (typeof p2 === "number") {
                this._setLength(p2);
            }
            else {
                this._setLength(p1.length);
            }
            data = p1;
        }
        else if (p1 instanceof GVector) {
            this._setLength(p1.elementCount);
            data = p1.elementData;
        }
        else if (p1 instanceof Tuple2d) {
            this._setLength(2);
            this.set_tuple2(p1);
        }
        else if (p1 instanceof Tuple3d) {
            this._setLength(3);
            this.set_tuple3(p1);
        }
        else if (p1 instanceof Tuple4d) {
            this._setLength(4);
            this.set_tuple4(p1);
        }
        if (data) {
            arraycopy(data, 0, this.elementData, 0, this.elementCount);
        }
    }
    _setLength(length) {
        this.elementCount = length;
        this.elementData = new Array(length);
    }
    /**
     * Returns the square root of the sum of the squares of this
     * vector (its length in n-dimensional space).
     * @return  length of this vector
     */
    norm() {
        return Math.sqrt(this.normSquared());
    }
    /**
     * Returns the sum of the squares of this vector (its length
     * sqaured in n-dimensional space). <p>
     * @return  length squared of this vector
     */
    normSquared() {
        let s = 0.0;
        for (let i = 0; i < this.elementCount; i++) {
            s += this.elementData[i] * this.elementData[i];
        }
        return s;
    }
    /**
     * Normalizes this vector in place.
     *
     * Sets the value of this vector to the normalization of
     * vector v1.
     * @param v1 the un-normalized vector
     */
    normalize(v1) {
        if (v1)
            this.set(v1);
        let len = this.norm();
        for (let i = 0; i < this.elementCount; i++)
            this.elementData[i] /= len;
    }
    /**
     * Scales this vector by the scale factor s.
     * @param s the scalar value
     *
     * Sets the value of this vector to the scalar multiplication of
     * the scale factor with the vector v1.
     * @param s the scalar value
     * @param v1 the source vector
     */
    scale(s, v1) {
        if (v1)
            this.set(v1);
        for (let i = 0; i < this.elementCount; i++)
            this.elementData[i] *= s;
    }
    /**
     * Sets the value of this vector to the scalar multiplication by
     * s of vector v1 plus vector v2 (this = s*v1 + v2).
     * @param s the scalar value
     * @param v1 the vector to be multiplied
     * @param v2 the vector to be added
     */
    scaleAdd(s, v1, v2) {
        let v1data = v1.elementData;
        let v2data = v2.elementData;
        if (this.elementCount !== v1.elementCount)
            throw new Error("this.size:" + this.elementCount + " != v1's size:" + v1.elementCount);
        if (this.elementCount !== v2.elementCount)
            throw new Error("this.size:" + this.elementCount + " != v2's size:" + v2.elementCount);
        for (let i = 0; i < this.elementCount; i++) {
            this.elementData[i] = s * v1data[i] + v2data[i];
        }
    }
    /**
     * Sets the value of this vector to sum of itself and the
     * specified vector
     * @param vector the second vector
     *
     * Sets the value of this vector to the vector sum of vectors
     * vector1 and vector2.
     * @param vector1 the first vector
     * @param vector2 the second vector
     */
    add(vector1, vector2) {
        if (vector2) {
            this.set(vector1);
            vector1 = vector2;
        }
        let v1data = vector1.elementData;
        if (this.elementCount !== vector1.elementCount)
            throw new Error("this.size:" + this.elementCount + " != v2's size:" + vector1.elementCount);
        for (let i = 0; i < this.elementCount; i++) {
            this.elementData[i] += v1data[i];
        }
    }
    /**
     * Sets the value of this vector to the vector difference of
     * itself and vector (this = this - vector).
     * @param vector - the other vector
     *
     * Sets the value of this vector to the vector difference of
     * vectors vector1 and vector2 (this = vector1 - vector2).
     * @param vector1 the first vector
     * @param vector2 the second vector
     */
    sub(vector1, vector2) {
        if (vector2) {
            this.set(vector1);
            vector1 = vector2;
        }
        let v1data = vector1.elementData;
        if (this.elementCount !== vector1.elementCount)
            throw new Error("this.size:" + this.elementCount + " != vector's size:" + vector1.elementCount);
        for (let i = 0; i < this.elementCount; i++) {
            this.elementData[i] -= v1data[i];
        }
    }
    /**
     * Multiplies matrix m1 times Vector v1 and places the result
     * into this vector (this = m1*v1).
     * @param m1 The matrix in the multiplication
     * @param v1 The vector that is multiplied
     */
    mul_matrix_vector(m1, v1) {
        let v1data = v1.elementData;
        let v1size = v1.elementCount;
        let nCol = m1.getNumCol();
        let nRow = m1.getNumRow();
        if (v1size !== nCol)
            throw new Error("v1.size:" + v1size + " != m1.nCol:" + nCol);
        if (this.elementCount !== nRow)
            throw new Error("this.size:" + this.elementCount + " != m1.nRow:" + nRow);
        for (let i = 0; i < this.elementCount; i++) {
            let sum = 0.0;
            for (let j = 0; j < nCol; j++) {
                sum += m1.getElement(i, j) * v1data[j];
            }
            this.elementData[i] = sum;
        }
    }
    /**
     * Multiplies the transpose of vector v1 (ie, v1 becomes a row
     * vector with respect to the multiplication) times matrix m1
     * and places the result into this vector
     * (this = transpose(v1)*m1). The result is technically a row
     * vector, but the GVector class only knows about column vectors,
     * and so the result is stored as a column vector.
     * @param m1 The matrix in the multiplication
     * @param v1 The vector that is temporarily transposed
     */
    mul_vector_matrix(v1, m1) {
        let v1data = v1.elementData;
        let v1size = v1.elementCount;
        let nCol = m1.getNumCol();
        let nRow = m1.getNumRow();
        if (v1size !== nRow)
            throw new Error("v1.size:" + v1size + " != m1.nRow:" + nRow);
        if (this.elementCount !== nCol)
            throw new Error("this.size:" + this.elementCount + " != m1.nCol:" + nCol);
        for (let i = 0; i < this.elementCount; i++) {
            let sum = 0.0;
            for (let j = 0; j < nRow; j++) {
                sum += m1.getElement(j, i) * v1data[j];
            }
            this.elementData[i] = sum;
        }
    }
    /**
     * Negates the value of this vector: this = -this. */
    negate() {
        for (let i = 0; i < this.elementCount; i++)
            this.elementData[i] = -this.elementData[i];
    }
    /**
     * Sets all the values in this vector to zero. */
    zero() {
        for (let i = 0; i < this.elementCount; i++)
            this.elementData[i] = 0.0;
    }
    /**
     * Changes the size of this vector dynamically. If the size is
     * increased no data values will be lost. If the size is
     * decreased, only those data values whose vector positions
     * were eliminated will be lost.
     * @param length number of desired elements in this vector
     */
    setSize(newSize) {
        if (newSize < 0)
            throw new Error("newSize:" + newSize + " < 0");
        if (this.elementCount < newSize) {
            let oldData = this.elementData;
            this.elementData = new Array(newSize);
            for (let i = 0; i < this.elementCount; i++)
                arraycopy(oldData, 0, this.elementData, 0, this.elementCount);
        }
        this.elementCount = newSize;
    }
    /**
     * Sets the value of this vector to the values found in the
     * array parameter. The array should be at least equal in
     * length to the number of elements in the vector.
     * @param vector the source array
     */
    set_array(vector) {
        arraycopy(vector, 0, this.elementData, 0, this.elementCount);
    }
    /**
     * Sets the value of this vector to the values found in
     * vector vector.
     * @param vector the source vector
     */
    set(vector) {
        arraycopy(vector.elementData, 0, this.elementData, 0, this.elementCount);
    }
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple2(tuple) {
        this.elementData[0] = tuple.x;
        this.elementData[1] = tuple.y;
    }
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple3(tuple) {
        this.elementData[0] = tuple.x;
        this.elementData[1] = tuple.y;
        this.elementData[2] = tuple.z;
    }
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple4(tuple) {
        this.elementData[0] = tuple.x;
        this.elementData[1] = tuple.y;
        this.elementData[2] = tuple.z;
        this.elementData[3] = tuple.w;
    }
    /**
     * Returns the number of elements in this vector.
     * @return  number of elements in this vector
     */
    getSize() {
        return this.elementCount;
    }
    /**
     * Retrieves the value at the specified index value of this
     * vector.
     * @param index the index of the element to retrieve (zero indexed)
     * @return  the value at the indexed element
     */
    getElement(index) {
        if (index < 0 || index > this.elementCount)
            throw new Error("index:" + index + "must be in [0, " + (this.elementCount - 1) + "]");
        return this.elementData[index];
    }
    /**
     * Modifies the value at the specified index of this vector.
     * @param index the index if the element to modify (zero indexed)
     * @param value the new vector element value
     */
    setElement(index, value) {
        if (index < 0 || index > this.elementCount)
            throw new Error("index:" + index + " must be in [0, " + (this.elementCount - 1) + "]");
        this.elementData[index] = value;
    }
    /**
     * Returns a string that contains the values of this GVector.
     * @return  the String representation
     */
    toString() {
        let buf = "";
        buf += "(";
        for (let i = 0; i < this.elementCount - 1; i++) {
            buf += this.elementData[i];
            buf += ",";
        }
        buf += this.elementData[this.elementCount - 1];
        buf += ")";
        return buf;
    }
    ///**
    // * Returns a hash number based on the data values in this
    // * object.  Two different GMatrix objects with identical data values
    // * (ie, returns true for equals(GMatrix) ) will return the same hash
    // * number.  Two objects with different data members may return the
    // * same hash value, although this is not likely.
    // * @return the integer hash value
    //*/
    //hashCode(): number {
    //    let hash: number = 0;
    //    for (let i: number = 0; i < elementCount; i++) {
    //        let bits: number = Double.doubleToLongBits(elementData[i]);
    //        hash ^= (((bits ^ (bits >> 32))) as number);
    //    }
    //    return hash;
    //}
    /**
     * Returns true if all of the data members of GVector vector1
     * are equal to the corresponding data members in this GVector.
     * @param vector1 The vector with which the comparison is made.
     * @return  true or false
     */
    equals(vector1) {
        if (!vector1)
            return false;
        if (this.elementCount !== vector1.elementCount)
            return false;
        let v1data = vector1.elementData;
        for (let i = 0; i < this.elementCount; i++) {
            if (this.elementData[i] !== v1data[i])
                return false;
        }
        return true;
    }
    /**
     * Returns true if the L-infinite distance between this vector
     * and vector v1 is less than or equal to the epsilon parameter,
     * otherwise returns false. The L-infinite distance is equal
     * to MAX[abs(x1-x2), abs(y1-y2), . . . ]. <p>
     * @param v1 The vector to be compared to this vector
     * @param epsilon the threshold value
     */
    epsilonEquals(v1, epsilon) {
        if (this.elementCount !== v1.elementCount)
            return false;
        let v1data = v1.elementData;
        for (let i = 0; i < this.elementCount; i++) {
            if (Math.abs(this.elementData[i] - v1data[i]) > epsilon)
                return false;
        }
        return true;
    }
    /**
     * Returns the dot product of this vector and vector v1.
     * @param v1 the other vector
     * @return  the dot product of this and v1
     */
    dot(v1) {
        let v1data = v1.elementData;
        if (this.elementCount !== v1.elementCount)
            throw new Error("this.size:" + this.elementCount + " != v1.size:" + v1.elementCount);
        let sum = 0.0;
        for (let i = 0; i < this.elementCount; ++i)
            sum += this.elementData[i] * v1data[i];
        return sum;
    }
    /**
     * Solves for x in Ax = b, where x is this vector (nx1),
     * A is mxn, b is mx1, and A = U*W*transpose(V);
     * U,W,V must be precomputed and can be found by taking the
     * singular value decomposition (SVD) of A using the method
     * SVD found in the GMatrix class.
     * @param U The U matrix produced by the GMatrix method SVD
     * @param W The W matrix produced by the GMatrix method SVD
     * @param V The V matrix produced by the GMatrix method SVD
     * @param b The b vector in the linear equation Ax = b
     */
    SVDBackSolve(U, W, V, b) {
        if (this.elementCount !== U.getNumRow() || this.elementCount !== U.getNumCol())
            throw new Error("this.size:" + this.elementCount + " != U.nRow,nCol:" + U.getNumRow() + "," + U.getNumCol());
        if (this.elementCount !== W.getNumRow())
            throw new Error("this.size:" + this.elementCount + " != W.nRow:" + W.getNumRow());
        if (b.elementCount !== W.getNumCol())
            throw new Error("b.size:" + b.elementCount + " != W.nCol:" + W.getNumCol());
        if (b.elementCount !== V.getNumRow() || b.elementCount !== V.getNumCol())
            throw new Error("b.size:" + this.elementCount + " != V.nRow,nCol:" + V.getNumRow() + "," + V.getNumCol());
        let m = U.getNumRow();
        let n = V.getNumRow();
        let tmp = new Array(n);
        for (let j = 0; j < n; j++) {
            let s = 0.0;
            let wj = W.getElement(j, j);
            if (wj !== 0.0) {
                for (let i = 0; i < m; i++)
                    s += U.getElement(i, j) * b.elementData[i];
                s /= wj;
            }
            tmp[j] = s;
        }
        for (let j = 0; j < n; j++) {
            let s = 0.0;
            for (let jj = 0; jj < n; jj++)
                s += V.getElement(j, jj) * tmp[jj];
            this.elementData[j] = s;
        }
    }
    /**
     * LU Decomposition Back Solve; this method takes the LU matrix
     * and the permutation vector produced by the GMatrix method LUD
     * and solves the equation (LU)*x = b by placing the solution
     * vector x into this vector. This vector should be the same
     * length or longer than b.
     * @param LU The matrix into which the lower and upper
     * decompositions have been placed
     * @param b The b vector in the equation (LU)*x = b
     * @param permutation The row permuations that were necessary
     * to produce the LU matrix parameter
     */
    LUDBackSolve(LU, b, permutation) {
        if (this.elementCount !== b.elementCount)
            throw new Error("this.size:" + this.elementCount + " != b.size:" + b.elementCount);
        if (this.elementCount !== LU.getNumRow())
            throw new Error("this.size:" + this.elementCount + " != LU.nRow:" + LU.getNumRow());
        if (this.elementCount !== LU.getNumCol())
            throw new Error("this.size:" + this.elementCount + " != LU.nCol:" + LU.getNumCol());
        let n = this.elementCount;
        let indx = permutation.elementData;
        let x = this.elementData;
        let bdata = b.elementData;
        for (let i = 0; i < n; i++) {
            x[i] = bdata[indx[i]];
        }
        let ii = -1;
        for (let i = 0; i < n; i++) {
            let sum = x[i];
            if (0 <= ii) {
                for (let j = ii; j <= i - 1; j++)
                    sum -= LU.getElement(i, j) * x[j];
            }
            else if (sum !== 0.0) {
                ii = i;
            }
            x[i] = sum;
        }
        for (let i = n - 1; i >= 0; i--) {
            let sum = x[i];
            for (let j = i + 1; j < n; j++)
                sum -= LU.getElement(i, j) * x[j];
            x[i] = sum / LU.getElement(i, i);
        }
    }
    /**
     * Returns the (n-space) angle in radians between this vector
     * and the vector parameter; the return value is constrained to
     * the range [0,PI].
     * @param v1 The other vector
     * @return  The angle in radians in the range [0,PI]
     */
    angle(v1) {
        return Math.acos(this.dot(v1) / this.norm() / v1.norm());
    }
    interpolate(v1, p2, alpha) {
        if (typeof p2 === "number") {
            alpha = p2;
        }
        else {
            this.set(v1);
            v1 = p2;
        }
        let v1data = v1.elementData;
        if (this.elementCount !== v1.elementCount)
            throw new Error("this.size:" + this.elementCount + " != v1.size:" + v1.elementCount);
        let beta = (1.0 - alpha);
        for (let i = 0; i < this.elementCount; ++i) {
            this.elementData[i] = beta * this.elementData[i] + alpha * v1data[i];
        }
    }
}
//# sourceMappingURL=GVector.js.map