import { GMatrix } from './GMatrix';
import { Tuple2d } from './Tuple2d';
import { Tuple3d } from './Tuple3d';
import { Tuple4d } from './Tuple4d';
/**
 * A double precision, general, and dynamically resizeable one
 * dimensional vector class. Index numbering begins with zero.
 * @version specification 1.1, implementation $Revision: 1.10 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export declare class GVector {
    private elementCount;
    private elementData;
    /**
     * Constructs a new generalized mathematic Vector with zero
     * elements; length reprents the number of elements in the
     * vector.
     * @param length number of elements in this vector.
     */
    constructor(length: number);
    /**
     * Constructs a new generalized mathematic Vector with zero
     * elements; length reprents the number of elements in the
     * vector. !! this comment is a bug in Sun's API !!
     * @param vector the values for the new vector.
     */
    constructor(vector: number[]);
    /**
     * Constructs a new GVector and copies the initial values from
     * the parameter vector.
     * @param vector the source for the new GVector's initial values
     */
    constructor(vector: GVector);
    /**
     * Constructs a new GVector and copies the initial values from
     * the Tuple
     * @param vector the source for the new GVector's initial values
     */
    constructor(tuple: Tuple2d);
    /**
     * Constructs a new GVector and copies the initial values from
     * the Tuple
     * @param vector the source for the new GVector's initial values
     */
    constructor(tuple: Tuple3d);
    /**
     * Constructs a new GVector and copies the initial values from
     * the Tuple
     * @param vector the source for the new GVector's initial values
     */
    constructor(tuple: Tuple4d);
    /**
     * Constructs a new GVector by copying length elements from the
     * array parameter.  The parameter length must be less than or
     * equal to vector.length.
     * @param vector The array from which the values will be copied.
     * @param length The number of values copied from the array.
     */
    constructor(vector: number[], length: number);
    private _setLength(length);
    /**
     * Returns the square root of the sum of the squares of this
     * vector (its length in n-dimensional space).
     * @return  length of this vector
     */
    norm(): number;
    /**
     * Returns the sum of the squares of this vector (its length
     * sqaured in n-dimensional space). <p>
     * @return  length squared of this vector
     */
    normSquared(): number;
    /**
     * Normalizes this vector in place.
     *
     * Sets the value of this vector to the normalization of
     * vector v1.
     * @param v1 the un-normalized vector
     */
    normalize(v1?: GVector): void;
    /**
     * Scales this vector by the scale factor s.
     * @param s the scalar value
     *
     * Sets the value of this vector to the scalar multiplication of
     * the scale factor with the vector v1.
     * @param s the scalar value
     * @param v1 the source vector
     */
    scale(s: number, v1?: GVector): void;
    /**
     * Sets the value of this vector to the scalar multiplication by
     * s of vector v1 plus vector v2 (this = s*v1 + v2).
     * @param s the scalar value
     * @param v1 the vector to be multiplied
     * @param v2 the vector to be added
     */
    scaleAdd(s: number, v1: GVector, v2: GVector): void;
    /**
     * Sets the value of this vector to sum of itself and the
     * specified vector
     * @param vector the second vector
     *
     * Sets the value of this vector to the vector sum of vectors
     * vector1 and vector2.
     * @param vector1 the first vector
     * @param vector2 the second vector
     */
    add(vector1: GVector, vector2?: GVector): void;
    /**
     * Sets the value of this vector to the vector difference of
     * itself and vector (this = this - vector).
     * @param vector - the other vector
     *
     * Sets the value of this vector to the vector difference of
     * vectors vector1 and vector2 (this = vector1 - vector2).
     * @param vector1 the first vector
     * @param vector2 the second vector
     */
    sub(vector1: GVector, vector2?: GVector): void;
    /**
     * Multiplies matrix m1 times Vector v1 and places the result
     * into this vector (this = m1*v1).
     * @param m1 The matrix in the multiplication
     * @param v1 The vector that is multiplied
     */
    mul_matrix_vector(m1: GMatrix, v1: GVector): void;
    /**
     * Multiplies the transpose of vector v1 (ie, v1 becomes a row
     * vector with respect to the multiplication) times matrix m1
     * and places the result into this vector
     * (this = transpose(v1)*m1). The result is technically a row
     * vector, but the GVector class only knows about column vectors,
     * and so the result is stored as a column vector.
     * @param m1 The matrix in the multiplication
     * @param v1 The vector that is temporarily transposed
     */
    mul_vector_matrix(v1: GVector, m1: GMatrix): void;
    /**
     * Negates the value of this vector: this = -this. */
    negate(): void;
    /**
     * Sets all the values in this vector to zero. */
    zero(): void;
    /**
     * Changes the size of this vector dynamically. If the size is
     * increased no data values will be lost. If the size is
     * decreased, only those data values whose vector positions
     * were eliminated will be lost.
     * @param length number of desired elements in this vector
     */
    setSize(newSize: number): void;
    /**
     * Sets the value of this vector to the values found in the
     * array parameter. The array should be at least equal in
     * length to the number of elements in the vector.
     * @param vector the source array
     */
    set_array(vector: number[]): void;
    /**
     * Sets the value of this vector to the values found in
     * vector vector.
     * @param vector the source vector
     */
    set(vector: GVector): void;
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple2(tuple: Tuple2d): void;
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple3(tuple: Tuple3d): void;
    /**
     * Sets the value of this vector to the values in tuple.
     * @param tuple the source for the new GVector's new values
     */
    set_tuple4(tuple: Tuple4d): void;
    /**
     * Returns the number of elements in this vector.
     * @return  number of elements in this vector
     */
    getSize(): number;
    /**
     * Retrieves the value at the specified index value of this
     * vector.
     * @param index the index of the element to retrieve (zero indexed)
     * @return  the value at the indexed element
     */
    getElement(index: number): number;
    /**
     * Modifies the value at the specified index of this vector.
     * @param index the index if the element to modify (zero indexed)
     * @param value the new vector element value
     */
    setElement(index: number, value: number): void;
    /**
     * Returns a string that contains the values of this GVector.
     * @return  the String representation
     */
    toString(): string;
    /**
     * Returns true if all of the data members of GVector vector1
     * are equal to the corresponding data members in this GVector.
     * @param vector1 The vector with which the comparison is made.
     * @return  true or false
     */
    equals(vector1: GVector): boolean;
    /**
     * Returns true if the L-infinite distance between this vector
     * and vector v1 is less than or equal to the epsilon parameter,
     * otherwise returns false. The L-infinite distance is equal
     * to MAX[abs(x1-x2), abs(y1-y2), . . . ]. <p>
     * @param v1 The vector to be compared to this vector
     * @param epsilon the threshold value
     */
    epsilonEquals(v1: GVector, epsilon: number): boolean;
    /**
     * Returns the dot product of this vector and vector v1.
     * @param v1 the other vector
     * @return  the dot product of this and v1
     */
    dot(v1: GVector): number;
    /**
     * Solves for x in Ax = b, where x is this vector (nx1),
     * A is mxn, b is mx1, and A = U*W*transpose(V);
     * U,W,V must be precomputed and can be found by taking the
     * singular value decomposition (SVD) of A using the method
     * SVD found in the GMatrix class.
     * @param U The U matrix produced by the GMatrix method SVD
     * @param W The W matrix produced by the GMatrix method SVD
     * @param V The V matrix produced by the GMatrix method SVD
     * @param b The b vector in the linear equation Ax = b
     */
    SVDBackSolve(U: GMatrix, W: GMatrix, V: GMatrix, b: GVector): void;
    /**
     * LU Decomposition Back Solve; this method takes the LU matrix
     * and the permutation vector produced by the GMatrix method LUD
     * and solves the equation (LU)*x = b by placing the solution
     * vector x into this vector. This vector should be the same
     * length or longer than b.
     * @param LU The matrix into which the lower and upper
     * decompositions have been placed
     * @param b The b vector in the equation (LU)*x = b
     * @param permutation The row permuations that were necessary
     * to produce the LU matrix parameter
     */
    LUDBackSolve(LU: GMatrix, b: GVector, permutation: GVector): void;
    /**
     * Returns the (n-space) angle in radians between this vector
     * and the vector parameter; the return value is constrained to
     * the range [0,PI].
     * @param v1 The other vector
     * @return  The angle in radians in the range [0,PI]
     */
    angle(v1: GVector): number;
    /**
     * Linearly interpolates between vectors v1 and v2 and places
     * the result into this tuple: this = (1-alpha)*v1 + alpha*v2.
     * @param v1 the first vector
     * @param v2 the second vector
     * @param alpha the alpha interpolation parameter
     */
    interpolate(v1: GVector, v2: GVector, alpha: number): void;
    /**
     * Linearly interpolates between this vector and vector v1 and
     * places the result into this tuple:
     * this = (1-alpha)*this + alpha*v1.
     * @param v1 the first vector
     * @param alpha the alpha interpolation parameter
     */
    interpolate(v1: GVector, alpha: number): void;
}
