/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
import { GVector } from './GVector';
import { arraycopy } from './util';
/**
 * A double precision, general, real, and dynamically resizeable
 * two dimensional N x M matrix class. Row and column numbering
 * begins with zero. The representation is row major.
 * @version specification 1.1, implementation $Revision: 1.13 $, $Date: 1999/11/25 10:30:23 $
 * @author Kenji hiranabe
 */
export class GMatrix {
    constructor(p1, nCol, matrix) {
        if (typeof p1 === "number") {
            let nRow = p1;
            if (nRow < 0)
                throw new Error(nRow + " < 0");
            if (nCol < 0)
                throw new Error(nCol + " < 0");
            this.nRow = nRow;
            this.nCol = nCol;
            this.elementData = new Array(nRow * nCol);
            if (matrix) {
                this.set_array(matrix);
            }
            else {
                this.setIdentity();
            }
        }
        else {
            let matrix = p1;
            this.nRow = matrix.nRow;
            this.nCol = matrix.nCol;
            let newSize = this.nRow * nCol;
            this.elementData = new Array(newSize);
            arraycopy(matrix.elementData, 0, this.elementData, 0, newSize);
        }
    }
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1 (this = this * m1).
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together (this = m1 * m2).
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul(m1, m2) {
        if (!m2) {
            m2 = m1;
            m1 = this;
        }
        if (this.nRow !== m1.nRow)
            throw new Error("nRow:" + this.nRow + " != m1.nRow:" + m1.nRow);
        if (this.nCol !== m2.nCol)
            throw new Error("nCol:" + this.nCol + " != m2.nCol:" + m2.nCol);
        if (m1.nCol !== m2.nRow)
            throw new Error("m1.nCol:" + m1.nCol + " != m2.nRow:" + m2.nRow);
        let newData = new Array(this.nCol * this.nRow);
        for (let i = 0; i < this.nRow; i++) {
            for (let j = 0; j < this.nCol; j++) {
                let sum = 0.0;
                for (let k = 0; k < m1.nCol; k++)
                    sum += m1.elementData[i * m1.nCol + k] * m2.elementData[k * m2.nCol + j];
                newData[i * this.nCol + j] = sum;
            }
        }
        this.elementData = newData;
    }
    /**
     * Computes the outer product of the two vectors; multiplies the
     * the first vector by the transpose of the second vector
     * and places the matrix result into this matrix. This matrix must
     * be as big or bigger than getSize(v1)xgetSize(v2).
     * @param v1 the first vector, treated as a row vector
     * @param v2 the second vector, treated as a column vector
     */
    mul_vector_vector(v1, v2) {
        if (this.nRow < v1.getSize())
            throw new Error("nRow:" + this.nRow + " < v1.getSize():" + v1.getSize());
        if (this.nCol < v2.getSize())
            throw new Error("nCol:" + this.nCol + " < v2.getSize():" + v2.getSize());
        for (let i = 0; i < this.nRow; i++)
            for (let j = 0; j < this.nCol; j++)
                this.elementData[i * this.nCol + j] = v1.getElement(i) * v2.getElement(j);
    }
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add(m1, m2) {
        if (m2) {
            if (this.nRow !== m1.nRow || this.nCol !== m1.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m1:(" + m1.nRow + "x" + m1.nCol + ").");
            if (this.nRow !== m2.nRow || this.nCol !== m2.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m2:(" + m2.nRow + "x" + m2.nCol + ").");
            for (let i = 0; i < this.nRow * this.nCol; i++)
                this.elementData[i] = m1.elementData[i] + m2.elementData[i];
        }
        else {
            if (this.nRow !== m1.nRow || this.nCol !== m1.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m1:(" + m1.nRow + "x" + m1.nCol + ").");
            for (let i = 0; i < this.nRow * this.nCol; i++)
                this.elementData[i] += m1.elementData[i];
        }
    }
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2 (this = m1 - m2).
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1, m2) {
        if (m2) {
            if (this.nRow !== m1.nRow || this.nCol !== m1.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m1:(" + m1.nRow + "x" + m1.nCol + ").");
            if (this.nRow !== m2.nRow || this.nCol !== m2.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m2:(" + m2.nRow + "x" + m2.nCol + ").");
            for (let i = 0; i < this.nRow * this.nCol; i++)
                this.elementData[i] = m1.elementData[i] - m2.elementData[i];
        }
        else {
            if (this.nRow !== m1.nRow || this.nCol !== m1.nCol)
                throw new Error("this:(" + this.nRow + "x" + this.nCol + ") != m1:(" + m1.nRow + "x" + m1.nCol + ").");
            for (let i = 0; i < this.nRow * this.nCol; i++)
                this.elementData[i] -= m1.elementData[i];
        }
    }
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix to the negation of the GMatrix parameter.
     * @param m1 The source matrix
     */
    negate(m1) {
        if (m1)
            this.set(m1);
        for (let i = 0; i < this.nRow * this.nCol; i++)
            this.elementData[i] = -this.elementData[i];
    }
    /**
     * Sets this GMatrix to the identity matrix.*/
    setIdentity() {
        this.setZero();
        let min = this.nRow < this.nCol ? this.nRow : this.nCol;
        for (let i = 0; i < min; i++)
            this.elementData[i * this.nCol + i] = 1.0;
    }
    /**
     * Sets all the values in this matrix to zero.*/
    setZero() {
        for (let i = 0; i < this.nRow * this.nCol; i++)
            this.elementData[i] = 0.0;
    }
    /**
     * Subtracts this matrix from the identity matrix and puts the values
     * back into this (this = I - this).*/
    identityMinus() {
        this.negate();
        let min = this.nRow < this.nCol ? this.nRow : this.nCol;
        for (let i = 0; i < min; i++)
            this.elementData[i * this.nCol + i] += 1.0;
    }
    /**
     * Inverts this matrix in place.
     *
     * Inverts matrix m1 and places the new values into this matrix.  Matrix
     * m1 is not modified.
     * @param m1 the matrix to be inverted
     */
    invert(m1) {
        if (m1)
            this.set(m1);
        if (this.nRow !== this.nCol)
            throw new Error("not a square matrix");
        let n = this.nRow;
        let LU = new GMatrix(n, n);
        let permutation = new GVector(n);
        let column = new GVector(n);
        let unit = new GVector(n);
        this.LUD(LU, permutation);
        for (let j = 0; j < n; j++) {
            unit.zero();
            unit.setElement(j, 1.0);
            column.LUDBackSolve(LU, unit, permutation);
            this.setColumn(j, column);
        }
    }
    /**
     * Copies a sub-matrix derived from this matrix into the target matrix.
     * The upper left of the sub-matrix is located at (rowSource, colSource);
     * the lower right of the sub-matrix is located at
     * (lastRowSource,lastColSource).  The sub-matrix is copied into the
     * the target matrix starting at (rowDest, colDest).
     * @param rowSource the top-most row of the sub-matrix
     * @param colSource the left-most column of the sub-matrix
     * @param numRow the number of rows in the sub-matrix
     * @param numCol the number of columns in the sub-matrix
     * @param rowDest the top-most row of the position of the copied sub-matrix
     * within the target matrix
     * @param colDest the left-most column of the position of the copied sub-matrix
     * within the target matrix
     * @param target the matrix into which the sub-matrix will be copied
     */
    copySubMatrix(rowSource, colSource, numRow, numCol, rowDest, colDest, target) {
        if (rowSource < 0 || colSource < 0 || rowDest < 0 || colDest < 0)
            throw new Error("rowSource,colSource,rowDest,colDest < 0.");
        else if (this.nRow < numRow + rowSource || this.nCol < numCol + colSource)
            throw new Error("Source GMatrix too small.");
        else if (target.nRow < numRow + rowDest || target.nCol < numCol + colDest)
            throw new Error("Target GMatrix too small.");
        for (let i = 0; i < numRow; i++)
            for (let j = 0; j < numCol; j++)
                target.elementData[(i + rowDest) * this.nCol + (j + colDest)] = this.elementData[(i + rowSource) * this.nCol + (j + colSource)];
    }
    /**
     * Changes the size of this matrix dynamically.  If the size is increased
     * no data values will be lost.  If the size is decreased, only those data
     * values whose matrix positions were eliminated will be lost.
     * @param nRow number of desired rows in this matrix
     * @param nCol number of desired columns in this matrix
     */
    setSize(nRow, nCol) {
        if (nRow < 0 || nCol < 0)
            throw new Error("nRow or nCol < 0");
        let oldnRow = this.nRow;
        let oldnCol = this.nCol;
        let oldSize = this.nRow * this.nCol;
        this.nRow = nRow;
        this.nCol = nCol;
        let newSize = nRow * nCol;
        let oldData = this.elementData;
        if (oldnCol === nCol) {
            if (nRow <= oldnRow)
                return;
            this.elementData = new Array(newSize);
            arraycopy(oldData, 0, this.elementData, 0, oldSize);
        }
        else {
            this.elementData = new Array(newSize);
            this.setZero();
            for (let i = 0; i < oldnRow; i++)
                arraycopy(oldData, i * oldnCol, this.elementData, i * nCol, oldnCol);
        }
    }
    /**
     * Sets the value of this matrix to the values found in the array parameter.
     * The values are copied in one row at a time, in row major
     * fashion.  The array should be at least equal in length to
     * the number of matrix rows times the number of matrix columns
     * in this matrix.
     * @param matrix the row major source array
     */
    set_array(matrix) {
        let size = this.nRow * this.nCol;
        arraycopy(matrix, 0, this.elementData, 0, size);
    }
    /**
     * Sets the value of this matrix to that of the Matrix3d provided.
     * @param m1 the source matrix
     */
    set_matrix3(m1) {
        this.elementData[0] = m1.m00;
        this.elementData[1] = m1.m01;
        this.elementData[2] = m1.m02;
        this.elementData[this.nCol] = m1.m10;
        this.elementData[this.nCol + 1] = m1.m11;
        this.elementData[this.nCol + 2] = m1.m12;
        this.elementData[2 * this.nCol] = m1.m20;
        this.elementData[2 * this.nCol + 1] = m1.m21;
        this.elementData[2 * this.nCol + 2] = m1.m22;
    }
    /**
     * Sets the value of this matrix to that of the Matrix4d provided.
     * @param m1 the source matrix
     */
    set_matrix4(m1) {
        this.elementData[0] = m1.m00;
        this.elementData[1] = m1.m01;
        this.elementData[2] = m1.m02;
        this.elementData[3] = m1.m03;
        this.elementData[this.nCol] = m1.m10;
        this.elementData[this.nCol + 1] = m1.m11;
        this.elementData[this.nCol + 2] = m1.m12;
        this.elementData[this.nCol + 3] = m1.m13;
        this.elementData[2 * this.nCol] = m1.m20;
        this.elementData[2 * this.nCol + 1] = m1.m21;
        this.elementData[2 * this.nCol + 2] = m1.m22;
        this.elementData[2 * this.nCol + 3] = m1.m23;
        this.elementData[3 * this.nCol] = m1.m30;
        this.elementData[3 * this.nCol + 1] = m1.m31;
        this.elementData[3 * this.nCol + 2] = m1.m32;
        this.elementData[3 * this.nCol + 3] = m1.m33;
    }
    /**
     * Sets the value of this matrix to the values found in matrix m1.
     * @param m1 the source matrix
     */
    set(m1) {
        if (m1.nRow < this.nRow || m1.nCol < this.nCol)
            throw new Error("m1 smaller than this matrix");
        arraycopy(m1.elementData, 0, this.elementData, 0, this.nRow * this.nCol);
    }
    /**
     * Returns the number of rows in this matrix.
     * @return number of rows in this matrix
     */
    getNumRow() {
        return this.nRow;
    }
    /**
     * Returns the number of colmuns in this matrix.
     * @return number of columns in this matrix
     */
    getNumCol() {
        return this.nCol;
    }
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row the row number to be retrieved (zero indexed)
     * @param column the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row, column) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        if (this.nCol <= column)
            throw new Error("column:" + column + " > matrix's nCol:" + this.nCol);
        if (column < 0)
            throw new Error("column:" + column + " < 0");
        return this.elementData[row * this.nCol + column];
    }
    /**
     * Modifies the value at the specified row and column of this matrix.
     * @param row the row number to be modified (zero indexed)
     * @param column the column number to be modified (zero indexed)
     * @param value the new matrix element value
     */
    setElement(row, column, value) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        if (this.nCol <= column)
            throw new Error("column:" + column + " > matrix's nCol:" + this.nCol);
        if (column < 0)
            throw new Error("column:" + column + " < 0");
        this.elementData[row * this.nCol + column] = value;
    }
    /**
     * Places the values of the specified row into the array parameter.
     * @param row the target row number
     * @param array the array into which the row values will be placed
     */
    getRow_array(row, array) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        if (array.length < this.nCol)
            throw new Error("array length:" + array.length + " smaller than matrix's nCol:" + this.nCol);
        arraycopy(this.elementData, row * this.nCol, array, 0, this.nCol);
    }
    /**
     * Places the values of the specified row into the vector parameter.
     * @param row the target row number
     * @param vector the vector into which the row values will be placed
     */
    getRow(row, vector) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        if (vector.getSize() < this.nCol)
            throw new Error("vector size:" + vector.getSize() + " smaller than matrix's nCol:" + this.nCol);
        for (let i = 0; i < this.nCol; i++) {
            vector.setElement(i, this.elementData[row * this.nCol + i]);
        }
    }
    /**
     * Places the values of the specified column into the array parameter.
     * @param col the target column number
     * @param array the array into which the column values will be placed
     */
    getColumn_array(col, array) {
        if (this.nCol <= col)
            throw new Error("col:" + col + " > matrix's nCol:" + this.nCol);
        if (col < 0)
            throw new Error("col:" + col + " < 0");
        if (array.length < this.nRow)
            throw new Error("array.length:" + array.length + " < matrix's nRow=" + this.nRow);
        for (let i = 0; i < this.nRow; i++)
            array[i] = this.elementData[i * this.nCol + col];
    }
    /**
     * Places the values of the specified column into the vector parameter.
     * @param col the target column number
     * @param vector the vector into which the column values will be placed
     */
    getColumn(col, vector) {
        if (this.nCol <= col)
            throw new Error("col:" + col + " > matrix's nCol:" + this.nCol);
        if (col < 0)
            throw new Error("col:" + col + " < 0");
        if (vector.getSize() < this.nRow)
            throw new Error("vector size:" + vector.getSize() + " < matrix's nRow:" + this.nRow);
        for (let i = 0; i < this.nRow; i++) {
            vector.setElement(i, this.elementData[i * this.nCol + col]);
        }
    }
    /**
     * Places the values in the upper 3X3 of this GMatrix into the matrix m1.
     * @param m1 The matrix that will hold the new values
     */
    get_matrix3(m1) {
        m1.m00 = this.elementData[0];
        m1.m01 = this.elementData[1];
        m1.m02 = this.elementData[2];
        m1.m10 = this.elementData[this.nCol];
        m1.m11 = this.elementData[this.nCol + 1];
        m1.m12 = this.elementData[this.nCol + 2];
        m1.m20 = this.elementData[2 * this.nCol];
        m1.m21 = this.elementData[2 * this.nCol + 1];
        m1.m22 = this.elementData[2 * this.nCol + 2];
    }
    /**
     * Places the values in the upper 4X4 of this GMatrix into the matrix m1.
     * @param m1 The matrix that will hold the new values
     */
    get_matrix4(m1) {
        m1.m00 = this.elementData[0];
        m1.m01 = this.elementData[1];
        m1.m02 = this.elementData[2];
        m1.m03 = this.elementData[3];
        m1.m10 = this.elementData[this.nCol];
        m1.m11 = this.elementData[this.nCol + 1];
        m1.m12 = this.elementData[this.nCol + 2];
        m1.m13 = this.elementData[this.nCol + 3];
        m1.m20 = this.elementData[2 * this.nCol];
        m1.m21 = this.elementData[2 * this.nCol + 1];
        m1.m22 = this.elementData[2 * this.nCol + 2];
        m1.m23 = this.elementData[2 * this.nCol + 3];
        m1.m30 = this.elementData[3 * this.nCol];
        m1.m31 = this.elementData[3 * this.nCol + 1];
        m1.m32 = this.elementData[3 * this.nCol + 2];
        m1.m33 = this.elementData[3 * this.nCol + 3];
    }
    /**
     * Places the values in the this matrix into the matrix m1; m1
     * should be at least as large as this GMatrix.
     * @param m1 The matrix that will hold the new values
     */
    get(m1) {
        if (m1.nRow < this.nRow || m1.nCol < this.nCol)
            throw new Error("m1 matrix is smaller than this matrix.");
        if (m1.nCol === this.nCol) {
            arraycopy(this.elementData, 0, m1.elementData, 0, this.nRow * this.nCol);
        }
        else {
            for (let i = 0; i < this.nRow; i++) {
                arraycopy(this.elementData, i * this.nCol, m1.elementData, i * m1.nCol, this.nCol);
            }
        }
    }
    /**
     * Copy the values from the array into the specified row of this
     * matrix.
     * @param row the row of this matrix into which the array values
     * will be copied.
     * @param array the source array
     */
    setRow_array(row, array) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        if (array.length < this.nCol)
            throw new Error("array length:" + array.length + " < matrix's nCol=" + this.nCol);
        arraycopy(array, 0, this.elementData, row * this.nCol, this.nCol);
    }
    /**
     * Copy the values from the array into the specified row of this
     * matrix.
     * @param row the row of this matrix into which the vector values
     * will be copied.
     * @param vector the source vector
     */
    setRow(row, vector) {
        if (this.nRow <= row)
            throw new Error("row:" + row + " > matrix's nRow:" + this.nRow);
        if (row < 0)
            throw new Error("row:" + row + " < 0");
        let vecSize = vector.getSize();
        if (vecSize < this.nCol)
            throw new Error("vector's size:" + vecSize + " < matrix's nCol=" + this.nCol);
        for (let i = 0; i < this.nCol; i++) {
            this.elementData[row * this.nCol + i] = vector.getElement(i);
        }
    }
    /**
     * Copy the values from the array into the specified column of this
     * matrix.
     * @param row the column of this matrix into which the array values
     * will be copied.
     * @param array the source array
     */
    setColumn_array(col, array) {
        if (this.nCol <= col)
            throw new Error("col:" + col + " > matrix's nCol=" + this.nCol);
        if (col < 0)
            throw new Error("col:" + col + " < 0");
        if (array.length < this.nRow)
            throw new Error("array length:" + array.length + " < matrix's nRow:" + this.nRow);
        for (let i = 0; i < this.nRow; i++)
            this.elementData[i * this.nCol + col] = array[i];
    }
    /**
     * Copy the values from the array into the specified column of this
     * matrix.
     * @param row the column of this matrix into which the vector values
     * will be copied.
     * @param vector the source vector
     */
    setColumn(col, vector) {
        if (this.nCol <= col)
            throw new Error("col:" + col + " > matrix's nCol=" + this.nCol);
        if (col < 0)
            throw new Error("col:" + col + " < 0");
        let vecSize = vector.getSize();
        if (vecSize < this.nRow)
            throw new Error("vector size:" + vecSize + " < matrix's nRow=" + this.nRow);
        for (let i = 0; i < this.nRow; i++)
            this.elementData[i * this.nCol + col] = vector.getElement(i);
    }
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1, m2) {
        this.mul(m2, m1);
        this.transpose();
    }
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this. */
    mulTransposeRight(m1, m2) {
        if (m1.nCol !== m2.nCol || this.nRow !== m1.nRow || this.nCol !== m2.nRow)
            throw new Error("matrices mismatch");
        for (let i = 0; i < this.nRow; i++) {
            for (let j = 0; j < this.nCol; j++) {
                let sum = 0.0;
                for (let k = 0; k < m1.nCol; k++)
                    sum += m1.elementData[i * m1.nCol + k] * m2.elementData[j * m2.nCol + k];
                this.elementData[i * this.nCol + j] = sum;
            }
        }
    }
    /**
     * Multiplies the transpose of matrix m1 times the matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1, m2) {
        this.transpose(m1);
        this.mul(m2);
    }
    /**
     * Transposes this matrix in place.
     *
     * Places the matrix values of the transpose of matrix m1 into this matrix. <p>
     * @param m1 the matrix to be transposed (but not modified)
     */
    transpose(m1) {
        if (m1)
            this.set(m1);
        for (let i = 0; i < this.nRow; i++) {
            for (let j = i + 1; j < this.nCol; j++) {
                let tmp = this.elementData[i * this.nCol + j];
                this.elementData[i * this.nCol + j] = this.elementData[j * this.nCol + i];
                this.elementData[j * this.nCol + i] = tmp;
            }
        }
    }
    /**
     * Returns a string that contains the values of this GMatrix.
     * @return the String representation
     */
    toString() {
        let nl = "\n";
        let out = "[";
        out += nl;
        for (let i = 0; i < this.nRow; i++) {
            out += "  [";
            for (let j = 0; j < this.nCol; j++) {
                if (0 < j)
                    out += "\t";
                out += this.elementData[i * this.nCol + j];
            }
            if (i + 1 < this.nRow) {
                out += "]";
                out += nl;
            }
            else {
                out += "] ]";
            }
        }
        return out.toString();
    }
    ///**
    // * Returns a hash number based on the data values in this
    // * object.  Two different GMatrix objects with identical data values
    // * (ie, returns true for equals(GMatrix) ) will return the same hash
    // * number.  Two objects with different data members may return the
    // * same hash value, although this is not likely.
    // * @return the integer hash value
    //*/
    //hashCode(): number {
    //    let hash: number = 0;
    //    for (let i: number = 0; i < this.nRow * this.nCol; i++) {
    //        let bits: number = Double.doubleToLongBits(elementData[i]);
    //        hash ^= (((bits ^ (bits >> 32))) as number);
    //    }
    //    return hash;
    //}
    /**
     * Returns true if all of the data members of Matrix4d m1 are
     * equal to the corresponding data members in this Matrix4d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1) {
        if (!m1)
            return false;
        if (m1.nRow !== this.nRow)
            return false;
        if (m1.nCol !== this.nCol)
            return false;
        for (let i = 0; i < this.nRow; i++)
            for (let j = 0; j < this.nCol; j++)
                if (this.elementData[i * this.nCol + j] !== m1.elementData[i * this.nCol + j])
                    return false;
        return true;
    }
    /**
     * Returns true if the L-infinite distance between this matrix and
     * matrix m1 is less than or equal to the epsilon parameter,
     * otherwise returns false. The L-infinite distance is equal to
     * MAX[i=0,1,2, . . .n ; j=0,1,2, . . .n ; abs(this.m(i,j) - m1.m(i,j)] .
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1, epsilon) {
        if (m1.nRow !== this.nRow)
            return false;
        if (m1.nCol !== this.nCol)
            return false;
        for (let i = 0; i < this.nRow; i++)
            for (let j = 0; j < this.nCol; j++)
                if (epsilon < Math.abs(this.elementData[i * this.nCol + j] - m1.elementData[i * this.nCol + j]))
                    return false;
        return true;
    }
    /**
     * Returns the trace of this matrix.
     * @return the trace of this matrix.
     */
    trace() {
        let min = this.nRow < this.nCol ? this.nRow : this.nCol;
        let trace = 0.0;
        for (let i = 0; i < min; i++)
            trace += this.elementData[i * this.nCol + i];
        return trace;
    }
    /**
     * Sets this matrix to a uniform scale matrix; all of the values are reset.
     * @param scale The new scale value
     */
    setScale(scale) {
        this.setZero();
        let min = this.nRow < this.nCol ? this.nRow : this.nCol;
        for (let i = 0; i < min; i++)
            this.elementData[i * this.nCol + i] = scale;
    }
    setDiag(i, value) {
        this.elementData[i * this.nCol + i] = value;
    }
    getDiag(i) {
        return this.elementData[i * this.nCol + i];
    }
    dpythag(a, b) {
        let absa = Math.abs(a);
        let absb = Math.abs(b);
        if (absa > absb) {
            if (absa === 0.0)
                return 0.0;
            let term = absb / absa;
            if (Math.abs(term) <= Number.MIN_VALUE)
                return absa;
            return (absa * Math.sqrt(1.0 + term * term));
        }
        else {
            if (absb === 0.0)
                return 0.0;
            let term = absa / absb;
            if (Math.abs(term) <= Number.MIN_VALUE)
                return absb;
            return (absb * Math.sqrt(1.0 + term * term));
        }
    }
    /**
     * Finds the singular value decomposition (SVD) of this matrix such that
     * this = U*W*transpose(V); and returns the rank of this matrix; the values
     * of U,W,V are all overwritten. Note that the matrix V is output as V,
     * and not transpose(V). If this matrix is mxn, then U is mxm, W is a
     * diagonal matrix that is mxn, and V is nxn. Using the notation W = diag(w),
     * then the inverse of this matrix is: inverse(this) = V*diag(1/w)*tranpose(U),
     * where diag(1/w) is the same matrix as W except that the reciprocal of each
     * of the diagonal components is used.
     * @param U The computed U matrix in the equation this = U*W*transpose(V)
     * @param W The computed W matrix in the equation this = U*W*transpose(V)
     * @param V The computed V matrix in the equation this = U*W*transpose(V)
     * @return The rank of this matrix.
     */
    SVD(u, w, v) {
        if (u.nRow !== this.nRow || u.nCol !== this.nRow)
            throw new Error("The U Matrix invalid size");
        if (v.nRow !== this.nCol || v.nCol !== this.nCol)
            throw new Error("The V Matrix invalid size");
        if (w.nCol !== this.nCol || w.nRow !== this.nRow)
            throw new Error("The W Matrix invalid size");
        let m = this.nRow;
        let n = this.nCol;
        let imax = m > n ? m : n;
        let A = u.elementData;
        let V = v.elementData;
        let i;
        let its;
        let j;
        let jj;
        let k;
        let l = 0;
        let nm = 0;
        let anorm;
        let c;
        let f;
        let g;
        let h;
        let s;
        let scale;
        let x;
        let y;
        let z;
        let rv1 = new Array(n);
        this.get(u);
        for (i = m; i < imax; i++)
            for (j = 0; j < imax; j++)
                A[i * m + j] = 0.0;
        for (j = n; j < imax; j++)
            for (i = 0; i < imax; i++)
                A[i * m + j] = 0.0;
        w.setZero();
        g = scale = anorm = 0.0;
        for (i = 0; i < n; i++) {
            l = i + 1;
            rv1[i] = scale * g;
            g = s = scale = 0.0;
            if (i < m) {
                for (k = i; k < m; k++)
                    scale += Math.abs(A[k * m + i]);
                if (scale !== 0.0) {
                    for (k = i; k < m; k++) {
                        A[k * m + i] /= scale;
                        s += A[k * m + i] * A[k * m + i];
                    }
                    f = A[i * m + i];
                    g = (f < 0.0 ? Math.sqrt(s) : -Math.sqrt(s));
                    h = f * g - s;
                    A[i * m + i] = f - g;
                    for (j = l; j < n; j++) {
                        for (s = 0.0, k = i; k < m; k++)
                            s += A[k * m + i] * A[k * m + j];
                        f = s / h;
                        for (k = i; k < m; k++)
                            A[k * m + j] += f * A[k * m + i];
                    }
                    for (k = i; k < m; k++)
                        A[k * m + i] *= scale;
                }
            }
            w.setDiag(i, scale * g);
            g = s = scale = 0.0;
            if (i < m && i !== n - 1) {
                for (k = l; k < n; k++)
                    scale += Math.abs(A[i * m + k]);
                if (scale !== 0.0) {
                    for (k = l; k < n; k++) {
                        A[i * m + k] /= scale;
                        s += A[i * m + k] * A[i * m + k];
                    }
                    f = A[i * m + l];
                    g = (f < 0.0 ? Math.sqrt(s) : -Math.sqrt(s));
                    h = f * g - s;
                    A[i * m + l] = f - g;
                    for (k = l; k < n; k++)
                        rv1[k] = A[i * m + k] / h;
                    for (j = l; j < m; j++) {
                        for (s = 0.0, k = l; k < n; k++)
                            s += A[j * m + k] * A[i * m + k];
                        for (k = l; k < n; k++)
                            A[j * m + k] += s * rv1[k];
                    }
                    for (k = l; k < n; k++)
                        A[i * m + k] *= scale;
                }
            }
            let a1 = Math.abs(w.getDiag(i)) + Math.abs(rv1[i]);
            if (a1 > anorm)
                anorm = a1;
        }
        for (i = n - 1; i >= 0; i--) {
            if (i < n - 1) {
                if (g !== 0.0) {
                    for (j = l; j < n; j++)
                        V[j * n + i] = (A[i * m + j] / A[i * m + l]) / g;
                    for (j = l; j < n; j++) {
                        for (s = 0.0, k = l; k < n; k++)
                            s += A[i * m + k] * V[k * n + j];
                        for (k = l; k < n; k++)
                            V[k * n + j] += s * V[k * n + i];
                    }
                }
                for (j = l; j < n; j++)
                    V[i * n + j] = V[j * n + i] = 0.0;
            }
            V[i * n + i] = 1.0;
            g = rv1[i];
            l = i;
        }
        let imin = m < n ? m : n;
        for (i = imin - 1; i >= 0; i--) {
            l = i + 1;
            g = w.getDiag(i);
            for (j = l; j < n; j++)
                A[i * m + j] = 0.0;
            if (g !== 0.0) {
                g = 1.0 / g;
                for (j = l; j < n; j++) {
                    for (s = 0.0, k = l; k < m; k++)
                        s += A[k * m + i] * A[k * m + j];
                    f = (s / A[i * m + i]) * g;
                    for (k = i; k < m; k++)
                        A[k * m + j] += f * A[k * m + i];
                }
                for (j = i; j < m; j++)
                    A[j * m + i] *= g;
            }
            else
                for (j = i; j < m; j++)
                    A[j * m + i] = 0.0;
            ++A[i * m + i];
        }
        for (k = n - 1; k >= 0; k--) {
            for (its = 1; its <= 30; its++) {
                let flag = true;
                for (l = k; l >= 0; l--) {
                    nm = l - 1;
                    if ((Math.abs(rv1[l]) + anorm) === anorm) {
                        flag = false;
                        break;
                    }
                    if ((Math.abs(w.getDiag(nm)) + anorm) === anorm)
                        break;
                }
                if (flag) {
                    c = 0.0;
                    s = 1.0;
                    for (i = l; i <= k; i++) {
                        f = s * rv1[i];
                        rv1[i] = c * rv1[i];
                        if ((Math.abs(f) + anorm) === anorm)
                            break;
                        g = w.getDiag(i);
                        h = this.dpythag(f, g);
                        w.setDiag(i, h);
                        h = 1.0 / h;
                        c = g * h;
                        s = -f * h;
                        for (j = 0; j < m; j++) {
                            y = A[j * m + nm];
                            z = A[j * m + i];
                            A[j * m + nm] = y * c + z * s;
                            A[j * m + i] = z * c - y * s;
                        }
                    }
                }
                z = w.getDiag(k);
                if (l === k) {
                    if (z < 0.0) {
                        w.setDiag(k, -z);
                        for (j = 0; j < n; j++)
                            V[j * n + k] = -V[j * n + k];
                    }
                    break;
                }
                if (its === 30) {
                    return 0;
                }
                x = w.getDiag(l);
                nm = k - 1;
                y = w.getDiag(nm);
                g = rv1[nm];
                h = rv1[k];
                f = ((y - z) * (y + z) + (g - h) * (g + h)) / (2.0 * h * y);
                g = this.dpythag(f, 1.0);
                f = ((x - z) * (x + z) + h * ((y / (f + (f >= 0.0 ? Math.abs(g) : -Math.abs(g)))) - h)) / x;
                c = s = 1.0;
                for (j = l; j <= nm; j++) {
                    i = j + 1;
                    g = rv1[i];
                    y = w.getDiag(i);
                    h = s * g;
                    g = c * g;
                    z = this.dpythag(f, h);
                    rv1[j] = z;
                    c = f / z;
                    s = h / z;
                    f = x * c + g * s;
                    g = g * c - x * s;
                    h = y * s;
                    y *= c;
                    for (jj = 0; jj < n; jj++) {
                        x = V[jj * n + j];
                        z = V[jj * n + i];
                        V[jj * n + j] = x * c + z * s;
                        V[jj * n + i] = z * c - x * s;
                    }
                    z = this.dpythag(f, h);
                    w.setDiag(j, z);
                    if (z !== 0.0) {
                        z = 1.0 / z;
                        c = f * z;
                        s = h * z;
                    }
                    f = c * g + s * y;
                    x = c * y - s * g;
                    for (jj = 0; jj < m; jj++) {
                        y = A[jj * m + j];
                        z = A[jj * m + i];
                        A[jj * m + j] = y * c + z * s;
                        A[jj * m + i] = z * c - y * s;
                    }
                }
                rv1[l] = 0.0;
                rv1[k] = f;
                w.setDiag(k, x);
            }
        }
        let rank = 0;
        for (i = 0; i < n; i++)
            if (w.getDiag(i) > 0.0)
                rank++;
        return rank;
    }
    swapRows(i, j) {
        for (let k = 0; k < this.nCol; k++) {
            let tmp = this.elementData[i * this.nCol + k];
            this.elementData[i * this.nCol + k] = this.elementData[j * this.nCol + k];
            this.elementData[j * this.nCol + k] = tmp;
        }
    }
    /**
     * LU Decomposition; this matrix must be a square matrix; the LU GMatrix
     * parameter must be the same size as this matrix. The matrix LU will be
     * overwritten as the combination of a lower diagonal and upper diagonal
     * matrix decompostion of this matrix; the diagonal elements of L (unity)
     * are not stored. The GVector parameter records the row permutation
     * effected by the partial pivoting, and is used as a parameter to the
     * GVector method LUDBackSolve to solve sets of linear equations. This
     * method returns +/- 1 depending on whether the number of row interchanges
     * was even or odd, respectively.
     * @param permutation The row permutation effected by the
     * partial pivoting
     * @return +-1 depending on whether the number of row interchanges
     * was even or odd respectively
     */
    LUD(LU, permutation) {
        if (this.nRow !== this.nCol)
            throw new Error("not a square matrix");
        let n = this.nRow;
        if (n !== LU.nRow)
            throw new Error("this.nRow:" + n + " != LU.nRow:" + LU.nRow);
        if (n !== LU.nCol)
            throw new Error("this.nCol:" + n + " != LU.nCol:" + LU.nCol);
        if (permutation.getSize() < n)
            throw new Error("permutation.size:" + permutation.getSize() + " < this.nCol:" + n);
        if (this !== LU)
            LU.set(this);
        let even = 1;
        let a = LU.elementData;
        for (let i = 0; i < n; i++)
            permutation.setElement(i, i);
        for (let j = 0; j < n; j++) {
            let big;
            let dum;
            let sum;
            let imax;
            for (let i = 0; i < j; i++) {
                sum = a[i * n + j];
                for (let k = 0; k < i; k++) {
                    if (a[i * n + k] !== 0.0 && a[k * n + j] !== 0.0)
                        sum -= a[i * n + k] * a[k * n + j];
                }
                a[i * n + j] = sum;
            }
            big = 0.0;
            imax = j;
            for (let i = j; i < n; i++) {
                sum = a[i * n + j];
                for (let k = 0; k < j; k++) {
                    if (a[i * n + k] !== 0.0 && a[k * n + j] !== 0.0)
                        sum -= a[i * n + k] * a[k * n + j];
                }
                a[i * n + j] = sum;
                dum = Math.abs(sum);
                if (dum >= big) {
                    big = dum;
                    imax = i;
                }
            }
            if (j !== imax) {
                LU.swapRows(imax, j);
                let tmp = permutation.getElement(imax);
                permutation.setElement(imax, permutation.getElement(j));
                permutation.setElement(j, tmp);
                even = -even;
            }
            if (j !== n - 1) {
                dum = 1.0 / a[j * n + j];
                for (let i = j + 1; i < n; i++)
                    a[i * n + j] *= dum;
            }
        }
        return even;
    }
}
//# sourceMappingURL=GMatrix.js.map