import { GVector } from './GVector';
import { Matrix3d } from './Matrix3d';
import { Matrix4d } from './Matrix4d';
/**
 * A double precision, general, real, and dynamically resizeable
 * two dimensional N x M matrix class. Row and column numbering
 * begins with zero. The representation is row major.
 * @version specification 1.1, implementation $Revision: 1.13 $, $Date: 1999/11/25 10:30:23 $
 * @author Kenji hiranabe
 */
export declare class GMatrix {
    /**
     * The data of the GMatrix.(1D array. The (i,j) element is stored in elementData[i*nCol + j])*/
    private elementData;
    /**
     * The number of rows in this matrix.*/
    private nRow;
    /**
     * The number of columns in this matrix.*/
    private nCol;
    /**
     * Constructs an nRow by nCol identity matrix.
     * Note that even though row and column numbering begins with
     * zero, nRow and nCol will be one larger than the maximum
     * possible matrix index values.
     * @param nRow number of rows in this matrix.
     * @param nCol number of columns in this matrix.
     */
    constructor(nRow: number, nCol: number);
    /**
     * Constructs an nRow by nCol matrix initialized to the values
     * in the matrix array.  The array values are copied in one row at
     * a time in row major fashion.  The array should be at least
     * nRow*nCol in length.
     * Note that even though row and column numbering begins with
     * zero, nRow and nCol will be one larger than the maximum
     * possible matrix index values.
     * @param nRow number of rows in this matrix.
     * @param nCol number of columns in this matrix.
     * @param matrix a 1D array that specifies a matrix in row major fashion
     */
    constructor(nRow: number, nCol: number, matrix: number[]);
    /**
     * Constructs a new GMatrix and copies the initial values
     * from the parameter matrix.
     * @param matrix the source of the initial values of the new GMatrix
     */
    constructor(matrix: GMatrix);
    /**
     * Sets the value of this matrix to the result of multiplying itself
     * with matrix m1 (this = this * m1).
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the result of multiplying
     * the two argument matrices together (this = m1 * m2).
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    mul(m1: GMatrix, m2?: GMatrix): void;
    /**
     * Computes the outer product of the two vectors; multiplies the
     * the first vector by the transpose of the second vector
     * and places the matrix result into this matrix. This matrix must
     * be as big or bigger than getSize(v1)xgetSize(v2).
     * @param v1 the first vector, treated as a row vector
     * @param v2 the second vector, treated as a column vector
     */
    mul_vector_vector(v1: GVector, v2: GVector): void;
    /**
     * Sets the value of this matrix to sum of itself and matrix m1.
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the matrix sum of matrices m1 and m2.
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    add(m1: GMatrix, m2?: GMatrix): void;
    /**
     * Sets the value of this matrix to the matrix difference of itself
     * and matrix m1 (this = this - m1).
     * @param m1 the other matrix
     *
     * Sets the value of this matrix to the matrix difference
     * of matrices m1 and m2 (this = m1 - m2).
     * @param m1 the first matrix
     * @param m2 the second matrix
     */
    sub(m1: GMatrix, m2?: GMatrix): void;
    /**
     * Negates the value of this matrix: this = -this.
     *
     * Sets the value of this matrix to the negation of the GMatrix parameter.
     * @param m1 The source matrix
     */
    negate(m1?: GMatrix): void;
    /**
     * Sets this GMatrix to the identity matrix.*/
    setIdentity(): void;
    /**
     * Sets all the values in this matrix to zero.*/
    setZero(): void;
    /**
     * Subtracts this matrix from the identity matrix and puts the values
     * back into this (this = I - this).*/
    identityMinus(): void;
    /**
     * Inverts this matrix in place.
     *
     * Inverts matrix m1 and places the new values into this matrix.  Matrix
     * m1 is not modified.
     * @param m1 the matrix to be inverted
     */
    invert(m1?: GMatrix): void;
    /**
     * Copies a sub-matrix derived from this matrix into the target matrix.
     * The upper left of the sub-matrix is located at (rowSource, colSource);
     * the lower right of the sub-matrix is located at
     * (lastRowSource,lastColSource).  The sub-matrix is copied into the
     * the target matrix starting at (rowDest, colDest).
     * @param rowSource the top-most row of the sub-matrix
     * @param colSource the left-most column of the sub-matrix
     * @param numRow the number of rows in the sub-matrix
     * @param numCol the number of columns in the sub-matrix
     * @param rowDest the top-most row of the position of the copied sub-matrix
     * within the target matrix
     * @param colDest the left-most column of the position of the copied sub-matrix
     * within the target matrix
     * @param target the matrix into which the sub-matrix will be copied
     */
    copySubMatrix(rowSource: number, colSource: number, numRow: number, numCol: number, rowDest: number, colDest: number, target: GMatrix): void;
    /**
     * Changes the size of this matrix dynamically.  If the size is increased
     * no data values will be lost.  If the size is decreased, only those data
     * values whose matrix positions were eliminated will be lost.
     * @param nRow number of desired rows in this matrix
     * @param nCol number of desired columns in this matrix
     */
    setSize(nRow: number, nCol: number): void;
    /**
     * Sets the value of this matrix to the values found in the array parameter.
     * The values are copied in one row at a time, in row major
     * fashion.  The array should be at least equal in length to
     * the number of matrix rows times the number of matrix columns
     * in this matrix.
     * @param matrix the row major source array
     */
    set_array(matrix: number[]): void;
    /**
     * Sets the value of this matrix to that of the Matrix3d provided.
     * @param m1 the source matrix
     */
    set_matrix3(m1: Matrix3d): void;
    /**
     * Sets the value of this matrix to that of the Matrix4d provided.
     * @param m1 the source matrix
     */
    set_matrix4(m1: Matrix4d): void;
    /**
     * Sets the value of this matrix to the values found in matrix m1.
     * @param m1 the source matrix
     */
    set(m1: GMatrix): void;
    /**
     * Returns the number of rows in this matrix.
     * @return number of rows in this matrix
     */
    getNumRow(): number;
    /**
     * Returns the number of colmuns in this matrix.
     * @return number of columns in this matrix
     */
    getNumCol(): number;
    /**
     * Retrieves the value at the specified row and column of this matrix.
     * @param row the row number to be retrieved (zero indexed)
     * @param column the column number to be retrieved (zero indexed)
     * @return the value at the indexed element
     */
    getElement(row: number, column: number): number;
    /**
     * Modifies the value at the specified row and column of this matrix.
     * @param row the row number to be modified (zero indexed)
     * @param column the column number to be modified (zero indexed)
     * @param value the new matrix element value
     */
    setElement(row: number, column: number, value: number): void;
    /**
     * Places the values of the specified row into the array parameter.
     * @param row the target row number
     * @param array the array into which the row values will be placed
     */
    getRow_array(row: number, array: number[]): void;
    /**
     * Places the values of the specified row into the vector parameter.
     * @param row the target row number
     * @param vector the vector into which the row values will be placed
     */
    getRow(row: number, vector: GVector): void;
    /**
     * Places the values of the specified column into the array parameter.
     * @param col the target column number
     * @param array the array into which the column values will be placed
     */
    getColumn_array(col: number, array: number[]): void;
    /**
     * Places the values of the specified column into the vector parameter.
     * @param col the target column number
     * @param vector the vector into which the column values will be placed
     */
    getColumn(col: number, vector: GVector): void;
    /**
     * Places the values in the upper 3X3 of this GMatrix into the matrix m1.
     * @param m1 The matrix that will hold the new values
     */
    get_matrix3(m1: Matrix3d): void;
    /**
     * Places the values in the upper 4X4 of this GMatrix into the matrix m1.
     * @param m1 The matrix that will hold the new values
     */
    get_matrix4(m1: Matrix4d): void;
    /**
     * Places the values in the this matrix into the matrix m1; m1
     * should be at least as large as this GMatrix.
     * @param m1 The matrix that will hold the new values
     */
    get(m1: GMatrix): void;
    /**
     * Copy the values from the array into the specified row of this
     * matrix.
     * @param row the row of this matrix into which the array values
     * will be copied.
     * @param array the source array
     */
    setRow_array(row: number, array: number[]): void;
    /**
     * Copy the values from the array into the specified row of this
     * matrix.
     * @param row the row of this matrix into which the vector values
     * will be copied.
     * @param vector the source vector
     */
    setRow(row: number, vector: GVector): void;
    /**
     * Copy the values from the array into the specified column of this
     * matrix.
     * @param row the column of this matrix into which the array values
     * will be copied.
     * @param array the source array
     */
    setColumn_array(col: number, array: number[]): void;
    /**
     * Copy the values from the array into the specified column of this
     * matrix.
     * @param row the column of this matrix into which the vector values
     * will be copied.
     * @param vector the source vector
     */
    setColumn(col: number, vector: GVector): void;
    /**
     * Multiplies the transpose of matrix m1 times the transpose of matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeBoth(m1: GMatrix, m2: GMatrix): void;
    /**
     * Multiplies matrix m1 times the transpose of matrix m2, and places the
     * result into this. */
    mulTransposeRight(m1: GMatrix, m2: GMatrix): void;
    /**
     * Multiplies the transpose of matrix m1 times the matrix m2, and places the
     * result into this.
     * @param m1 The matrix on the left hand side of the multiplication
     * @param m2 The matrix on the right hand side of the multiplication
     */
    mulTransposeLeft(m1: GMatrix, m2: GMatrix): void;
    /**
     * Transposes this matrix in place.
     *
     * Places the matrix values of the transpose of matrix m1 into this matrix. <p>
     * @param m1 the matrix to be transposed (but not modified)
     */
    transpose(m1?: GMatrix): void;
    /**
     * Returns a string that contains the values of this GMatrix.
     * @return the String representation
     */
    toString(): string;
    /**
     * Returns true if all of the data members of Matrix4d m1 are
     * equal to the corresponding data members in this Matrix4d.
     * @param m1 The matrix with which the comparison is made.
     * @return true or false
     */
    equals(m1: GMatrix): boolean;
    /**
     * Returns true if the L-infinite distance between this matrix and
     * matrix m1 is less than or equal to the epsilon parameter,
     * otherwise returns false. The L-infinite distance is equal to
     * MAX[i=0,1,2, . . .n ; j=0,1,2, . . .n ; abs(this.m(i,j) - m1.m(i,j)] .
     * @param m1 The matrix to be compared to this matrix
     * @param epsilon the threshold value
     */
    epsilonEquals(m1: GMatrix, epsilon: number): boolean;
    /**
     * Returns the trace of this matrix.
     * @return the trace of this matrix.
     */
    trace(): number;
    /**
     * Sets this matrix to a uniform scale matrix; all of the values are reset.
     * @param scale The new scale value
     */
    setScale(scale: number): void;
    setDiag(i: number, value: number): void;
    getDiag(i: number): number;
    dpythag(a: number, b: number): number;
    /**
     * Finds the singular value decomposition (SVD) of this matrix such that
     * this = U*W*transpose(V); and returns the rank of this matrix; the values
     * of U,W,V are all overwritten. Note that the matrix V is output as V,
     * and not transpose(V). If this matrix is mxn, then U is mxm, W is a
     * diagonal matrix that is mxn, and V is nxn. Using the notation W = diag(w),
     * then the inverse of this matrix is: inverse(this) = V*diag(1/w)*tranpose(U),
     * where diag(1/w) is the same matrix as W except that the reciprocal of each
     * of the diagonal components is used.
     * @param U The computed U matrix in the equation this = U*W*transpose(V)
     * @param W The computed W matrix in the equation this = U*W*transpose(V)
     * @param V The computed V matrix in the equation this = U*W*transpose(V)
     * @return The rank of this matrix.
     */
    SVD(u: GMatrix, w: GMatrix, v: GMatrix): number;
    swapRows(i: number, j: number): void;
    /**
     * LU Decomposition; this matrix must be a square matrix; the LU GMatrix
     * parameter must be the same size as this matrix. The matrix LU will be
     * overwritten as the combination of a lower diagonal and upper diagonal
     * matrix decompostion of this matrix; the diagonal elements of L (unity)
     * are not stored. The GVector parameter records the row permutation
     * effected by the partial pivoting, and is used as a parameter to the
     * GVector method LUDBackSolve to solve sets of linear equations. This
     * method returns +/- 1 depending on whether the number of row interchanges
     * was even or odd, respectively.
     * @param permutation The row permutation effected by the
     * partial pivoting
     * @return +-1 depending on whether the number of row interchanges
     * was even or odd respectively
     */
    LUD(LU: GMatrix, permutation: GVector): number;
}
