/*
   Copyright (C) 1997,1998,1999
   Kenji Hiranabe, Eiwa System Management, Inc.

   This program is free software.
   Implemented by Kenji Hiranabe(hiranabe@esm.co.jp),
   conforming to the Java(TM) 3D API specification by Sun Microsystems.

   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies and
   that both that copyright notice and this permission notice appear
   in supporting documentation. Kenji Hiranabe and Eiwa System Management,Inc.
   makes no representations about the suitability of this software for any
   purpose.  It is provided "AS IS" with NO WARRANTY.
*/
"use strict";
/**
 * A 4 element axis angle represented by double precision floating point
 * x,y,z,angle components. An axis angle is a rotation of angle (radians) about
 * the vector (x,y,z).
 * @version specification 1.1, implementation $Revision: 1.11 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export class AxisAngle4d {
    constructor(p, y, z, angle) {
        if (Array.isArray(p)) {
            this.set_array(p);
        }
        else if (typeof p === "object") {
            if (typeof y === "number")
                this.set_axis_angle(p, y);
            else
                this.set_axisAngle(p);
        }
        else if (typeof p === "number") {
            this.x = p;
            this.y = y;
            this.z = z;
            this.angle = angle;
        }
        else {
            this.x = 0.0;
            this.y = 0.0;
            this.z = 1.0;
            this.angle = 0.0;
        }
    }
    /**
     * Sets the value of this AxisAngle4d to the specified axis and
     * angle.
     * @param axis the axis
     * @param angle the angle
     * @since Java 3D 1.2
     */
    set_axis_angle(axis, angle) {
        this.x = axis.x;
        this.y = axis.y;
        this.z = axis.z;
        this.angle = angle;
    }
    /**
     * Sets the value of this axis angle to the specified x,y,z,angle.
     * @param x the x coordinate
     * @param y the y coordinate
     * @param z the z coordinate
     * @param angle the angle
     */
    set_xyz_angle(x, y, z, angle) {
        this.x = x;
        this.y = y;
        this.z = z;
        this.angle = angle;
    }
    /**
     * Sets the value of this axis angle from the 4 values specified in the array.
     * @param a the array of length 4 containing x,y,z,angle in order
     */
    set_array(a) {
        this.x = a[0];
        this.y = a[1];
        this.z = a[2];
        this.angle = a[3];
    }
    /**
     * Sets the value of this axis angle to the value of axis angle t1.
     * @param t1 the axis angle to be copied
     */
    set_axisAngle(a1) {
        this.x = a1.x;
        this.y = a1.y;
        this.z = a1.z;
        this.angle = a1.angle;
    }
    /**
     * Gets the value of this axis angle into the array a of
     * length four in x,y,z,angle order.
     * @param a the array of length four
     */
    get_array(a) {
        a[0] = this.x;
        a[1] = this.y;
        a[2] = this.z;
        a[3] = this.angle;
    }
    /**
     * Sets the value of this axis-angle to the rotational component of the
     * passed matrix.
     * @param m1 the matrix4d
     */
    set_matrix4(m1) {
        this.setFromMat(m1.m00, m1.m01, m1.m02, m1.m10, m1.m11, m1.m12, m1.m20, m1.m21, m1.m22);
    }
    /**
     * Sets the value of this axis-angle to the rotational component of the
     * passed matrix.
     * @param m1 the matrix3d
     */
    set_matrix3(m1) {
        this.setFromMat(m1.m00, m1.m01, m1.m02, m1.m10, m1.m11, m1.m12, m1.m20, m1.m21, m1.m22);
    }
    /**
     * Sets the value of this axis-angle to the rotational equivalent of the
     * passed quaternion.
     * @param q1 the Quat4d
     */
    set_quat(q1) {
        this.setFromQuat(q1.x, q1.y, q1.z, q1.w);
    }
    setFromMat(m00, m01, m02, m10, m11, m12, m20, m21, m22) {
        let cos = (m00 + m11 + m22 - 1.0) * 0.5;
        this.x = m21 - m12;
        this.y = m02 - m20;
        this.z = m10 - m01;
        let sin = 0.5 * Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
        this.angle = Math.atan2(sin, cos);
    }
    setFromQuat(x, y, z, w) {
        let sin_a2 = Math.sqrt(x * x + y * y + z * z);
        this.angle = 2.0 * Math.atan2(sin_a2, w);
        this.x = x;
        this.y = y;
        this.z = z;
    }
    /**
     * Returns a string that contains the values of this AxisAngle4d. The form is (x,y,z,angle).
     * @return the String representation
     */
    toString() {
        return "(" + this.x + ", " + this.y + ", " + this.z + ", " + this.angle + ")";
    }
    /**
     * Returns true if all of the data members of AxisAngle4d t1 are equal to the corresponding
     * data members in this
     * @param a1 the vector with which the comparison is made.
     */
    equals(a1) {
        return a1 && this.x === a1.x && this.y === a1.y && this.z === a1.z && this.angle === a1.angle;
    }
    /**
     * Returns true if the L-infinite distance between this axis-angle and axis-angle t1 is
     * less than or equal to the epsilon parameter, otherwise returns false. The L-infinite
     * distance is equal to MAX[abs(x1-x2), abs(y1-y2), abs(z1-z2), abs(angle1-angle2)].
     * @param a1 the axis-angle to be compared to this axis-angle
     * @param epsilon the threshold value
     */
    epsilonEquals(a1, epsilon) {
        return (Math.abs(a1.x - this.x) <= epsilon) && (Math.abs(a1.y - this.y) <= epsilon) && (Math.abs(a1.z - this.z) <= epsilon) && (Math.abs(a1.angle - this.angle) <= epsilon);
    }
}
//# sourceMappingURL=AxisAngle4d.js.map