import { Vector3d } from './Vector3d';
import { Matrix3d } from './Matrix3d';
import { Matrix4d } from './Matrix4d';
import { Quat4d } from './Quat4d';
/**
 * A 4 element axis angle represented by double precision floating point
 * x,y,z,angle components. An axis angle is a rotation of angle (radians) about
 * the vector (x,y,z).
 * @version specification 1.1, implementation $Revision: 1.11 $, $Date: 1999/10/05 07:03:50 $
 * @author Kenji hiranabe
 */
export declare class AxisAngle4d {
    /**
     * The x coordinate.
     */
    x: number;
    /**
     * The y coordinate.
     */
    y: number;
    /**
     * The z coordinate.*/
    z: number;
    /**
     * The angle.
     */
    angle: number;
    /**
     * Constructs and initializes an AxisAngle4d from the specified x, y, z,
     * and angle.
     * @param x the x coordinate
     * @param y the y coordinate
     * @param z the z coordinate
     * @param angle the angle.
     */
    constructor(x: number, y: number, z: number, angle: number);
    /**
     * Constructs and initializes an AxisAngle4d from the components contained
     * in the array.
     * @param a the array of length 4 containing x,y,z,angle in order
     */
    constructor(a: number[]);
    /**
     * Constructs and initializes a AxisAngle4d from the specified AxisAngle4d.
     * @param a1 the AxisAngle4d containing the initialization x y z angle data
     */
    constructor(a1: AxisAngle4d);
    /**
     * Constructs and initializes a AxisAngle4d to (0,0,1,0).*/
    constructor();
    /**
     * Constructs and initializes an AxisAngle4d from the specified axis
     * and angle.
     * @param axis the axis
     * @param angle the angle
     * @since Java 3D 1.2
     */
    constructor(axis: Vector3d, angle: number);
    /**
     * Sets the value of this AxisAngle4d to the specified axis and
     * angle.
     * @param axis the axis
     * @param angle the angle
     * @since Java 3D 1.2
     */
    set_axis_angle(axis: Vector3d, angle: number): void;
    /**
     * Sets the value of this axis angle to the specified x,y,z,angle.
     * @param x the x coordinate
     * @param y the y coordinate
     * @param z the z coordinate
     * @param angle the angle
     */
    set_xyz_angle(x: number, y: number, z: number, angle: number): void;
    /**
     * Sets the value of this axis angle from the 4 values specified in the array.
     * @param a the array of length 4 containing x,y,z,angle in order
     */
    set_array(a: number[]): void;
    /**
     * Sets the value of this axis angle to the value of axis angle t1.
     * @param t1 the axis angle to be copied
     */
    set_axisAngle(a1: AxisAngle4d): void;
    /**
     * Gets the value of this axis angle into the array a of
     * length four in x,y,z,angle order.
     * @param a the array of length four
     */
    get_array(a: number[]): void;
    /**
     * Sets the value of this axis-angle to the rotational component of the
     * passed matrix.
     * @param m1 the matrix4d
     */
    set_matrix4(m1: Matrix4d): void;
    /**
     * Sets the value of this axis-angle to the rotational component of the
     * passed matrix.
     * @param m1 the matrix3d
     */
    set_matrix3(m1: Matrix3d): void;
    /**
     * Sets the value of this axis-angle to the rotational equivalent of the
     * passed quaternion.
     * @param q1 the Quat4d
     */
    set_quat(q1: Quat4d): void;
    private setFromMat(m00, m01, m02, m10, m11, m12, m20, m21, m22);
    private setFromQuat(x, y, z, w);
    /**
     * Returns a string that contains the values of this AxisAngle4d. The form is (x,y,z,angle).
     * @return the String representation
     */
    toString(): string;
    /**
     * Returns true if all of the data members of AxisAngle4d t1 are equal to the corresponding
     * data members in this
     * @param a1 the vector with which the comparison is made.
     */
    equals(a1: AxisAngle4d): boolean;
    /**
     * Returns true if the L-infinite distance between this axis-angle and axis-angle t1 is
     * less than or equal to the epsilon parameter, otherwise returns false. The L-infinite
     * distance is equal to MAX[abs(x1-x2), abs(y1-y2), abs(z1-z2), abs(angle1-angle2)].
     * @param a1 the axis-angle to be compared to this axis-angle
     * @param epsilon the threshold value
     */
    epsilonEquals(a1: AxisAngle4d, epsilon: number): boolean;
}
