import { KPoint2d } from 'kiteclib';
import { MathUtil } from 'kiteclib';
import { KArea } from 'kiteclib';
//module nethome.geom
/**
 * @author kawae**/
export class WorkSheet {
    /**
     * Constructor for Shikichi.**/
    constructor() {
        /**
         * グリッド分割数**/
        this.mGridDiv = 1;
        this.mArea = new KArea();
        this.mAreaPolygon = [];
        this.mModule = 0;
        this.mGridCenterX = 0;
        this.mGridCenterY = 0;
        this.mExtraGridX = [];
        this.mExtraGridY = [];
    }
    /**
     * コピーインスタンスの作成
     * @return
     * @author kawae
     * @since 2004/10/02
    **/
    getCopy() {
        var ws = new WorkSheet();
        ws.setName(this.getName());
        ws.setArea(this.mArea.minX, this.mArea.minY, this.mArea.maxX, this.mArea.maxY);
        ws.mGridDiv = this.mGridDiv;
        ws.mModule = this.mModule;
        ws.mGridCenterX = this.mGridCenterX;
        ws.mGridCenterY = this.mGridCenterY;
        return ws;
    }
    /**
     * ワークシート名の設定
     * @param s
     * @author kawae
     * @since 2004/10/02
    **/
    setName(s) {
        this.mName = s;
    }
    /**
     * ワークシート名の取得
     * @return
     * @author kawae
     * @since 2004/10/02
    **/
    getName() {
        return this.mName;
    }
    /**
     * 領域の設定
     * @param minx
     * @param miny
     * @param maxx
     * @param maxy
     * @author kawae
     * @since 2004/10/02
    **/
    setArea(minx, miny, maxx, maxy) {
        this.mArea.maximize();
        this.mArea.updateMinMax(minx, miny);
        this.mArea.updateMinMax(maxx, maxy);
        this.mAreaPolygon.length = 0;
        this.mAreaPolygon.push(new KPoint2d(minx, miny));
        this.mAreaPolygon.push(new KPoint2d(maxx, miny));
        this.mAreaPolygon.push(new KPoint2d(maxx, maxy));
        this.mAreaPolygon.push(new KPoint2d(minx, maxy));
    }
    /**
     * 領域の取得
     * @return
     * @author kawae
     * @since 2004/10/02
    **/
    getArea() {
        return this.mArea;
    }
    updateArea() {
        this.mArea.maximize();
        for (var i = 0; i < this.mAreaPolygon.length; i++) {
            var p = this.mAreaPolygon[i];
            this.mArea.updateMinMax(p.x, p.y);
        }
    }
    getAreaPolygon() {
        return this.mAreaPolygon.slice();
    }
    /**
     * モジュールの取得
     * @return
    **/
    getModule() {
        return this.mModule;
    }
    /**
     * モジュールの設定
     * @param f
    **/
    setModule(f) {
        if (f < 0)
            throw new Error("negative module");
        this.mModule = f;
    }
    /**
     * グリッドのベースモジュールに対する分割数の設定
     * @param i
    **/
    setGridDivide(i) {
        if (i <= 0)
            throw new Error("negative grid division");
        this.mGridDiv = i;
    }
    getGridDivide() {
        return this.mGridDiv;
    }
    getCurrentGrid() {
        return this.mModule / this.mGridDiv;
    }
    setGridCenter(x, y) {
        this.mGridCenterX = x;
        this.mGridCenterY = y;
    }
    getGridCenterX() {
        return this.mGridCenterX;
    }
    getGridCenterY() {
        return this.mGridCenterY;
    }
    /**
     * 最も近いXグリッドを探す
     * @param x
     * @param extra 追加グリッドも含めて探すかのフラグ。
     * @return
    **/
    getNearGridX(x, extra) {
        return this.getNearGrid(x, extra, this.getCurrentGrid(), this.getGridCenterX(), this.mExtraGridX);
    }
    /**
     * 最も近いYグリッドを探す
     * @param y
     * @param extra 追加グリッドも含めて探すかのフラグ。
     * @return
    **/
    getNearGridY(y, extra) {
        return this.getNearGrid(y, extra, this.getCurrentGrid(), this.getGridCenterY(), this.mExtraGridY);
    }
    getNearGrid(v, extra, currentGrid, gridCenter, extraGrid) {
        if (currentGrid == 0)
            return v;
        let near = Number.MAX_VALUE;
        let step = v <= gridCenter ? -1 : 1;
        let c = Math.floor(((v - gridCenter) / currentGrid) * step);
        while (true) {
            let len = Math.abs(v - (gridCenter + currentGrid * c * step));
            if (near > len)
                near = len;
            else
                break;
            c++;
        }
        let gridcandidate = step * (c - 1) * currentGrid + gridCenter;
        if (extra && extraGrid != null) {
            for (let i = 0; i < extraGrid.length; i++) {
                let f = extraGrid[i];
                let len = Math.abs(v - f);
                if (len < near) {
                    gridcandidate = f;
                    near = len;
                }
            }
        }
        return gridcandidate;
    }
    /**
     * デフォルトのスナップX
     * @param x
     * @return double
    **/
    getDefNearGridX(x) {
        var near = Number.MAX_VALUE;
        var div = this.getModule() / WorkSheet.DEF_NEAR_GRID_DIVIDE;
        var c = ((x - x % div) / div) | 0;
        var res = 0;
        for (var i = c - 1; i <= c + 1; i++) {
            if (Math.abs(div * i - x) < near) {
                near = Math.abs(div * i - x);
                res = i;
            }
        }
        return res * div;
    }
    getDefNearGridY(y) {
        var near = Number.MAX_VALUE;
        var div = this.getModule() / WorkSheet.DEF_NEAR_GRID_DIVIDE;
        var c = ((y - y % div) / div) | 0;
        var res = 0;
        for (var i = c - 1; i <= c + 1; i++) {
            if (Math.abs(div * i - y) < near) {
                near = Math.abs(div * i - y);
                res = i;
            }
        }
        return res * div;
    }
    isOnGridX(x) {
        var xx = this.getNearGridX(x, true);
        if (Math.abs(xx - x) < MathUtil.EPS)
            return true;
        return false;
    }
    isOnGridY(y) {
        var yy = this.getNearGridY(y, true);
        if (Math.abs(yy - y) < MathUtil.EPS)
            return true;
        return false;
    }
    isOnBaseGridX(x) {
        var c = Math.abs(x % this.mModule);
        if (c < MathUtil.EPS || Math.abs(c - this.mModule) < MathUtil.EPS) {
            return true;
        }
        return false;
    }
    isOnBaseGridY(y) {
        var c = Math.abs(y % this.mModule);
        if (c < MathUtil.EPS || Math.abs(c - this.mModule) < MathUtil.EPS) {
            return true;
        }
        return false;
    }
    getNearGridX2(x, dir) {
        if (this.getCurrentGrid() == 0)
            return x;
        var ss = Math.floor(((x - this.getGridCenterX()) / this.getCurrentGrid()));
        var candx = this.getGridCenterX() + ss * this.getCurrentGrid();
        var candx1;
        var candx2;
        if (candx < x) {
            candx1 = candx;
            candx2 = this.getGridCenterX() + (ss + 1) * this.getCurrentGrid();
        }
        else {
            candx1 = this.getGridCenterX() + (ss - 1) * this.getCurrentGrid();
            candx2 = candx;
        }
        if (Math.abs(candx1 - x) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE) {
            return candx1;
        }
        if (Math.abs(candx2 - x) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE) {
            return candx2;
        }
        if (this.mExtraGridX != null) {
            for (var i = 0; i < this.mExtraGridX.length; i++) {
                var f = this.mExtraGridX[i];
                if (Math.abs(f - x) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE)
                    return f;
                if (candx1 < f && x > f)
                    candx1 = f;
                if (x < f && candx2 > f)
                    candx2 = f;
            }
        }
        if (dir)
            return candx2;
        return candx1;
    }
    getNearGridY2(y, dir) {
        if (this.getCurrentGrid() == 0)
            return y;
        var ss = Math.floor(((y - this.getGridCenterY()) / this.getCurrentGrid()));
        var candy = this.getGridCenterY() + ss * this.getCurrentGrid();
        var candy1;
        var candy2;
        if (candy < y) {
            candy1 = candy;
            candy2 = this.getGridCenterY() + (ss + 1) * this.getCurrentGrid();
        }
        else {
            candy1 = this.getGridCenterY() + (ss - 1) * this.getCurrentGrid();
            candy2 = candy;
        }
        if (Math.abs(candy1 - y) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE) {
            return candy1;
        }
        if (Math.abs(candy2 - y) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE) {
            return candy2;
        }
        if (this.mExtraGridY != null) {
            for (var i = 0; i < this.mExtraGridY.length; i++) {
                var f = this.mExtraGridY[i];
                if (Math.abs(f - y) < this.getCurrentGrid() / WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE)
                    return f;
                if (candy1 < f && y > f)
                    candy1 = f;
                if (y < f && candy2 > f)
                    candy2 = f;
            }
        }
        if (dir) {
            return candy2;
        }
        return candy1;
    }
    /**
     * 追加グリッドを抜いたグリッドの取得
     * @param x
     * @param bx
     * @param module
     * @return
    **/
    getNearNotExtraGrid(x, bx, module) {
        var near = Number.MAX_VALUE;
        var step = x <= bx ? -1 : 1;
        var c = Math.floor(((x - bx) / module) * step);
        while (true) {
            var len = Math.abs(x - (bx + module * c * step));
            if (near > len)
                near = len;
            else
                break;
            c++;
        }
        return step * (c - 1);
    }
    getExtraGridX() {
        return this.mExtraGridX;
    }
    getExtraGridY() {
        return this.mExtraGridY;
    }
    addExtraGridX(x) {
        if (Math.abs(this.getNearGridX(x, true) - x) < MathUtil.EPS)
            return false;
        if (this.mExtraGridX == null)
            this.mExtraGridX = [];
        for (var i = 0; i < this.mExtraGridX.length; i++) {
            var f = this.mExtraGridX[i];
            if (Math.abs(f - x) < MathUtil.EPS)
                return false;
        }
        this.mExtraGridX.push(x);
        return true;
    }
    addExtraGridY(y) {
        if (Math.abs(this.getNearGridY(y, true) - y) < MathUtil.EPS)
            return false;
        if (this.mExtraGridY == null)
            this.mExtraGridY = [];
        for (var i = 0; i < this.mExtraGridY.length; i++) {
            var f = this.mExtraGridY[i];
            if (Math.abs(f - y) < MathUtil.EPS)
                return false;
        }
        this.mExtraGridY.push(y);
        return true;
    }
}
WorkSheet.NEAR_GRID2_THRESHOLD_DIVIDE = 30;
WorkSheet.DEF_NEAR_GRID_DIVIDE = 50;
//# sourceMappingURL=WorkSheet.js.map