var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var ObjectNode_1 = require('./ObjectNode');
//module jp.kitec.lib.util.tree
/**
 * ツリーなどの階層構造における複数のノードを保持する親ノード。<BR>
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
**/
var ObjectFolder = (function (_super) {
    __extends(ObjectFolder, _super);
    /**
     * 指定された名前と親を持つ新たな親ノードを構築する。
     * @param name		このノードの名前
     * @param parent	このノードの親
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    function ObjectFolder(name) {
        if (name === void 0) { name = null; }
        _super.call(this, name, null);
        this.mChildren = [];
    }
    /**
     * 階層構造内の全てのObjectFolderのノードを削除する**/
    ObjectFolder.prototype.removeAllElements = function () {
        this.mChildren.forEach(function (o) {
            if (o instanceof ObjectFolder) {
                o.removeAllElements();
            }
        });
        this.mChildren.length = 0;
    };
    /**
     * 指定されたパスに対応する親ノード、およびノードからなる
     * 階層構造を作成し、このノード下へ追加する。
     * @param s	パス
     * @return 最後に追加されたノード。
     * ノードが追加されたなかった場合は、このノード。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.addFolderFromPath = function (path) {
        var f = this;
        var st = path.split(/[/\\\\]/);
        for (var i = 0; i < st.length; i++) {
            var s = st[i];
            if (s.length == 0)
                continue;
            var cur = null;
            for (var j = 0; j < f.getChildren().length; j++) {
                var o = f.getChildren()[j];
                if (o instanceof ObjectFolder) {
                    var fol = o;
                    if (fol.getName().toLowerCase() === s.toLowerCase()) {
                        cur = fol;
                        break;
                    }
                }
            }
            if (cur == null) {
                var of = new ObjectFolder(s);
                f.addChild(of);
                f = cur = of;
            }
            else
                f = cur;
        }
        return f;
    };
    /**
     * このノードが保持する階層構造の直下の階層を対象として、指定された
     * 名前を持つノード、または親ノードを検索し、該当するノードが存在
     * する場合、そのノードを返す。
     * @param name				ノードの名前
     * @param includeFolder	親ノードを検索対象とする場合、true。
     * false を指定すると、名前が一致しても
     * 対象からは除外される。
     * @return ノード。該当するノードが存在しない場合は、null。
     * @see #getNode(String,boolean)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.getNode = function (name, includeFolder) {
        if (includeFolder === void 0) { includeFolder = true; }
        for (var i = 0; i < this.mChildren.length; i++) {
            var o = (this.mChildren[i]);
            if (!includeFolder && o instanceof ObjectFolder)
                continue;
            var on = o;
            if (on.getName().toLowerCase() == name.toLowerCase())
                return on;
        }
        return null;
    };
    /**
     * このノードが保持する階層構造の全ノードを、１つの階層ごとに
     * 親ノード、ノードの順に並び替える。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.sortFolder = function () {
        var fold = [];
        var node = [];
        for (var i = 0; i < this.mChildren.length; i++) {
            var o = this.mChildren[i];
            if (o instanceof ObjectFolder)
                fold.push((o));
            else
                node.push(o);
        }
        this.mChildren.length = 0;
        for (var i = 0; i < fold.length; i++) {
            fold[i].sortFolder();
            this.mChildren.push(fold[i]);
        }
        for (var i__2 = 0; i__2 < node.length; i__2++)
            this.mChildren.push(node[i__2]);
    };
    /**
     * このノードが保持する階層構造のノード群（子ノード）を返す。
     * @return ノード群
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.getChildren = function () {
        return this.mChildren;
    };
    /**
     * このノードの階層構造に、指定されたノードを追加する。
     * @param n			新たなに追加するノード
     * @param sameregist	このノードと同じ名前を持つノードが存在する場合、
     * 階層構造に追加するか否かを指定する。
     * 追加する場合は、true。追加しない場合は、false。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.addChild = function (n, sameregist) {
        if (sameregist === void 0) { sameregist = false; }
        if (this.mChildren.indexOf(n) >= 0)
            return;
        if (!sameregist) {
            for (var i = 0; i < this.mChildren.length; i++) {
                var node = this.mChildren[i];
                if (n.getName() === node.getName())
                    return;
            }
        }
        n.setParent(this);
        this.mChildren.push(n);
    };
    /**
     * このノードの階層構造の中から、指定された名前と同じ名前を持つ
     * すべてのノードを返す。大文字小文字は区別しない。
     * @param name ノードの名前
     * @return 指定された名前と同じ名前を持つノード群
    **/
    ObjectFolder.prototype.getNamedElement = function (s) {
        s = s.toLowerCase();
        var res = [];
        for (var i = 0; i < this.mChildren.length; i++) {
            var on = this.mChildren[i];
            if (on.getName().toLowerCase() == s) {
                res.push(on);
            }
        }
        return res;
    };
    /**
     * このノードの階層構造のすべての階層を対象として、指定された名前と
     * 同じ名前を持つノードを検索し、そのノードを返す。
     * 同じ名前を持つノードが複数存在する場合、次の内容でノードが優先度
     * が決まる。
     * (1)所属する階層が異なる場合、階層が上のノード
     * (2)同じ階層の場合、最初に見つかった（先に追加された）ノード
     * @param name	ノードの名前
     * @return 指定された名前と同じ名前を持つノード
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    ObjectFolder.prototype.findNode = function (name) {
        name = name.toLowerCase();
        var queue = [];
        queue.push(this);
        while (queue.length > 0) {
            var of = queue.shift();
            for (var i = 0; i < of.mChildren.length; i++) {
                var on = of.mChildren[i];
                if (on instanceof ObjectFolder) {
                    queue.push(on);
                }
                else {
                    if (on.getName().toLowerCase() == name) {
                        return on;
                    }
                }
            }
        }
        return null;
    };
    ObjectFolder.prototype.findNodeFirst = function (name) {
        for (var i = 0; i < this.mChildren.length; i++) {
            var on = this.mChildren[i];
            if (on instanceof ObjectFolder) {
                var tmp = on.findNode(name);
                if (tmp != null)
                    return tmp;
            }
            else {
                if (on.getName().toLowerCase() == name.toLowerCase()) {
                    return on;
                }
            }
        }
        return null;
    };
    /**
     * ObjectFolderが持つ階層構造のすべてのノードを v に追加する。
     * @param v	ノードを設定先となるコレクション
     * @since   2010/04/12
     * @author  matsuzaki
    **/
    ObjectFolder.prototype.getAllChildNode = function (v) {
        for (var i = 0; i < this.getChildren().length; i++) {
            var o = this.mChildren[i];
            if (o instanceof ObjectFolder) {
                o.getAllChildNode(v);
            }
            else if (o instanceof ObjectNode_1.ObjectNode) {
                v.push(o);
            }
        }
    };
    ObjectFolder.prototype.addNode = function (name, val) {
        if (typeof val == "boolean")
            this.addChild(new ObjectNode_1.ObjectNode(name, val ? "1" : "0"), true);
        else if (typeof val == "string")
            this.addChild(new ObjectNode_1.ObjectNode(name, val), true);
        else if (typeof val == "number")
            this.addChild(new ObjectNode_1.ObjectNode(name, val.toString()), true);
        else
            throw new Error();
    };
    /**
     * ObjectFolderにString値を設定
     * valがnullの場合、追加しない
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    ObjectFolder.addNodeString = function (of, name, val) {
        if (val == null)
            return;
        of.addChild(new ObjectNode_1.ObjectNode(name, val), true);
    };
    /**
     * ObjectFolderにint値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    ObjectFolder.addNodeInt = function (of, name, val) {
        of.addChild(new ObjectNode_1.ObjectNode(name, val.toString()), true);
    };
    /**
     * ObjectFolderにfloat値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    ObjectFolder.addNodeFloat = function (of, name, val) {
        of.addChild(new ObjectNode_1.ObjectNode(name, val.toString()), true);
    };
    /**
     * ObjectFolderにdouble値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    ObjectFolder.addNodeDouble = function (of, name, val) {
        of.addChild(new ObjectNode_1.ObjectNode(name, val.toString()), true);
    };
    /**
     * ObjectFolderにboolean値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    ObjectFolder.addNodeBool = function (of, name, val) {
        of.addChild(new ObjectNode_1.ObjectNode(name, val ? "1" : "0"), true);
    };
    return ObjectFolder;
})(ObjectNode_1.ObjectNode);
exports.ObjectFolder = ObjectFolder;
//# sourceMappingURL=ObjectFolder.js.map