var common_1 = require('../common');
var Item_1 = require('./Item');
//module jp.kitec.lib.util
/**
 * 要素へのアクセスが、昇順／降順の双方向に可能な循環リスト。<BR>
 * また、すべての要素は、輪となって繋がっている。<BR>
 * Itemを直接操作する場合、整合性チェックのためO(n)のチェックが入るので注意が必要。<BR>
 * 比較する時にオブジェクトの同一性ではなくリファレンスの同一性を使用する。<BR>
 * visibility public<BR>
 * package    jp.kitec.lib.util<BR>
 * name       RefList<BR>
 * implements Serializable<BR>
 * @since 　2002/12/20
 * @author　fujita
 * @version 2002/12/20
 * TODO implement java.util.List<E>
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
**/
var RefList = (function () {
    /**
     * アイテム数 0 の、空の RefList を構築する。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    function RefList() {
        this.head = null;
        this.regCount = 0;
    }
    ///**
    // * デバッグ用。
    // * @since   2002/12/20
    // * @author  Kawae
    // * @version 2002/12/20
    //**/
    //public constructor(s : string) {
    //    this.head = null;
    //    this.regCount = 0;
    //}
    /**
     * 指定されたアイテムを、この RefList の先頭を指す要素として設定する。
     * 指定されたアイテムが、この RefList の要素に含まれない場合、
     * 先頭へは設定されない。
     * @param it 先頭の要素として設定するアイテム
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.setHead = function (it) {
        if (!this.containsItem(it))
            return;
        this.head = it;
    };
    /**
     * この RefList の先頭の要素を返す。
     * @return 先頭の要素
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.firstElement = function () {
        if (this.regCount != 0)
            return this.head.getObject();
        return null;
    };
    /**
     * この RefList の最後の要素を返す。
     * @return 最後の要素
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.lastElement = function () {
        if (this.regCount != 0)
            return this.head.prevItem.getObject();
        return null;
    };
    /**
     * 現在の先頭の要素を基点として、
     * 指定されたインデックス位置の要素を返す。
     * @param c インデックス
     * @return 指定されたインデックス位置にある要素
     * @see #itemAt(int)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.elementAt = function (c) {
        return this.itemAt(c).getObject();
    };
    /**
     * 現在の先頭の要素を基点として、
     * 指定されたインデックス位置のアイテムを返す。
     * @param c インデックス
     * @return 指定されたインデックス位置にあるアイテム
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.itemAt = function (c) {
        var it = this.head;
        if (it == null)
            throw new common_1.error.lang.IndexOutOfBoundsException();
        if (c >= 0) {
            for (var i = 0; i < c; i++)
                it = it.nextItem;
        }
        else {
            for (var i = 0; i > c; i--)
                it = it.prevItem;
        }
        return it;
    };
    /**
     * 指定されたデータを持つ要素を返す。
     * @param o 要素
     * @return 指定された要素を持つアイテム
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.itemOf = function (o) {
        var it = this.head;
        for (var i = 0; i < this.regCount; i++) {
            if (it.getObject() === o) {
                return it;
            }
            it = it.nextItem;
        }
        return null;
    };
    /**
     * 現在の RefList の要素数を返す。
     * @return  RefList の要素数
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.size = function () {
        return this.regCount;
    };
    /**
     * 指定された要素を RefList 最後に追加し、サイズを 1 つ増やす。
     * @param o 新たに追加する要素
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.addElement = function (o) {
        if (o == null)
            return false;
        if (this.head == null)
            this.insertElementItemAt(o, this.head);
        else
            this.insertElementItemAt(o, this.head.prevItem);
        return true;
    };
    /**
     * この RefList が保持するすべての要素を削除する。
     * 削除後、 RefList のサイズは 0 となる。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.removeAllElements = function () {
        var c = this.regCount;
        for (var i = 0; i < c; i++)
            this.removeItemAt(this.head);
    };
    /**
     * 指定された要素を RefList から検索し、最初に見つかった要素を削除する。
     * @param o	削除する要素
     * @return	該当する要素が見つからなければ、null。
     * 見つかれば削除された要素の前のアイテム
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.removeElement = function (o) {
        var delitem = null;
        var it = this.head;
        for (var i = 0; i < this.regCount; i++) {
            if (it.getObject() === o) {
                delitem = it;
                break;
            }
            it = it.nextItem;
        }
        if (delitem != null)
            return this.removeItemAt(delitem);
        return null;
    };
    /**
     * 指定されたアイテムを RefList から検索し、最初に見つかったアイテムを削除する。
     * @param ref		削除するアイテム
     * @return	該当するアイテムが見つからなければnull
     * 見つかれば削除されたアイテムの前のアイテム
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.removeItemAt = function (ref) {
        var now;
        if (ref == null || this.head == null || ref.nextItem == null)
            return null;
        if (this.regCount > 1) {
            if (ref == this.head)
                this.head = ref.nextItem;
            ref.prevItem.nextItem = ref.nextItem;
            now = ref.nextItem.prevItem = ref.prevItem;
            ref.nextItem = ref.prevItem = null;
            ref = null;
        }
        else {
            now = this.head = null;
        }
        this.regCount--;
        return now;
    };
    /**
     * 指定されたオブジェクトが要素かどうかを判定する。
     * @param o	オブジェクト
     * @return 指定されたオブジェクトが equals メソッドによって
     * RefList 内の要素と同じと確認された場合にだけ true、
     * そうでない場合は false
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.contains = function (o) {
        var it = this.head;
        for (var i = 0; i < this.regCount; i++) {
            if (it.getObject() === o)
                return true;
            it = it.nextItem;
        }
        return false;
    };
    /**
     * 指定されたアイテムが要素かどうかを判定する。
     * @param ptr	アイテム
     * @return 指定されたアイテムが equals メソッドによって
     * RefList 内の要素と同じと確認された場合にだけ true、
     * そうでない場合は false
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.containsItem = function (ptr) {
        var it = this.head;
        for (var i = 0; i < this.regCount; i++) {
            if (ptr === it)
                return true;
            it = it.nextItem;
        }
        return false;
    };
    /**
     * 指定されたオブジェクトを新たな要素として
     * RefList の指定された位置(指定されたindexの直後)へ挿入する。
     * @param ptr		挿入される要素
     * @param index	新しい要素を挿入する位置
     * @return	要素が追加された場合、追加されたアイテム。
     * インデックスが不正な場合など、
     * 追加されなかった場合は、null
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.insertElementAt = function (data, index) {
        if (data == null)
            return null;
        if (index < 0 || index >= this.regCount)
            return null;
        var it = this.head;
        for (var i = 0; i < this.regCount; i++) {
            if (i == index)
                break;
            it = it.nextItem;
        }
        return this.insertElementItemAt(data, it);
    };
    /**
     * 指定されたオブジェクトを新たな要素として
     * RefList の指定されたアイテムの直後に挿入する。
     * TODO:注意！！！　insertの順番が間違っている可能性あり。
     * @param data	挿入される要素
     * @param ptr		新しい要素を挿入する直前のアイテム
     * @return	要素が追加された場合、追加されたアイテム。
     * インデックスが不正な場合など、
     * 追加されなかった場合は、null
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.insertElementItemAt = function (data, ptr) {
        if ((ptr == null && this.head != null) || data == null)
            return null;
        if (!this.containsItem(ptr) && this.head != null)
            return null;
        var newItem = new Item_1.Item(data);
        if (ptr == null) {
            newItem.nextItem = newItem.prevItem = newItem;
            this.head = newItem;
        }
        else {
            newItem.prevItem = ptr;
            newItem.nextItem = ptr.nextItem;
            ptr.nextItem.prevItem = newItem;
            ptr.nextItem = newItem;
        }
        this.regCount++;
        return newItem;
    };
    /**
     * この RefList の要素を共有する新たな RefList を構築し、返す。
     * @return	要素を共有する新たな RefLsit
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.getCopy = function () {
        var tmp = new RefList();
        for (var i = 0; i < this.size(); i++)
            tmp.addElement(this.elementAt(i));
        return tmp;
    };
    /**
     * 指定された 2 つのアイテムが示す要素を入れ替える。
     * @param i1 要素入れ替えるアイテム 1
     * @param i2 要素入れ替えるアイテム 2
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.swap = function (i1, i2) {
        var o = (i1.getObject());
        i1.setObject((i2.getObject()));
        i2.setObject(o);
    };
    /**
     * it1 ～ it2 の範囲に含まれるすべての要素をこの RefList から削除する。
     * @param i1	削除する最初の要素
     * @param i2	削除する最後の要素
     * @param dir	削除を行う方向。true の場合、i1 から順（後方）に i2 まで
     * の要素を削除する。false の場合、i1 から逆順（前方）に要素
     * を削除する。
     * @return 指定された範囲の要素が削除された場合、true。指定された要素が
     * RefList に含まれない場合など削除がされなかった場合、false
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.deleteAtoB = function (it1, it2, dir) {
        var start = null;
        var end = null;
        var it = this.head;
        if (it1 == it2 && it1 != null) {
            this.removeItemAt(it1);
            return true;
        }
        for (var i = 0; i < this.regCount; i++) {
            if (it == it1)
                start = it;
            else if (it == it2)
                end = it;
            if (start != null && end != null)
                break;
            it = it.nextItem;
        }
        if (start == null || end == null)
            return false;
        var count = this.regCount;
        it = start;
        for (var i = 0; i < count; i++) {
            var itnext = dir ? it.nextItem : it.prevItem;
            this.removeItemAt(it);
            if (it == end)
                break;
            it = itnext;
        }
        return true;
    };
    /**
     * この RefList が保持する要素の順番を反転する。
     * 例）	処理前：[A][B][C][D][E]
     * 処理後：[A][E][D][C][B]
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    RefList.prototype.flip = function () {
        var n;
        var ncur;
        var nend;
        ncur = this.head;
        nend = this.head.nextItem;
        do {
            n = this.head.prevItem;
            this.removeItemAt(n);
            this.insertElementItemAt(n.getObject(), ncur);
            ncur = ncur.nextItem;
        } while ((n.getObject()) != (nend.getObject()));
    };
    //hashCode(): number {
    //    let hashCode = 1;
    //    var it: Item<Object> = this.head;
    //    for (var i: number = 0; i < this.regCount; i++) {
    //        let obj = (it.getObject()) as Object;
    //        hashCode = 31 * hashCode + (obj == null ? 0 : obj.hashCode());
    //        it = it.nextItem;
    //    }
    //    return hashCode;
    //}
    RefList.prototype.equals = function (o) {
        if (o == this)
            return true;
        if (!(o instanceof RefList))
            return false;
        var rl2 = o;
        if (this.regCount != rl2.regCount)
            return false;
        var it1 = this.head;
        var it2 = rl2.head;
        for (var i = 0; i < this.regCount; i++) {
            var o1 = it1.getObject();
            var o2 = it2.getObject();
            if (!(o1 === null ? o2 === null : o1 === o2))
                return false;
            it1 = it1.nextItem;
            it2 = it2.nextItem;
        }
        return true;
    };
    //iterator(): Iterator<T> {
    //    return new RefList$anon0();
    //}
    /**
     * {@inheritDoc}**/
    RefList.prototype.get = function (index) {
        return this.elementAt(index);
    };
    /**
     * {@inheritDoc}**/
    RefList.prototype.set = function (index, element) {
        var item = this.itemAt(index);
        var org = item.getObject();
        item.setObject(element);
        return org;
    };
    /**
     * {@inheritDoc}**/
    RefList.prototype.add = function (index, element) {
        this.insertElementAt(element, index);
    };
    /**
     * {@inheritDoc}**/
    RefList.prototype.remove = function (index) {
        var removeItem = this.itemAt(index);
        this.removeItemAt(removeItem);
        return removeItem.getObject();
    };
    return RefList;
})();
exports.RefList = RefList;
//# sourceMappingURL=RefList.js.map