//module jp.kitec.lib.util
/**
 * 幾何計算クラス**/
var MathUtil;
(function (MathUtil) {
    MathUtil.EPS = 1.0e-3;
    MathUtil.D2R = (Math.PI / 180);
    /**
     * 小数点s桁以降を切り捨てる。
     * @param f
     * @param s
     * @return
    **/
    function kirisute(f, s) {
        var kei = Math.pow(10, s - 1);
        return (Math.floor(f * kei) / kei);
    }
    MathUtil.kirisute = kirisute;
    /**
     * 外積(のZ値)を求める**/
    function crossProduct(p0x, p0y, p1x, p1y, p2x, p2y) {
        var dx1 = p1x - p0x;
        var dy1 = p1y - p0y;
        var dx2 = p2x - p0x;
        var dy2 = p2y - p0y;
        return dx1 * dy2 - dx2 * dy1;
    }
    MathUtil.crossProduct = crossProduct;
    /**
     * ２つの角度の差(360度内で）
     * getAnglGapRad()とは挙動が違うので注意**/
    function getAnglGapDeg(a, b) {
        b -= a;
        return MathUtil.getAngl360(b);
    }
    MathUtil.getAnglGapDeg = getAnglGapDeg;
    /**
     * ２つの角度の差(2PI度内で）
     * getAnglGapDeg()とは挙動が違うので注意**/
    function getAnglGapRad(a, b) {
        a = MathUtil.getAngl2PI(a);
        b = MathUtil.getAngl2PI(b);
        return Math.abs(a - b);
    }
    MathUtil.getAnglGapRad = getAnglGapRad;
    /**
     * 360度内で開始角と終了角の間にあるか(ラジアン)**/
    function isInnerAngl2PI(s, e, a) {
        return MathUtil.isInnerAngl360(MathUtil.getAngl360(s), MathUtil.getAngl360(e), MathUtil.getAngl360(a));
    }
    MathUtil.isInnerAngl2PI = isInnerAngl2PI;
    /**
     * 360度内で開始角と終了角の間にあるか(度)**/
    function isInnerAngl360(s, e, a) {
        var ss = s;
        var ee = e;
        var aa = a;
        while (ss > ee)
            ee += 360;
        ss -= s;
        ee -= s;
        aa -= s;
        while (aa < 0)
            aa += 360;
        if (ss < aa && aa < ee)
            return true;
        return false;
    }
    MathUtil.isInnerAngl360 = isInnerAngl360;
    /**
     * 円弧の始点・終点と点の距離を調べる
     * @param x
     * @param y
     * @param r
     * @param s
     * @param e
     * @param x
     * @param y
     * @param dir
     * @return
    **/
    function getLengthToArcStartEnd(cx, cy, r, s, x, y) {
        var xx = cx + r * Math.cos(s * MathUtil.D2R);
        var yy = cy + r * Math.sin(s * MathUtil.D2R);
        return MathUtil.getLength2D(x, y, xx, yy);
    }
    MathUtil.getLengthToArcStartEnd = getLengthToArcStartEnd;
    function rachet(angl, rat) {
        angl += Math.PI * 2 / rat;
        while (angl >= (Math.PI * 2))
            angl -= (Math.PI * 2);
        for (var i = 0; i < rat + 1; i++) {
            if (Math.abs(angl - i * Math.PI * 2 / rat) < (Math.PI / rat)) {
                angl = (Math.PI * 2 / rat * i);
                break;
            }
        }
        return angl;
    }
    MathUtil.rachet = rachet;
    /**
     * 2直線の角度の取得（時計回り方向）
     * @param ax1
     * @param ay1
     * @param ax2
     * @param ay2
     * @param bx1
     * @param by1
     * @param bx2
     * @param by2
     * @return
    **/
    function getAngle2Lines(x1, y1, x2, y2, x3, y3) {
        var angl1 = MathUtil.getAngle2D(x2, y2, x1, y1);
        var angl2 = MathUtil.getAngle2D(x2, y2, x3, y3);
        angl2 -= angl1;
        return MathUtil.getAngl2PI(angl2);
    }
    MathUtil.getAngle2Lines = getAngle2Lines;
    /**
     * ２Dでの角度の計算
     * @param x1 点１のX座標
     * @param y1 点１のY座標
     * @param x2 点２のX座標
     * @param y2 点２のY座標
     * @return 角度（ラジアン）
    **/
    function getAngle2D(x1, y1, x2, y2) {
        var dx;
        var dy;
        var t;
        var a;
        var area;
        dx = x2 - x1;
        dy = y2 - y1;
        if (dx >= 0.0) {
            if (dy >= 0.0)
                area = 0;
            else {
                area = 3;
                t = dx;
                dx = -dy;
                dy = t;
            }
        }
        else {
            if (dy >= 0.0) {
                area = 1;
                t = dx;
                dx = dy;
                dy = -t;
            }
            else {
                area = 2;
                dx = -dx;
                dy = -dy;
            }
        }
        if (dy > dx)
            a = Math.PI / 2.0 - Math.atan(dx / dy);
        else
            a = Math.atan(dy / dx);
        return a + area * (Math.PI / 2.0);
    }
    MathUtil.getAngle2D = getAngle2D;
    /**
     * 直線をローカル座標X軸とした場合の点のXの値を取得
     * @param x		検証する点X
     * @param y		検証する点Y
     * @param x1	原点側X
     * @param y1	原点側Y
     * @param x2	X軸側X
     * @param y2	X軸側Y
     * @return		ローカル座標系でのX座標
    **/
    function getX(x, y, x1, y1, x2, y2) {
        var cc;
        var ss;
        var rot = MathUtil.getAngle2D(x1, y1, x2, y2);
        cc = Math.cos(rot);
        ss = Math.sin(rot);
        x -= x1;
        y -= y1;
        return MathUtil.norm(x * cc + y * ss);
    }
    MathUtil.getX = getX;
    /**
     * 線分p1-p2が線分p3-p4に含まれるか**/
    function inIncludeLine(p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y) {
        return (MathUtil.isOnLimitLinePoint2D(p1x, p1y, p3x, p3y, p4x, p4y)
            && MathUtil.isOnLimitLinePoint2D(p2x, p2y, p3x, p3y, p4x, p4y));
    }
    MathUtil.inIncludeLine = inIncludeLine;
    /**
     * 直線をローカル座標X軸とした場合の点のYの値を取得
     * @param x
     * @param y
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @return
    **/
    function getY(x, y, x1, y1, x2, y2) {
        var cc;
        var ss;
        var rot = MathUtil.getAngle2D(x1, y1, x2, y2);
        cc = Math.cos(rot);
        ss = Math.sin(rot);
        x -= x1;
        y -= y1;
        return MathUtil.norm(-x * ss + y * cc);
    }
    MathUtil.getY = getY;
    /**
     * 象限を調べる。
     * @param px 検査する点のX座標
     * @param py 検査する点のY座標
     * @param x1 点１のX座標
     * @param y1 点１のY座標
     * @param x2 点２のX座標
     * @param y2 点２のY座標
    **/
    function getQuadrant(px, py, x1, y1, x2, y2) {
        var baseangl = MathUtil.getAngle2D(x1, y1, x2, y2);
        var angl = MathUtil.getAngle2D(x1, y1, px, py);
        var res = angl - baseangl < 0.0 ? angl - baseangl + (Math.PI * 2) : angl - baseangl;
        if (res >= 0.0 && res < (Math.PI / 2))
            return 1;
        else if (res >= (Math.PI / 2) && res < Math.PI)
            return 2;
        else if (res >= (Math.PI) && res < (Math.PI / 2 * 3))
            return 3;
        else
            return 4;
    }
    MathUtil.getQuadrant = getQuadrant;
    /**
     * 角度（ラジアン）を２＊PIの範疇に変更する
     * @param d
     * @return
    **/
    function getAngl2PI(d) {
        if (Math.abs(d) < MathUtil.EPS)
            d = 0;
        while (d < 0)
            d += (Math.PI * 2);
        while (d > (Math.PI * 2))
            d -= (Math.PI * 2);
        return d;
    }
    MathUtil.getAngl2PI = getAngl2PI;
    /**
     * 角度（度）360度をの範疇に変更する
     * @param d
     * @return
    **/
    function getAngl360(d) {
        while (d < 0)
            d += 360;
        while (d > 360)
            d -= 360;
        return d;
    }
    MathUtil.getAngl360 = getAngl360;
    /**
     * ２Dでの点の回転
     * @param xc 回転中心のX座標
     * @param yc 回転中心のX座標
     * @param rot 回転角度
     * @param xsrc 対象の点のX座標
     * @param ysrc 対象の点のY座標
     * @param res 結果座標
    **/
    function rotation2D(xc, yc, rot, xsrc, ysrc, res) {
        var cc;
        var ss;
        cc = Math.cos(-rot);
        ss = Math.sin(-rot);
        xsrc -= xc;
        ysrc -= yc;
        if (res != null) {
            res[0] = MathUtil.norm(xsrc * cc + ysrc * ss + xc);
            res[1] = MathUtil.norm(-xsrc * ss + ysrc * cc + yc);
        }
    }
    MathUtil.rotation2D = rotation2D;
    /**
     * 2次元の2点の間の距離を調べる
     * @param x1 点１のX座標
     * @param y1 点１のY座標
     * @param x2 点２のX座標
     * @param y2 点２のY座標
    **/
    function getLength2D(x1, y1, x2, y2) {
        return MathUtil.norm(Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)));
    }
    MathUtil.getLength2D = getLength2D;
    /**
     * 2点が同じ位置にあるか検査
     * @param p1x 点1のX座標
     * @param p1y 点1のY座標
     * @param p2x 点2のX座標
     * @param p2y 点2のY座標
    **/
    function isSame2D(p1x, p1y, p2x, p2y) {
        if (Math.abs(p2x - p1x) < MathUtil.EPS && Math.abs(p2y - p1y) < MathUtil.EPS)
            return true;
        return false;
    }
    MathUtil.isSame2D = isSame2D;
    /**
     * 2直線が同じか調べる
     * @param pa1x 点１
    **/
    function isSameLine(pa1x, pa1y, pa2x, pa2y, pb1x, pb1y, pb2x, pb2y) {
        if ((MathUtil.isSame2D(pa1x, pa1y, pb1x, pb1y) && MathUtil.isSame2D(pa2x, pa2y, pb2x, pb2y))
            || (MathUtil.isSame2D(pa1x, pa1y, pb2x, pb2y) && MathUtil.isSame2D(pa2x, pa2y, pb1x, pb1y)))
            return true;
        return false;
    }
    MathUtil.isSameLine = isSameLine;
    /**
     * aはbに含まれるか**/
    function isIncludeLine(pa1x, pa1y, pa2x, pa2y, pb1x, pb1y, pb2x, pb2y) {
        if (MathUtil.isOnLimitLinePoint2D(pa1x, pa1y, pb1x, pb1y, pb2x, pb2y)
            && MathUtil.isOnLimitLinePoint2D(pa2x, pa2y, pb1x, pb1y, pb2x, pb2y))
            return true;
        return false;
    }
    MathUtil.isIncludeLine = isIncludeLine;
    /**
     * 2線分が平行に関係を持つか
     * (平行かつ重なるか)**/
    function isRelatePara2Line(pa1x, pa1y, pa2x, pa2y, pb1x, pb1y, pb2x, pb2y) {
        if (!MathUtil.isOnLine(pa1x, pa1y, pa2x, pa2y, pb1x, pb1y, pb2x, pb2y))
            return false;
        if (MathUtil.isOnLimitLinePoint2D(pa1x, pa1y, pb1x, pb1y, pb2x, pb2y)
            || MathUtil.isOnLimitLinePoint2D(pa2x, pa2y, pb1x, pb1y, pb2x, pb2y)
            || MathUtil.isOnLimitLinePoint2D(pb1x, pb1y, pa1x, pa1y, pa2x, pa2y)
            || MathUtil.isOnLimitLinePoint2D(pb2x, pb2y, pa1x, pa1y, pa2x, pa2y))
            return true;
        return false;
    }
    MathUtil.isRelatePara2Line = isRelatePara2Line;
    function isVerticalLimitLineToPoint2D(px, py, p1x, p1y, p2x, p2y, tmp) {
        var xp1;
        var yp1;
        var x21;
        var y21;
        var denom;
        xp1 = p1x - px;
        yp1 = p1y - py;
        x21 = p2x - p1x;
        y21 = p2y - p1y;
        denom = x21 * x21 + y21 * y21;
        var t = -(xp1 * x21 + yp1 * y21) / denom;
        if (t >= 0.0 && t <= 1.0) {
            if (tmp != null) {
                var x;
                var y;
                x = xp1 + t * x21;
                y = yp1 + t * y21;
                tmp[0] = (px + x);
                tmp[1] = (py + y);
            }
            return true;
        }
        return false;
    }
    MathUtil.isVerticalLimitLineToPoint2D = isVerticalLimitLineToPoint2D;
    /**
     * 点から直線(無限)に垂直に降ろした交点を調べる
     * @param px	検査する点のX座標
     * @param py	検査する点のY座標
     * @param p1x	線の点１のX座標
     * @param p1y	線の点１のY座標
     * @param p2x	線の点２のX座標
     * @param p2y	線の点２のY座標
    **/
    function isCrossPointVertLimitlessLineToPoint2D(px, py, p1x, p1y, p2x, p2y, res) {
        var a1 = MathUtil.getLineEquation2Da(p1x, p1y, p2x, p2y);
        var b1 = MathUtil.getLineEquation2Db(p1x, p1y, p2x, p2y);
        var c1 = MathUtil.getLineEquation2Dc(p1x, p1y, p2x, p2y);
        var a2 = b1;
        var b2 = -a1;
        var c2 = a1 * py - b1 * px;
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        if (res != null) {
            res[0] = MathUtil.norm((b1 * c2 - b2 * c1) / det);
            res[1] = MathUtil.norm((a2 * c1 - a1 * c2) / det);
        }
        return true;
    }
    MathUtil.isCrossPointVertLimitlessLineToPoint2D = isCrossPointVertLimitlessLineToPoint2D;
    /**
     * 点と線分の距離を調べる
     * @param px 検査する点のX座標
     * @param py 検査する点のY座標
     * @param p1x 線分の点1のx座標
     * @param p1y 線分の点1のY座標
     * @param p2x 線分の点2のx座標
     * @param p2y 線分の点2のY座標
    **/
    function getLengthLimitLineToPoint2D(px, py, p1x, p1y, p2x, p2y) {
        var xp1;
        var yp1;
        var x21;
        var y21;
        var denom;
        xp1 = p1x - px;
        yp1 = p1y - py;
        x21 = p2x - p1x;
        y21 = p2y - p1y;
        denom = x21 * x21 + y21 * y21;
        var t = -(xp1 * x21 + yp1 * y21) / denom;
        if (t >= 0.0 && t <= 1.0) {
            var x;
            var y;
            x = xp1 + t * x21;
            y = yp1 + t * y21;
            return MathUtil.norm(Math.sqrt(x * x + y * y));
        }
        return -1.0;
    }
    MathUtil.getLengthLimitLineToPoint2D = getLengthLimitLineToPoint2D;
    /**
     * 点と直線(無限)の距離を調べる
     * @param px	検査する点のX座標
     * @param py	検査する点のY座標
     * @param p1x	線分の点1のx座標
     * @param p1y	線分の点1のY座標
     * @param p2x	線分の点2のx座標
     * @param p2y	線分の点2のY座標
     * @return	点と線分の距離。計算できなければ-1を返す。
    **/
    function getLengthLimitlessLineToPoint2D(px, py, p1x, p1y, p2x, p2y) {
        if (MathUtil.isSame2D(p1x, p1y, p2x, p2y))
            return -1.0;
        var a = MathUtil.getLineEquation2Da(p1x, p1y, p2x, p2y);
        var b = MathUtil.getLineEquation2Db(p1x, p1y, p2x, p2y);
        var c = MathUtil.getLineEquation2Dc(p1x, p1y, p2x, p2y);
        var absq = a * a + b * b;
        if (absq < MathUtil.EPS)
            return -1.0;
        var sr = a * px + b * py + c;
        return MathUtil.norm(Math.sqrt(sr * sr / absq));
    }
    MathUtil.getLengthLimitlessLineToPoint2D = getLengthLimitlessLineToPoint2D;
    /**
     * 点と直線(無限)の位置を調べる
     * @param px	検査する点のX座標
     * @param py	検査する点のY座標
     * @param p1x	線分の点1のx座標
     * @param p1y	線分の点1のY座標
     * @param p2x	線分の点2のx座標
     * @param p2y	線分の点2のY座標
     * @return	上にあればtrue.下ならfalse。
    **/
    function isPointPickUporDown(px, py, p1x, p1y, p2x, p2y) {
        var a = MathUtil.getLineEquation2Da(p1x, p1y, p2x, p2y);
        var b = MathUtil.getLineEquation2Db(p1x, p1y, p2x, p2y);
        var c = MathUtil.getLineEquation2Dc(p1x, p1y, p2x, p2y);
        return a * px + b * py + c >= 0.0;
    }
    MathUtil.isPointPickUporDown = isPointPickUporDown;
    /**
     * 点１から点２の方向にofsx,
     * そこから垂直にofsy（左が正）の位置を算出
     * @param x1		点１のX座標
     * @param y1		点１のY座標
     * @param x2		点２のX座標
     * @param y2		点２のY座標
     * @param ofsx	xオフセット(線方向)
     * @param ofsy	yオフセット(法線方向)
     * @param res		結果座標
    **/
    function getOffsetPoint2D(x1, y1, x2, y2, ofsx, ofsy, res) {
        var angl = MathUtil.getAngle2D(x1, y1, x2, y2);
        if (res != null) {
            res[0] = MathUtil.norm(x1 + (ofsx * Math.cos(angl)) + (ofsy * Math.cos(angl + Math.PI / 2.0)));
            res[1] = MathUtil.norm(y1 + (ofsx * Math.sin(angl)) + (ofsy * Math.sin(angl + Math.PI / 2.0)));
        }
    }
    MathUtil.getOffsetPoint2D = getOffsetPoint2D;
    function getOffsetLength2D(x1, y1, x2, y2, ofsx, ofsy, res) {
        var angl = MathUtil.getAngle2D(x1, y1, x2, y2);
        if (res != null) {
            res[0] = ((ofsx * Math.cos(angl)) + (ofsy * Math.cos(angl + Math.PI / 2.0)));
            res[1] = ((ofsx * Math.sin(angl)) + (ofsy * Math.sin(angl + Math.PI / 2.0)));
        }
    }
    MathUtil.getOffsetLength2D = getOffsetLength2D;
    /**
     * 3点を通る円の半径と中心点の取得**/
    function get3PointCircle(x1, y1, x2, y2, x3, y3, res) {
        var xc11 = (x2 + x1) / 2;
        var yc11 = (y2 + y1) / 2;
        var xc21 = (x3 + x1) / 2;
        var yc21 = (y3 + y1) / 2;
        var xc12 = xc11 - (y2 - y1);
        var yc12 = yc11 + (x2 - x1);
        var xc22 = xc21 - (y3 - y1);
        var yc22 = yc21 + (x3 - x1);
        if (MathUtil.isCrossInfinity(xc11, yc11, xc12, yc12, xc21, yc21, xc22, yc22, res)) {
            return MathUtil.getLength2D(res[0], res[1], x1, y1);
        }
        return -1;
    }
    MathUtil.get3PointCircle = get3PointCircle;
    /**
     * 3点を通る円の半径と中心点の取得**/
    function get2LineArc(x1, y1, x2, y2, x3, y3, res) {
        var xc1 = x1 - (y2 - y1);
        var yc1 = y1 + (x2 - x1);
        var xc2 = x3 - (y3 - y2);
        var yc2 = y3 + (x3 - x2);
        if (MathUtil.isCrossInfinity(x1, y1, xc1, yc1, x3, y3, xc2, yc2, res))
            return MathUtil.getLength2D(res[0], res[1], x1, y1);
        return -1;
    }
    MathUtil.get2LineArc = get2LineArc;
    /**
     * 2線分が交点を持つか調べる
     * @param ap1x	線１の点１のX座標
     * @param ap1y	線１の点１のY座標
     * @param bp1x	線２の点１のX座標
     * @param bp1y	線２の点１のY座標
     * @param res		結果座標
    **/
    function isCrossSegment(ap1x, ap1y, ap2x, ap2y, bp1x, bp1y, bp2x, bp2y, res) {
        var x21 = ap2x - ap1x;
        var y21 = ap2y - ap1y;
        var x43 = bp2x - bp1x;
        var y43 = bp2y - bp1y;
        var x31 = bp1x - ap1x;
        var y31 = bp1y - ap1y;
        var det = x43 * y21 - y43 * x21;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        var s = (x43 * y31 - y43 * x31) / det;
        var t = (x21 * y31 - y21 * x31) / det;
        if (s < -MathUtil.EPS || s > (1.0 + MathUtil.EPS) || t < -MathUtil.EPS || t > (1.0 + MathUtil.EPS))
            return false;
        if (res != null) {
            res[0] = MathUtil.norm(ap1x + x21 * s);
            res[1] = MathUtil.norm(ap1y + y21 * s);
        }
        return true;
    }
    MathUtil.isCrossSegment = isCrossSegment;
    /**
     * 2直線(無限)が交点を持つか調べる
     * @param ap1x	線１の点１のX座標
     * @param ap1y	線１の点１のY座標
     * @param ap2x	線１の点２のX座標
     * @param ap2y	線１の点２のY座標
     * @param bp1x	線２の点１のX座標
     * @param bp1y	線２の点１のY座標
     * @param bp2x	線２の点２のX座標
     * @param bp2y	線２の点２のY座標
     * @param res		結果座標
    **/
    function isCrossInfinity(ap1x, ap1y, ap2x, ap2y, bp1x, bp1y, bp2x, bp2y, res) {
        var a1 = MathUtil.getLineEquation2Da(ap1x, ap1y, ap2x, ap2y);
        var b1 = MathUtil.getLineEquation2Db(ap1x, ap1y, ap2x, ap2y);
        var c1 = MathUtil.getLineEquation2Dc(ap1x, ap1y, ap2x, ap2y);
        var a2 = MathUtil.getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
        var b2 = MathUtil.getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
        var c2 = MathUtil.getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        if (res != null) {
            res[0] = MathUtil.norm((b1 * c2 - b2 * c1) / det);
            res[1] = MathUtil.norm((a2 * c1 - a1 * c2) / det);
        }
        return true;
    }
    MathUtil.isCrossInfinity = isCrossInfinity;
    function isCrossInfinityEq(a1, b1, c1, a2, b2, c2, res) {
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        if (res != null) {
            res[0] = MathUtil.norm((b1 * c2 - b2 * c1) / det);
            res[1] = MathUtil.norm((a2 * c1 - a1 * c2) / det);
        }
        return true;
    }
    MathUtil.isCrossInfinityEq = isCrossInfinityEq;
    function isCrossInfinityPointAndAngle(x1, y1, ang1, xa, ya, anga, tmpf) {
        var x2 = x1 + 100 * Math.cos(ang1);
        var y2 = y1 + 100 * Math.sin(ang1);
        var xb = xa + 100 * Math.cos(anga);
        var yb = ya + 100 * Math.sin(anga);
        return MathUtil.isCrossInfinity(x1, y1, x2, y2, xa, ya, xb, yb, tmpf);
    }
    MathUtil.isCrossInfinityPointAndAngle = isCrossInfinityPointAndAngle;
    function isCrossXandLimitLine(x, bp1x, bp1y, bp2x, bp2y, res) {
        var a1 = MathUtil.getLineEquation2Da(x, 0, x, 100);
        var b1 = MathUtil.getLineEquation2Db(x, 0, x, 100);
        var c1 = MathUtil.getLineEquation2Dc(x, 0, x, 100);
        var a2 = MathUtil.getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
        var b2 = MathUtil.getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
        var c2 = MathUtil.getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        if (res != null) {
            var xx = MathUtil.norm((b1 * c2 - b2 * c1) / det);
            var yy = MathUtil.norm((a2 * c1 - a1 * c2) / det);
            if (MathUtil.isOnLimitLinePoint2D(xx, yy, bp1x, bp1y, bp2x, bp2y)) {
                res[0] = xx;
                res[1] = yy;
                return true;
            }
        }
        return false;
    }
    MathUtil.isCrossXandLimitLine = isCrossXandLimitLine;
    function isCrossYandLimitLine(y, bp1x, bp1y, bp2x, bp2y, res) {
        var a1 = MathUtil.getLineEquation2Da(0, y, 100, y);
        var b1 = MathUtil.getLineEquation2Db(0, y, 100, y);
        var c1 = MathUtil.getLineEquation2Dc(0, y, 100, y);
        var a2 = MathUtil.getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
        var b2 = MathUtil.getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
        var c2 = MathUtil.getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return false;
        if (res != null) {
            var xx = MathUtil.norm((b1 * c2 - b2 * c1) / det);
            var yy = MathUtil.norm((a2 * c1 - a1 * c2) / det);
            if (MathUtil.isOnLimitLinePoint2D(xx, yy, bp1x, bp1y, bp2x, bp2y)) {
                res[0] = xx;
                res[1] = yy;
                return true;
            }
        }
        return false;
    }
    MathUtil.isCrossYandLimitLine = isCrossYandLimitLine;
    /**
     * 2直線が同一線上に存在するか
     * @param ap1x	線１の点１のX座標
     * @param ap1y	線１の点１のY座標
     * @param ap2x	線１の点２のX座標
     * @param ap2y	線１の点２のY座標
     * @param bp1x	線２の点１のX座標
     * @param bp1y	線２の点１のY座標
     * @param bp2x	線２の点２のX座標
     * @param bp2y	線２の点２のY座標
    **/
    function isOnLine(ap1x, ap1y, ap2x, ap2y, bp1x, bp1y, bp2x, bp2y) {
        var d1 = MathUtil.getLengthLimitlessLineToPoint2D(bp1x, bp1y, ap1x, ap1y, ap2x, ap2y);
        var d2 = MathUtil.getLengthLimitlessLineToPoint2D(bp2x, bp2y, ap1x, ap1y, ap2x, ap2y);
        return (d1 >= 0.0 && d1 < MathUtil.EPS) && (d2 >= 0.0 && d2 < MathUtil.EPS);
    }
    MathUtil.isOnLine = isOnLine;
    /**
     * 点が線分上にあるか調べる
     * @param px	検査する点のX座標
     * @param py	検査する点のY座標
     * @param p1x	線分の点１のX座標
     * @param p1y	線分の点１のY座標
     * @param p2x	線分の点２のX座標
     * @param p2y	線分の点２のY座標
    **/
    function isOnLimitLinePoint2D(px, py, p1x, p1y, p2x, p2y) {
        if (MathUtil.isSame2D(px, py, p1x, p1y) || MathUtil.isSame2D(px, py, p2x, p2y))
            return true;
        var r = MathUtil.getLengthLimitLineToPoint2D(px, py, p1x, p1y, p2x, p2y);
        if (r < MathUtil.EPS && r >= 0.0)
            return true;
        return false;
    }
    MathUtil.isOnLimitLinePoint2D = isOnLimitLinePoint2D;
    /**
     * 2直線が平行か調べる
     * @param ap1x	線１の点１のX座標
     * @param ap1y	線１の点１のY座標
     * @param ap2x	線１の点２のX座標
     * @param ap2y	線１の点２のY座標
     * @param bp1x	線２の点１のX座標
     * @param bp1y	線２の点１のY座標
     * @param bp2x	線２の点２のX座標
     * @param bp2y	線２の点２のY座標
    **/
    function isParallel2D(ap1x, ap1y, ap2x, ap2y, bp1x, bp1y, bp2x, bp2y) {
        var a1 = MathUtil.getLineEquation2Da(ap1x, ap1y, ap2x, ap2y);
        var b1 = MathUtil.getLineEquation2Db(ap1x, ap1y, ap2x, ap2y);
        var a2 = MathUtil.getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
        var b2 = MathUtil.getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
        var det = a1 * b2 - a2 * b1;
        if (Math.abs(det) < MathUtil.EPS)
            return true;
        return false;
    }
    MathUtil.isParallel2D = isParallel2D;
    /**
     * 二直線から垂直にオフセットした交点を計算**/
    function get2LineOffsetCross(x1, y1, x2, y2, x3, y3, ofs1, ofs2, tmp) {
        if (MathUtil.isParallel2D(x1, y1, x2, y2, x2, y2, x3, y3)) {
            MathUtil.getOffsetPoint2D(x2, y2, x3, y3, 0, ofs1, tmp);
            return true;
        }
        var p1x;
        var p1y;
        var p2x;
        var p2y;
        var p3x;
        var p3y;
        var p4x;
        var p4y;
        MathUtil.getOffsetPoint2D(x1, y1, x2, y2, 0, ofs1, tmp);
        p1x = MathUtil.norm(tmp[0]);
        p1y = MathUtil.norm(tmp[1]);
        MathUtil.getOffsetPoint2D(x2, y2, x1, y1, 0, -ofs1, tmp);
        p2x = MathUtil.norm(tmp[0]);
        p2y = MathUtil.norm(tmp[1]);
        MathUtil.getOffsetPoint2D(x2, y2, x3, y3, 0, ofs2, tmp);
        p3x = MathUtil.norm(tmp[0]);
        p3y = MathUtil.norm(tmp[1]);
        MathUtil.getOffsetPoint2D(x3, y3, x2, y2, 0, -ofs2, tmp);
        p4x = MathUtil.norm(tmp[0]);
        p4y = MathUtil.norm(tmp[1]);
        return MathUtil.isCrossInfinity(p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y, tmp);
    }
    MathUtil.get2LineOffsetCross = get2LineOffsetCross;
    /**
     * 楕円と楕円中心から点へ繋がる線の交点を算出
     * 線は無限。
     * @param x		楕円中心と繋がる点のX座標
     * @param y		楕円中心と繋がる点のY座標
     * @param cpx	楕円中心X
     * @param cpy	楕円中心Y
     * @param rh	楕円幅半径
     * @param rv	楕円高半径
     * @param tmp	交点
    **/
    function getCrossPointEllipse(x, y, cpx, cpy, rh, rv, tmp) {
        var ex;
        var ey;
        if ((x - cpx) != 0) {
            var c = (y - cpy) / (x - cpx);
            ex = rv * rh / Math.sqrt(Math.abs(rh * rh + rv * c * rv * c));
            if ((x - cpx) < 0)
                ex *= -1;
            ey = ex * c;
        }
        else {
            ex = 0;
            ey = ((y - cpy) >= 0) ? rh : -rh;
        }
        tmp[0] = (ex + cpx);
        tmp[1] = (ey + cpy);
    }
    MathUtil.getCrossPointEllipse = getCrossPointEllipse;
    /**
     * 楕円度の取得
     * @param rh
     * @param rv
     * @param angl
     * @return
    **/
    function getEllipseAngl(rh, rv, angl) {
        var angle = Math.atan((rh / rv * Math.tan(angl)));
        var quad = MathUtil.getQuadrant(Math.cos(angl), Math.sin(angl), 0, 0, 100, 0);
        switch (quad) {
            case 1:
                break;
            case 2:
                if (angle > 0)
                    angle *= -1;
                angle = (Math.PI + angle);
                break;
            case 3:
                if (angle < 0)
                    angle *= -1;
                angle = (Math.PI + angle);
                break;
            case 4:
                if (angle > 0)
                    angle *= -1;
                angle = (Math.PI * 2 + angle);
                break;
        }
        return angle;
    }
    MathUtil.getEllipseAngl = getEllipseAngl;
    /**
     * 直線の方程式の切片Aを算出
     * @param p1x	線分の点1のx座標
     * @param p1y	線分の点1のY座標
     * @param p2x	線分の点2のx座標
     * @param p2y	線分の点2のY座標
     * @return
    **/
    function getLineEquation2Da(p1x, p1y, p2x, p2y) {
        var xlk = p2x - p1x;
        var ylk = p2y - p1y;
        var rsq = xlk * xlk + ylk * ylk;
        var rinv = 1. / Math.sqrt(rsq);
        return MathUtil.norm(-ylk * rinv);
    }
    MathUtil.getLineEquation2Da = getLineEquation2Da;
    /**
     * 直線の方程式の切片Bを算出
     * @param p1x	線分の点1のx座標
     * @param p1y	線分の点1のY座標
     * @param p2x	線分の点2のx座標
     * @param p2y	線分の点2のY座標
     * @return
    **/
    function getLineEquation2Db(p1x, p1y, p2x, p2y) {
        var xlk = p2x - p1x;
        var ylk = p2y - p1y;
        var rsq = xlk * xlk + ylk * ylk;
        var rinv = 1. / Math.sqrt(rsq);
        return MathUtil.norm(xlk * rinv);
    }
    MathUtil.getLineEquation2Db = getLineEquation2Db;
    /**
     * 直線の方程式の切片Cを算出
     * @param p1x	線分の点1のx座標
     * @param p1y	線分の点1のY座標
     * @param p2x	線分の点2のx座標
     * @param p2y	線分の点2のY座標
     * @return
    **/
    function getLineEquation2Dc(p1x, p1y, p2x, p2y) {
        var xlk = p2x - p1x;
        var ylk = p2y - p1y;
        var rsq = xlk * xlk + ylk * ylk;
        var rinv = 1. / Math.sqrt(rsq);
        return MathUtil.norm((p1x * p2y - p2x * p1y) * rinv);
    }
    MathUtil.getLineEquation2Dc = getLineEquation2Dc;
    /**
     * 丸める**/
    function norm(f) {
        return (Math.round(f * 1e8) / 1e8);
    }
    MathUtil.norm = norm;
    /**
     * メソッド名　 : 配列の平均化？<BR>
     * 処理内容　　 : <BR>
     * 事前条件 　　: <BR>
     * 事後条件　　 : <BR>
     * @param vt 配列
     * @return        標準偏差？
     * @exception なし
     * @since         2003/09/10
     * @author        fujita
     * @version       2003/09/10
    **/
    function normalizeVector(vt) {
        var EPS = 1.0e-6;
        var vector_length;
        vector_length = Math.sqrt(vt[0] * vt[0] + vt[1] * vt[1] + vt[2] * vt[2]);
        if (vector_length > EPS) {
            var inv_vector_length = (1.0 / vector_length);
            vt[0] *= inv_vector_length;
            vt[1] *= inv_vector_length;
            vt[2] *= inv_vector_length;
        }
        return vector_length;
    }
    MathUtil.normalizeVector = normalizeVector;
    /**
     * メソッド名　 : ３次元空間における２点間の距離の取得<BR>
     * 処理内容　　 : ３次元空間において、a 点と b 点間の距離を算出し返す。<BR>
     * 事前条件 　　: <BR>
     * 事後条件　　 : <BR>
     * @param a ３次元空間での点
     * @param b ３次元空間での点
     * @return        ２点間の距離
     * @exception なし
     * @since         2003/09/10
     * @author        fujita
     * @version       2003/09/10
    **/
    function getDistanceOn3D(a, b) {
        var x;
        var y;
        var z;
        var aq;
        x = a[0] - b[0];
        y = a[1] - b[1];
        z = a[2] - b[2];
        aq = x * x + y * y + z * z;
        return (Math.sqrt(aq));
    }
    MathUtil.getDistanceOn3D = getDistanceOn3D;
    /**
     * メソッド名　 : <BR>
     * 処理内容　　 : <BR>
     * 事前条件 　　: <BR>
     * 事後条件　　 : <BR>
     * @param なし
     * @return        なし
     * @exception なし
     * @since         2003/09/10
     * @author        fujita
     * @version       2003/09/10
    **/
    function calcOuterProduct(a, b, c) {
        c[0] = a[1] * b[2] - a[2] * b[1];
        c[1] = a[2] * b[0] - a[0] * b[2];
        c[2] = a[0] * b[1] - a[1] * b[0];
    }
    MathUtil.calcOuterProduct = calcOuterProduct;
})(MathUtil = exports.MathUtil || (exports.MathUtil = {}));
//# sourceMappingURL=MathUtil.js.map