var KPoint2d_1 = require('../vecmath/KPoint2d');
var KFont_1 = require('../kwt/KFont');
var IRichGraphics_1 = require('../kwt/IRichGraphics');
var KLineType_1 = require('../kwt/KLineType');
var RichGraphicsUtil_1 = require('../kwt/RichGraphicsUtil');
var K2DFigureType_1 = require('../geom/K2DFigureType');
var PdfGraphics_1 = require('./PdfGraphics');
var util_1 = require('./util');
//module option.pdf
/**
 * IRichGraphics を使用して、PdfCommandGraphics への描画を行うアダプタ。
 * @since		2005/02/23
 * @author		fujita
 * @version	2005/02/23Copyright (c) 2005 KITec Inc,.. All rights reserved.
**/
var PdfRichGraphicsAdapter = (function () {
    /**
     * デフォルトの内容を持つ新たな PdfRichGraphicsAdapter を作成する。
     * @param g	PDF グラフィックス
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    function PdfRichGraphicsAdapter(g) {
        /**
         * ポリゴン等の描画用の x 座標のバッファ **/
        this.XBUF = new Array(PdfRichGraphicsAdapter.BUF_SIZE);
        /**
         * ポリゴン等の描画用の y 座標のバッファ **/
        this.YBUF = new Array(PdfRichGraphicsAdapter.BUF_SIZE);
        this.pdfPathMode = false;
        this.mGraphics = g;
    }
    PdfRichGraphicsAdapter.prototype.scale = function (scale) {
        this.mScale = scale;
    };
    PdfRichGraphicsAdapter.prototype.getScale = function () {
        return this.mScale;
    };
    /**
     * 指定された flag により、フォントをファイルに 埋め込むかどうかを指定する。
     * @param flag	フォントを埋め込む場合、true。そうでない場合は、false
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.setFontEmbed = function (flag) {
        this.mGraphics.setFontEmbed(flag);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#setClipArea(double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.setClipArea = function (x1, y1, x2, y2) {
        this.mGraphics.setClip(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1));
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#releaseClipArea()
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.releaseClipArea = function () {
        this.mGraphics.releaseClip();
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#setColor(java.awt.Color)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.setColor = function (c) {
        this.mGraphics.setColor(c);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawLine(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawLine = function (x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawLine(x1, y1, x2, y2);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawCircle(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawCircle = function (xc1, yc1, ra, ss, ee, lt) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - ra, yc1 + ra, ra * 2, ra * 2, ss, ee, false);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillCircle(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawFillCircle = function (xc1, yc1, ra, ss, ee, lt) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - ra, yc1 + ra, ra * 2, ra * 2, ss, ee, true);
    };
    /**
     * 指定された内容をもとに、楕円の弧を描く。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawEllipse = function (xc1, yc1, v, h, ss, ee, lt) {
        this.drawEllipse_(xc1, yc1, v, h, ss, ee, lt, false);
    };
    /**
     * 指定された内容をもとに、楕円の弧を描き、 その内部を塗りつぶす。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawFillEllipse = function (xc1, yc1, v, h, ss, ee, lt) {
        this.drawEllipse_(xc1, yc1, v, h, ss, ee, lt, true);
    };
    /**
     * 指定された内容をもとに、楕円の弧を描く。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @param fill	楕円を塗りつぶす場合、true。そうでない場合は、false
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawEllipse_ = function (xc1, yc1, v, h, ss, ee, lt, fill) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - h, yc1 + v, h * 2, v * 2, ss, ee, fill);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawRect(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawRect = function (x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), false);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillRect(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawFillRect = function (x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), true);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawRoundRect(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawRoundRect = function (x1, y1, x2, y2, round, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRoundRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), round * 2, round * 2, false);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillRoundRect(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawFillRoundRect = function (x1, y1, x2, y2, round, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRoundRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), round * 2, round * 2, true);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawPolygon(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawPolygon = function (poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        this.mGraphics.drawPolygon(this.XBUF, this.YBUF, poly.length);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillPolygon(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawFillPolygon = function (poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        this.mGraphics.fillPolygon(this.XBUF, this.YBUF, poly.length);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawPolyLine(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2005/02/23
     * @author		fujita
     * @version	2005/02/23
    **/
    PdfRichGraphicsAdapter.prototype.drawPolyLine = function (poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        for (var i = 0; i < poly.length - 1; i++) {
            this.drawLine(this.XBUF[i], this.YBUF[i], this.XBUF[i + 1], this.YBUF[i + 1], lt);
        }
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,java.lang.String,double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawImageId = function (img, id, x1, y1, x2, y2) {
        this.drawImagePosition(img, x1, y1, x2, y2);
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.drawImagePosition = function (img, x1, y1, x2, y2) {
        this.mGraphics.drawImageRect(img, Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1));
    };
    PdfRichGraphicsAdapter.prototype.drawImageRect = function (img, cx, cy, width, height, angle) {
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawText(java.lang.String,double,double,java.lang.String,int,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2005/02/23	fujita文字の回転を反映するよう変更。
    **/
    PdfRichGraphicsAdapter.prototype.drawText = function (s, x1, y1, fontname, fonttype, size, angle) {
        if (s == null || s.length == 0) {
            return;
        }
        this.mGraphics.setFontParam(fontname, fonttype, size);
        var f = KFont_1.KFont.getFont(fontname, fonttype, size);
        this.mGraphics.drawString(s, x1, y1, false, -angle, f.getFontMetrics(null).stringWidth(s), PdfGraphics_1.PdfGraphics.SPACE_FOR_PITCH);
    };
    ///**
    // * @see jp.kitec.kwt.IRichGraphics#drawShadePolygon(java.util.Vector,double,double,double,double,java.awt.Color,java.awt.Color)
    // * @since		2004/09/29
    // * @author		kawae
    // * @version	2004/09/29
    //**/
    //drawShadePolygon(polygon: Array<RefList<number[]>>, x1: number, y1: number, x2: number, y2: number, c1: KColor, c2: KColor): void {
    //    let pointsize = 0;
    //    let npts: number[] = new Array(polygon.size());
    //    for (let i: number = 0; i < polygon.size(); i++) {
    //        let r: RefList<number[]> = (polygon.get(i)) as RefList;
    //        pointsize += r.size();
    //    }
    //    let x: number[] = new Array(pointsize);
    //    let y: number[] = new Array(pointsize);
    //    let count = 0;
    //    for (let i: number = 0; i < polygon.size(); i++) {
    //        let r: RefList<number[]> = (polygon.get(i)) as RefList;
    //        npts[i] = r.size();
    //        for (let j: number = 0; j < r.size(); j++) {
    //            let f: number[] = (r.elementAt(j)) as Array;
    //            x[count] = f[0];
    //            y[count] = f[1];
    //            count++;
    //        }
    //    }
    //    let formatter = new DecimalFormat("000");
    //    let shadename = new StringBuilder("TYPE2_");
    //    shadename.append(formatter.format(c1.getRed()));
    //    shadename.append(formatter.format(c1.getGreen()));
    //    shadename.append(formatter.format(c1.getBlue()));
    //    shadename.append(formatter.format(c2.getRed()));
    //    shadename.append(formatter.format(c2.getGreen()));
    //    shadename.append(formatter.format(c2.getBlue()));
    //    this.mGraphics.drawShadingGroupPolygon(shadename.toString(), x, y, npts);
    //}
    /**
     * 線種を設定する。
     * @param lt	線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.setLineType = function (lt) {
        if (lt != null) {
            this.mGraphics.setLineWidth(lt.width());
            if (lt.stroke != null)
                this.mGraphics.setLinePattern(lt.stroke());
            else
                this.mGraphics.setLinePattern(PdfRichGraphicsAdapter.NULL_LINE.stroke());
        }
        else {
            this.mGraphics.setLineWidth(PdfRichGraphicsAdapter.NULL_LINE.width());
            this.mGraphics.setLinePattern(PdfRichGraphicsAdapter.NULL_LINE.stroke());
        }
    };
    /**
     * コレクションノードを配列に直す
     * @param pts
     * @author kawae
     * @since 2004/09/29
    **/
    PdfRichGraphicsAdapter.prototype.createFloatPointArrayBuffer = function (pts) {
        if (pts.length > this.XBUF.length) {
            this.XBUF = new Array(pts.length);
            this.YBUF = new Array(pts.length);
        }
        for (var i = 0; i < pts.length; i++) {
            this.XBUF[i] = pts[i].getX();
            this.YBUF[i] = pts[i].getY();
        }
    };
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,java.lang.String,java.lang.String,double,double,double,double)
     * @param img
     * @param prefix
     * @param id
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @since 2006/01/24
     * @author kamimura
     * @version 2006/01/24
    **/
    PdfRichGraphicsAdapter.prototype.drawImagePrefixId = function (img, prefix, id, x1, y1, x2, y2) {
        this.drawImagePosition(img, x1, y1, x2, y2);
    };
    PdfRichGraphicsAdapter.prototype.pathInit = function (i) {
        this.mCurrentPoint = null;
        switch (i) {
            case IRichGraphics_1.IRichGraphicsConst.WIND_EVEN_ODD:
                this.mWindingRule = 0;
                break;
            case IRichGraphics_1.IRichGraphicsConst.WIND_NON_ZERO:
                this.mWindingRule = 1;
                break;
        }
        this.mGraphics.draw("[multiline]");
        this.mGraphics.draw("path,start");
        this.pdfPathMode = true;
    };
    PdfRichGraphicsAdapter.prototype.pathClear = function () {
        this.mCurrentPoint = null;
        this.mGraphics.draw("path,end");
        this.mGraphics.draw("[multiline]");
        this.pdfPathMode = false;
    };
    PdfRichGraphicsAdapter.prototype.pathMoveTo = function (x, y, la) {
        var xx = this.getGlobalX(x, y, la);
        var yy = this.getGlobalY(x, y, la);
        if (isNaN(xx) || isNaN(yy))
            throw new Error();
        if (!isFinite(xx) || !isFinite(yy))
            throw new Error();
        var p = this.mCurrentPoint;
        if (p != null && RichGraphicsUtil_1.RichGraphicsUtil.isSame2D(xx, yy, p.getX(), p.getY()))
            return;
        var sb = new util_1.StringBuilder("moveto");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.updateCurrentPoint(xx, yy);
        this.appendPoint(sb, xx, yy);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    };
    PdfRichGraphicsAdapter.prototype.pathLineTo = function (x, y, la) {
        var sb = new util_1.StringBuilder("lineto");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        var xx = this.getGlobalX(x, y, la);
        var yy = this.getGlobalY(x, y, la);
        if (isNaN(xx) || isNaN(yy))
            throw new Error();
        if (!isFinite(xx) || !isFinite(yy))
            throw new Error();
        this.updateCurrentPoint(xx, yy);
        this.appendPoint(sb, xx, yy);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    };
    PdfRichGraphicsAdapter.prototype.pathQuadTo = function (x1, y1, x2, y2, la) {
    };
    PdfRichGraphicsAdapter.prototype.pathCurveTo = function (x1, y1, x2, y2, x3, y3, la) {
        var xa = this.getGlobalX(x1, y1, la);
        var ya = this.getGlobalY(x1, y1, la);
        var xb = this.getGlobalX(x2, y2, la);
        var yb = this.getGlobalY(x2, y2, la);
        var xc = this.getGlobalX(x3, y3, la);
        var yc = this.getGlobalY(x3, y3, la);
        if (isNaN(xa) || isNaN(ya))
            throw new Error();
        if (!isFinite(xa) || !isFinite(ya))
            throw new Error();
        if (isNaN(xb) || isNaN(yb))
            throw new Error();
        if (!isFinite(xb) || !isFinite(yb))
            throw new Error();
        if (isNaN(xc) || isNaN(yc))
            throw new Error();
        if (!isFinite(xc) || !isFinite(yc))
            throw new Error();
        this.updateCurrentPoint(xc, yc);
        var sb = new util_1.StringBuilder("bezier1");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xa, ya);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xb, yb);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xc, yc);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    };
    PdfRichGraphicsAdapter.prototype.getGlobalX = function (x, y, la) {
        if (la == null) {
            return x;
        }
        else {
            return la.getGlobalX(x, y);
        }
    };
    PdfRichGraphicsAdapter.prototype.getGlobalY = function (x, y, la) {
        if (la == null) {
            return y;
        }
        else {
            return la.getGlobalY(x, y);
        }
    };
    PdfRichGraphicsAdapter.prototype.appendPoint = function (sb, x, y) {
        this.mGraphics.appendPosition(sb, PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS, x, y);
    };
    PdfRichGraphicsAdapter.prototype.pathClose = function () {
        this.mGraphics.draw("close");
    };
    PdfRichGraphicsAdapter.prototype.pathAdd = function (col, la) {
        if (col.length < 2)
            return;
        var f1 = col[0];
        var f2 = col[1];
        var x1;
        var y1;
        if (RichGraphicsUtil_1.RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX1(), f2.getY1()) || RichGraphicsUtil_1.RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX2(), f2.getY2())) {
            x1 = f1.getX2();
            y1 = f1.getY2();
        }
        else {
            x1 = f1.getX1();
            y1 = f1.getY1();
        }
        this.pathMoveTo(x1, y1, la);
        for (var i = 0; i < col.length; i++) {
            f1 = col[i];
            f2 = col[(i + 1) % col.length];
            var startflip = false;
            if (RichGraphicsUtil_1.RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX1(), f2.getY1()) || RichGraphicsUtil_1.RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX2(), f2.getY2()))
                startflip = true;
            var type = f1.getType();
            if (type == K2DFigureType_1.K2DFigureType.ARC_CW || K2DFigureType_1.K2DFigureType.ARC_CCW) {
                var arc = f1;
                var r = arc.getRadius();
                var cx = arc.getCX();
                var cy = arc.getCY();
                var p1a = RichGraphicsUtil_1.RichGraphicsUtil.getAngle2D(cx, cy, arc.getX1(), arc.getY1());
                var p2a = RichGraphicsUtil_1.RichGraphicsUtil.getAngle2D(cx, cy, arc.getX2(), arc.getY2());
                var angl = p2a - p1a;
                while (angl < 0)
                    angl += 360;
                if (arc.isCw())
                    angl = -(360 - angl);
                if (startflip)
                    angl = -angl;
                this.pathArc(cx, cy, r, r, startflip ? p2a : p1a, angl, la);
            }
            else {
                if (startflip) {
                    this.pathLineTo(f1.getX1(), f1.getY1(), la);
                }
                else {
                    this.pathLineTo(f1.getX2(), f1.getY2(), la);
                }
            }
        }
    };
    PdfRichGraphicsAdapter.prototype.pathArc = function (cx, cy, r, h, startAngle, arcAngle, la) {
        var div = 4;
        var width = r * 2.0;
        var height = h * 2.0;
        var ath = arcAngle * Math.PI / 180.0;
        var theta = ath / div;
        var cos = Math.cos(theta);
        var sin = Math.sin(theta);
        var x = new Array(4);
        var y = new Array(4);
        x[0] = x[1] = 1.0;
        y[0] = 0.0;
        x[3] = cos;
        y[3] = sin;
        x[2] = (8.0 * Math.cos(theta / 2) - (x[0] + 3 * x[1] + x[3])) / 3;
        y[2] = y[3] - (-x[2] + x[3]) / (-Math.tan(theta));
        y[1] = (8.0 * Math.sin(theta / 2) - (y[0] + 3 * y[2] + y[3])) / 3;
        var stheta = startAngle * Math.PI / 180.0;
        var scos = Math.cos(stheta);
        var ssin = Math.sin(stheta);
        var xtmp = new Array(4);
        var ytmp = new Array(4);
        for (var j = 0; j < 4; j++) {
            xtmp[j] = x[j] * scos - y[j] * ssin;
            ytmp[j] = x[j] * ssin + y[j] * scos;
            x[j] = xtmp[j];
            y[j] = ytmp[j];
        }
        var aspect = height / width;
        if (isNaN(aspect))
            aspect = 1;
        this.pathMoveTo(r * x[0] + cx, r * y[0] * aspect + cy, la);
        for (var i = 0; i < div; i++) {
            this.pathCurveTo(r * x[1] + cx, r * y[1] * aspect + cy, r * x[2] + cx, r * y[2] * aspect + cy, r * x[3] + cx, r * y[3] * aspect + cy, la);
            for (var j = 1; j < 4; j++) {
                xtmp[j] = x[j] * cos - y[j] * sin;
                ytmp[j] = x[j] * sin + y[j] * cos;
                x[j] = xtmp[j];
                y[j] = ytmp[j];
            }
        }
    };
    PdfRichGraphicsAdapter.prototype.fillPath = function (lt) {
        this.setLineType(lt);
        var sb = new util_1.StringBuilder("fill");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.mWindingRule);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendColor(sb, PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    };
    PdfRichGraphicsAdapter.prototype.updateCurrentPoint = function (xx, yy) {
        if (this.mCurrentPoint == null)
            this.mCurrentPoint = new KPoint2d_1.KPoint2d(xx, yy);
        else
            this.mCurrentPoint.setXY(xx, yy);
    };
    PdfRichGraphicsAdapter.prototype.drawPath = function (lt) {
        this.setLineType(lt);
        var sb = new util_1.StringBuilder("stroke");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendColor(sb, PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLineWidth(sb);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLinePattern(sb, PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    };
    PdfRichGraphicsAdapter.prototype.shadePath = function (lt, x1, y1, x2, y2, c1, c2, cyclic) {
        var SEP = "_";
        //let fRgb: NumberFormat = new DecimalFormat("000");
        //let fAxis: NumberFormat = new DecimalFormat("00.00");
        var shadename = new util_1.StringBuilder("TYPE2");
        shadename.append(SEP);
        shadename.append(formatRgb(c1.getRed()));
        shadename.append(formatRgb(c1.getGreen()));
        shadename.append(formatRgb(c1.getBlue()));
        shadename.append(formatRgb(c2.getRed()));
        shadename.append(formatRgb(c2.getGreen()));
        shadename.append(formatRgb(c2.getBlue()));
        shadename.append(SEP);
        shadename.append(formatAxis(x1));
        shadename.append(":");
        shadename.append(formatAxis(y1));
        shadename.append(":");
        shadename.append(formatAxis(x2));
        shadename.append(":");
        shadename.append(formatAxis(y2));
        shadename.append(SEP);
        shadename.append(cyclic ? "1" : "0");
        var sb = new util_1.StringBuilder("shadestroke");
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(shadename.toString());
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLineWidth(sb);
        sb.append(PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLinePattern(sb, PdfGraphics_1.PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    };
    //pathArc(xc: number, yc: number, ra: number, startangl: number, angl: number, la: KLocalAxis): void {
    //    this.pathArc(xc, yc, ra, ra, startangl, angl, la);
    //}
    PdfRichGraphicsAdapter.prototype.drawMark = function (x, y, r, type) {
    };
    PdfRichGraphicsAdapter.prototype.getDrawArea = function () {
        return null;
    };
    PdfRichGraphicsAdapter.prototype.getDotPerLen = function () {
        return 0;
    };
    PdfRichGraphicsAdapter.prototype.setQuality = function (quality) {
    };
    PdfRichGraphicsAdapter.prototype.getQuality = function () {
        return false;
    };
    /**
     * ポリゴン等の描画用のバッファのサイズ **/
    PdfRichGraphicsAdapter.BUF_SIZE = 128;
    /**
     * 線の属性が指定されていない（null）の場合の線種 **/
    PdfRichGraphicsAdapter.NULL_LINE = new KLineType_1.KLineType("", [0, 0], 0.1, KLineType_1.KLineType.TYPE_DRAWING);
    PdfRichGraphicsAdapter.EPS = 1.0e-3;
    PdfRichGraphicsAdapter.D2R = Math.PI / 180;
    return PdfRichGraphicsAdapter;
})();
exports.PdfRichGraphicsAdapter = PdfRichGraphicsAdapter;
function formatRgb(n) {
    var s = n.toFixed(0);
    while (s.length < 3)
        s = "0" + s;
    return s;
}
function formatAxis(n) {
    return n.toFixed(2);
}
//# sourceMappingURL=PdfRichGraphicsAdapter.js.map