import { KFont } from '../kwt/KFont';
import { KFill } from '../kwt/KFill';
import { KColor } from '../kwt/KColor';
import { KImage } from '../kwt/KImage';
import { PdfCommand } from './PdfCommand';
import { StringBuilder } from './util';
/**
 * PDF コマンド・グラフィックス。<BR> PDF ファイルへの描画に必要な操作を提供する。
 * @since       2004/01/20
 * @author      fujita2004/08/18  Fujita 多角形グループ用の描画 コマンド、およびパターン、 シェーディングによる描画 コマンドを追加。 2004/09/30  Fujita クリッピング領域の設定、および 解除コマンドを追加。 Copyright (c) 2004 KITec Inc,.. All rights reserved.
**/
export declare class PdfGraphics {
    /**
     * 星型のパターンを表す定数 **/
    static PATTERN_STAR: string;
    /**
     * タイプ2（軸）シェーディングを表す定数 **/
    static SHADING_TYPE2: string;
    /**
     * 線の属性として切れ目のない直線を示す **/
    static SOLID_LINE: number[];
    /**
     * 出力幅に対する空白を文字間隔へ割り当てることを示す **/
    static SPACE_FOR_PITCH: string;
    /**
     * 出力幅に対する空白を文字幅へ割り当てることを示す **/
    static SPACE_FOR_CHAR: string;
    /**
     * テキストコマンドの区切り文字 **/
    static SEPARATOR_TEXT: string;
    /**
     * グラフィックストコマンドの区切り文字 **/
    static SEPARATOR_GRAPHICS: string;
    /**
     * コマンドとパラメータの区切り文字 **/
    static SEPARATOR_COMMAND: string;
    /**
     * テキストコマンドなどでの文字列を囲む引用符 **/
    static STRING_QUOTAION: string;
    /**
     * PDF コマンドの円弧描画コマンド **/
    static CMD_ARC: string;
    /**
     * PDF コマンドのイメージ描画コマンド **/
    static CMD_IMAGE: string;
    /**
     * PDF コマンドのJPEGイメージ描画コマンド **/
    static CMD_JPEG: string;
    /**
     * PDF コマンドのイメージに対するアイテム・コマンド **/
    static CMD_ITEM_IMAGE: string;
    /**
     * PDF コマンドのテキストに対するアイテム・コマンド **/
    static CMD_ITEM_TEXT: string;
    /**
     * PDF コマンドの線描画コマンド **/
    static CMD_LINE: string;
    /**
     * PDF コマンドのリンク描画コマンド **/
    static CMD_LINK: string;
    /**
     * PDF コマンドのポリゴン描画コマンド **/
    static CMD_POLYGON: string;
    /**
     * PDF コマンドのポリライン描画コマンド **/
    static CMD_POLYLINE: string;
    /**
     * PDF コマンドの矩形描画コマンド **/
    static CMD_RECT: string;
    /**
     * PDF コマンドのテキスト描画コマンド **/
    static CMD_TEXT: string;
    /**
     * PDF コマンドのグループポリゴン描画コマンド **/
    static CMD_GROUP_POLYGON: string;
    /**
     * PDF コマンドのパターン描画コマンド **/
    static CMD_PATTERN: string;
    /**
     * PDF コマンドのシェーディング描画コマンド **/
    static CMD_SHADING: string;
    /**
     * PDF コマンドのクリッピング設定コマンド **/
    static CMD_CLIP: string;
    /**
     * PDF コマンドのクリッピング解除コマンド **/
    static CMD_UNCLIP: string;
    /**
     * PDF コマンドのイメージ描画コマンド **/
    static CMD_3D: string;
    /**
     * イタリック体の文字の傾きの **/
    static FONT_SKEW: number;
    /**
     * ポイントをミリメートルへ変換する場合の比率 （1 pt = 1 / 2.8346 mm）**/
    static PT_TO_MM: number;
    /**
     * PDF コマンドでの矩形などの外形のみを描くことを表す **/
    static DRAW: string;
    /**
     * PDF コマンドでの矩形などを塗りつぶすことを表す **/
    static FILL: string;
    /**
     * PDF コマンドでの装飾のない通常のフォントを表す **/
    static PLAIN: string;
    /**
     * PDF コマンドでのボールド体のフォントを表す **/
    static BOLD: string;
    /**
     * PDF コマンドでのフォントを埋め込むことを表す **/
    static EMBED: string;
    /**
     * PDF コマンドでのフォント出力幅として 指定フォントの幅をそのまま使用することを表す**/
    static TEXT_WIDTH: number;
    /**
     * イタリック体の文字の傾き **/
    mFontSkew: number;
    /**
     * このグラフィックスの親 PdfCommand **/
    mPdfCmd: PdfCommand;
    /**
     * 描画色 **/
    mColor: KFill;
    /**
     * 現在の線の太さ **/
    mLineWidth: number;
    /**
     * 現在の線のパターン **/
    mLinePattern: number[];
    /**
     * 現在のフォントの名前 **/
    mFontName: string;
    /**
     * 現在のフォントのスタイル **/
    mFontStyle: number;
    /**
     * 現在のフォントのサイズ **/
    mFontSize: number;
    /**
     * PDF ファイルへの出力時、フォントの埋め込むか否か表すフラグ **/
    mIsFontEmbedded: boolean;
    /**
     * 現在の原点の x 成分 **/
    mOriginX: number;
    /**
     * 現在の原点の y 成分 **/
    mOriginY: number;
    /**
     * 最後に設定したイメージ名 **/
    mLastImageName: string;
    /**
     * 描画位置などのフォーマット **/
    /**
     * 指定された親を持つ新たな PdfCommandGraphics を作成する。
     * @param pdfcmd  PDF コマンド
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    constructor(pdfcmd: PdfCommand);
    /**
     * 描画色、フォントなどを初期状態へ設定する。
     * @since   2004/01/21
     * @author  fujita
     * @version 2004/01/21
    **/
    reset(): void;
    /**
     * 現在のページへの描画を終了する。
     * @since   2004/01/21
     * @author  fujita
     * @version 2004/01/21
    **/
    dispose(): void;
    /**
     * PDF ページ上の原点を現在の座標系の点 (x, y) に変換する。
     * @param x   x 座標
     * @param y   y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    translate(x: number, y: number): void;
    /**
     * 指定した座標が指定した矩形に現在のクリップを設定する。
     * @param x       新しいクリップ矩形の x 座標
     * @param y       新しいクリップ矩形の y 座標
     * @param width   新しいクリップ矩形の幅
     * @param height  新しいクリップ矩形の高さ
     * @see #releaseClip()
     * @since       2004/10/06
     * @author      fujita
     * @version 2004/10/06
    **/
    setClip(x: number, y: number, width: number, height: number): void;
    /**
     * 現在のクリッピング領域を解除し、出力先として 指定されている領域全体を描画領域とする。
     * @see #setClip(int,int,int,int)
     * @see #setClip(double,double,double,double)
     * @since       2004/09/30
     * @author      fujita
     * @version 2004/09/30
    **/
    releaseClip(): void;
    /**
     * @see java.awt.Graphics#getColor()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    getColor(): KColor;
    /**
     * @see java.awt.Graphics#setColor(Color)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    setColor(c: KFill): void;
    /**
     * @see java.awt.Graphics#getFont()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    getFont(): KFont;
    /**
     * 現在のフォント名を返す。
     * @return      フォント名
     * @see #setFontName(String)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    getFontName(): string;
    /**
     * 現在のフォント・スタイルを返す。
     * @return      フォント・スタイル
     * @see #setFontStyle(int)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    getFontStyle(): number;
    /**
     * 現在のフォント・サイズを返す。
     * @return      フォント・サイズ
     * @see #setFontSize(double)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    getFontSize(): number;
    /**
     * PDF ファイルへの出力にフォントを埋め込むか否かを判定する。 デフォルトは、フォントを埋め込まない。
     * @return  フォントを埋め込む場合、true。埋め込まない場合は、false。
     * @see #setFontEmbed(boolean)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    isFontEmbedded(): boolean;
    /**
     * @see java.awt.Graphics#setFont(Font)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    setFont(font: KFont): void;
    /**
     * 指定されたフォント名、スタイル、およびサイズを持つ フォントを設定する。
     * @param name    フォント名
     * @param style   フォント・スタイル
     * @param size    フォント・サイズ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    setFontParam(name: string, style: number, size: number): void;
    /**
     * 指定されたフォント名を持つフォントを設定する。
     * @param name    フォント名
     * @see #getFontName()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    setFontName(name: string): void;
    /**
     * 指定されたスタイルを持つフォントを設定する。
     * @param style   フォント・スタイル
     * @see #getFontStyle()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    setFontStyle(style: number): void;
    /**
     * 指定されたサイズを持つフォントを設定する。
     * @param size    フォント・サイズ
     * @see #getFontSize()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    setFontSize(size: number): void;
    /**
     * PDF ファイルへの出力にフォントを埋め込むか否かを設定する。<BR> デフォルトは、フォントを埋め込まない。
     * @param embed   フォントを埋め込む場合、true。埋め込まない場合は、false。
     * @see #isFontEmbedded()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    setFontEmbed(embed: boolean): void;
    /**
     * 描画に用いる線のパターンを設定する。<BR> double[]{ 0, 0 } の場合は、切れ目のない線となる。
     * @param dash    線のパターンとして実線と空白部分の長さを指定する
     * @see #getLinePattern()
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    setLinePattern(dash: number[]): void;
    /**
     * 現在の線のパターンを取得する。
     * @return      線のパターン
     * @see     #setLinePattern(final double[])
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    getLinePattern(): number[];
    /**
     * 描画に用いる線の太さを設定する。
     * @param width   線の太さ
     * @see #getLineWidth()
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    setLineWidth(width: number): void;
    /**
     * 現在の線の太さを返す。
     * @return      線の太さ
     * @see     #setLineWidth(final double)
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    getLineWidth(): number;
    /**
     * 最後に描画したイメージ名を返す。
     * @return      イメージ名
    **/
    getLastImageName(): string;
    /**
     * @see java.awt.Graphics#drawArc(int,int,int,int,int,int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    drawArc(x: number, y: number, width: number, height: number, startAngle: number, arcAngle: number): void;
    /**
     * @see java.awt.Graphics#fillArc(int,int,int,int,int,int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    fillArc(x: number, y: number, width: number, height: number, startAngle: number, arcAngle: number): void;
    /**
     * 指定された矩形をカバーする円弧または楕円弧の輪郭を PDF コマンドとして出力する。
     * @param x           孤の左上隅の x 座標
     * @param y           孤の左上隅の y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param startAngle  開始角度[°]
     * @param arcAngle    開始角度に対する弧の展開角度の大きさ[°]
     * @param fill        弧の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    drawArc_(x: number, y: number, width: number, height: number, startAngle: number, arcAngle: number, fill: boolean): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた 多角形の集合を指定されたパターンで塗りつぶす PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x座標の配列
     * @param yPoints     すべての多角形の y座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param name        パターン名
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    drawPatternedGroupPolygon(name: string, xPoints: number[], yPoints: number[], nPoints: number[]): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた 多角形の集合を指定されたシェーディングで 塗りつぶす PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x座標の配列
     * @param yPoints     すべての多角形の y座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param name        シェーディング名
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    drawShadingGroupPolygon(name: string, xPoints: number[], yPoints: number[], nPoints: number[]): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形の 集合を PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x 座標の配列
     * @param yPoints     すべての多角形の y 座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param fill        多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    drawGroupPolygon(xPoints: number[], yPoints: number[], nPoints: number[], fill: boolean): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形の 集合を PDF コマンドとして、指定された sb へ出力する。
     * @param sb          PDF コマンドを出力先となるオブジェクト
     * @param xPoints     すべての多角形の x 座標の配列
     * @param yPoints     すべての多角形の y 座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param fill        多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    private drawGroupPolygon_(sb, xPoints, yPoints, nPoints, fill);
    /**
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    drawImagePosition(img: KImage, dx1: number, dy1: number, dx2: number, dy2: number, sx1: number, sy1: number, sx2: number, sy2: number): boolean;
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を出力する。<BR>
     * イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画されるイメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    drawImageRect(img: KImage, x: number, y: number, width: number, height: number): boolean;
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたファイルをイメージとして出力する。<BR> イメージは、出力される領域の（x, y）に その左上隅を合わせて出力される。
     * @param file    描画されるイメージのファイル名
     * @param x       x 座標
     * @param y       y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param theta   出力するイメージの傾き
     * @return      ファイル名が null の場合、false。そうでない場合は、true
     * @since       2004/11/29
     * @author      Kawae
     * @version 2004/11/29
    **/
    private drawImage_(file, x, y, width, height, theta);
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたファイルをイメージとして出力する。<BR> イメージは、出力される領域の（x, y）に その左上隅を合わせて出力される。
     * @param file    描画されるイメージのファイル名
     * @param x       x 座標
     * @param y       y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param theta   出力するイメージの傾き
     * @return      ファイル名が null の場合、false。そうでない場合は、true
     * @since       2004/11/29
     * @author      Kawae
     * @version 2004/11/29
    **/
    drawImageAsJpeg(file: string, x: number, y: number, width: number, height: number, theta: number): boolean;
    /**
     * 指定されたイメージの現在利用可能な部分だけを PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画される指定イメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    private drawImage_1(img, x, y);
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img     描画される指定イメージ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    private drawImage_2(img, x, y, w, h, ow, oh, theta);
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> jpegイメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param pixels  描画される指定イメージのピクセルデータ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/11/29
     * @author      kawae
     * @version 2004/11/29
    **/
    drawImageAsJpegPixels(pixels: number[], x: number, y: number, w: number, h: number, ow: number, oh: number, theta: number): boolean;
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param pixels  描画される指定イメージのピクセルデータ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    private drawImagePixels(pixels, x, y, w, h, ow, oh, theta);
    private drawImageCommand(command, name, x, y, w, h, ow, oh, theta);
    /**
     * 指定されたイメージを、指定されたフォーム上の項目として出力する。
     * @param file        描画されるイメージのファイル名
     * @param formatName  フォーム名
     * @param ItemName    項目名
     * @since       2004/03/03
     * @author      fujita
     * @version 2004/03/03
    **/
    drawItemImage(file: string, formatName: string, ItemName: string): void;
    /**
     * 指定された文字列を、指定されたフォーム上の項目として出力する。
     * @param str         出力する文字列
     * @param formatName  フォーム名
     * @param ItemName    項目名
     * @since       2004/02/20
     * @author      fujita
     * @version 2004/02/20
    **/
    drawItemString(str: string, formatName: string, ItemName: string): void;
    /**
     * 指定された矩形の内部に指定された url へのリンクを出力する。<BR>
     * @param url         リンク先の URL
     * @param x1          矩形の左上の x 座標
     * @param y1          矩形の左上の y 座標
     * @param x2          矩形の右下の x 座標
     * @param y2          矩形の右下の y 座標
     * @param boderWidth  矩形を囲む線の幅。矩形を透明とする場合は、0
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    drawLink(url: string, x1: number, y1: number, x2: number, y2: number, boderWidth: number): void;
    /**
     * PDF ページ上の点 (x1, y1) と点 (x2, y2) との間に 現在の色を使って線を出力する。
     * @param x1  最初の点の x 座標
     * @param y1  最初の点の y 座標
     * @param x2  2 番目の点の x 座標
     * @param y2  2 番目の点の y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    drawLine(x1: number, y1: number, x2: number, y2: number): void;
    /**
     * @see java.awt.Graphics#drawPolygon(int[],int[],int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    drawPolyline(xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * @see java.awt.Graphics#drawPolygon(int[],int[],int)
     * @since       2004/01/20
     * @author      fujita
     * @version 2004/01/20
    **/
    drawPolygon(xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * @see java.awt.Graphics#fillPolygon(int[],int[],int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    fillPolygon(xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を 指定されたパターンで塗りつぶす PDF コマンドを 出力する。
     * @param name    パターン名
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    drawPatternedPolygon(name: string, xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を 指定されたシェーディングで塗りつぶす PDF コマンドを 出力する。
     * @param name    シェーディング名
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    drawShadingPolygon(name: string, xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を PDF コマンドとして出力する。
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/08/18  Fujita多角形の PDF コマンド 出力部分を別メソッドへ 移行。
    **/
    private drawPolygon_(xPoints, yPoints, nPoints, fill);
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を PDF コマンドとして、指定された sb へ出力する。
     * @param sb      PDF コマンドの出力先となるオブジェクト
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    private drawPolygon__(sb, xPoints, yPoints, nPoints, fill);
    /**
     * 指定された矩形の輪郭を出力する。
     * @param x       矩形の左上の x 座標
     * @param y       矩形の左上の y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param fill    矩形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    drawRect(x: number, y: number, width: number, height: number, fill: boolean): void;
    /**
     * 指定された矩形を、PDF 用の矩形領域の形式にて 指定された buffer へ出力する。
     * @param x           矩形の左上の x 座標
     * @param y           矩形の左上の y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param buffer      データの出力先
     * @param separator   データの区切り文字
     * @since       2004/09/30
     * @author      fujita
     * @version 2004/09/30
    **/
    drawRect_(x: number, y: number, width: number, height: number, buffer: StringBuilder, separator: string): void;
    /**
     * 丸いコーナー付きの輪郭の矩形を、現在の色を使用して出力する。<BR> 出力される矩形は、左端と右端がそれぞれ x と x + width、 上端と下端がそれぞれ y と y + height で指定される。
     * @param x           描画される矩形の x 座標
     * @param y           描画される矩形の y 座標
     * @param width       描画される矩形の幅
     * @param height      描画される矩形の高さ
     * @param arcWidth    4 隅の弧の水平方向の直径
     * @param arcHeight   4 隅の弧の垂直方向の直径
     * @param fill        矩形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/02/16
     * @author      fujita
     * @version 2004/02/16
    **/
    drawRoundRect(x: number, y: number, width: number, height: number, arcWidth: number, arcHeight: number, isFill: boolean): void;
    /**
     * 現在のフォントと色を使い、指定された文字列によって 指定されたテキストを出力する。<BR> 指定された配置方法により、文字のベースラインは、 位置 (x, y)に対して、左端、または中央となる。
     * @param str     出力する文字列
     * @param x       x 座標
     * @param y       y 座標
     * @param center  位置（x, y）を文字の中央とする場合、true。左端とする場合は、false
     * @param rotate  文字の回転角
     * @param width   テキストの出力幅
     * @param type    出力幅に対する空白の割り当て方法
     * @since       2004/02/09
     * @author      fujita
     * @version 2004/02/09
    **/
    drawString(str: string, x: number, y: number, center: boolean, rotate: number, width: number, type: string): void;
    /**
     * 指定された点（x, y）を PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param x           x 座標
     * @param y           y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendPosition(sb: StringBuilder, separator: string, x: number, y: number): void;
    /**
     * 指定された x 座標と y 座標の配列を PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param xPoints     x 座標の配列
     * @param yPoints     y 座標の配列
     * @param nPoint      点の総数
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendPoints(sb: StringBuilder, separator: string, xPoints: number[], yPoints: number[], nPoints: number): void;
    /**
     * 指定された幅と高さを PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param width       幅
     * @param height      高さ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendSize(sb: StringBuilder, separator: string, width: number, height: number): void;
    /**
     * 現在の色を PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendColor(sb: StringBuilder, separator: string): void;
    /**
     * 現在のフォントの名前とサイズを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendFontNameAndSize(sb: StringBuilder, separator: string): void;
    /**
     * 指定された文字列の幅を PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param width   文字列の幅
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendStringWidth(sb: StringBuilder, width: number): void;
    /**
     * 現在の線の太さを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendLineWidth(sb: StringBuilder): void;
    /**
     * 現在の線のパターンを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendLinePattern(sb: StringBuilder, separator: string): void;
    /**
     * 指定された幅を PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param width   幅
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendWidth(sb: StringBuilder, width: number): void;
    /**
     * 指定された高さを PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param height  高さ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    appendHeight(sb: StringBuilder, height: number): void;
    /**
     * 指定された数値（座標、幅、高さなど）を PDF コマンド用にフォーマットする。
     * @param number  変換する数値
     * @return      PDF コマンド用に変換された数値
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    format(number: number): string;
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画されるイメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    draw3D(idtf: number[], x1: number, y1: number, x2: number, y2: number, fov: number): boolean;
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param name    イメージデータの名前
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/13
     * @author      fujita
     * @version 2004/02/13
    **/
    draw3DName(name: string, x1: number, y1: number, x2: number, y2: number, fov: number): boolean;
    /**
     * テストメソッド
     * @param s
     * @author kawae
     * @since 2006/02/03
    **/
    draw(s: string): void;
}
