var KDimension_1 = require('../kwt/KDimension');
//import { AbstFile } from '../io/AbstFile'
//import { PrintStream } from '../../java/io/PrintStream'
var PdfGraphics_1 = require('./PdfGraphics');
var util_1 = require('./util');
//module option.pdf
/**
 * PDF コマンドを作成する。<BR> 作成方法は、<BR>　1.当クラスの生成する。<BR>
 * 　2.PDF 出力用の Graphics を取得する。（新たなページの開始）<BR>
 * 　3.Graphics のメソッドを使用しデータを書き込む<BR>
 * 　4.Graphics を破棄する。（ページの終了）<BR>
 * 　5.PDF 出力を終了する。<BR>以上で、PDF コマンドが作成される。<BR> 必要に応じて、2 ～ 3 を繰り返し、複数ページへの出力を行う。<BR> 作成された PDF コマンドデータは、getCommand() にて取得する。<BR> デフォルトの用紙サイズは、A4 の縦方向。変更する場合は、 setPageInfo(String, boolean) を実行する。
 * @since 　2004/01/20
 * @author　fujita
 * @version 2004/01/20Copyright (c) 2004 KITec Inc,.. All rights reserved.
**/
var PdfCommand = (function () {
    /**
     * 書き出し文字コード(060110) **/
    //mCode: string;
    /**
     * PDF コマンドを作成するための PdfCommand を構築する。
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    function PdfCommand(fileName /*, code: string*/) {
        /**
         * ユーザ座標の y 軸の方向 **/
        this.mUserCoordDirY = PdfCommand.TOP_TO_BOTTOM;
        this.count = 0;
        this.mPdfGraphics = new PdfGraphics_1.PdfGraphics(this);
        //this.mCode = code;
        this.openPdf();
        this.openFile(fileName);
        this.setPageInfomation(PdfCommand.A4, PdfCommand.PORTRAIT, PdfCommand.DEFAULT_DPI);
        this.createViewport(PdfCommand.DEFAULT_VIEWPORT_MAX);
        this.count = 0;
    }
    /**
     * 新しい PDF コマンドを出力用に開く。<BR>
     * @see #closePdf()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfCommand.prototype.openPdf = function () {
        if (this.mIsPDFOpened) {
            throw new Error("PDF is alread opened");
        }
        //this.mOutFile = new AbstFile();
        //if (this.mCode == null) {
        //    this.mOutStream = new PrintStream(this.mOutFile.openOutputStream(""), true, "UTF-8");
        //} else {
        //    this.mOutStream = new PrintStream(this.mOutFile.openOutputStream(""), true, this.mCode);
        //}
        this.mOutStream = new util_1.MyPrintStream();
        this.mOutStream.println("pdfstart");
        this.mTable = new Map();
        this.mPageCount = 0;
        this.mImageCount = 0;
        this.mFileName = null;
        this.mIsFileOpened = false;
        this.mIsPageStarted = false;
        this.mIsPDFOpened = true;
        this.mFormats = new Array();
    };
    /**
     * 現在、出力中の PDF コマンドを閉じる。
     * @see #openPdf()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfCommand.prototype.closePdf = function () {
        if (!this.mIsPDFOpened) {
            return;
        }
        this.mOutStream.println("pdfend");
        this.mOutStream.close();
        this.mIsPDFOpened = false;
    };
    ///**
    // * ページのサイズなどのページに関する情報を設定する。
    // * @param pageSize	出力するページのサイズ
    // * @param direction	ページを使用する方向（縦、または横）
    // * @since   2004/01/20
    // * @author  fujita
    // * @version 2004/01/20
    //**/
    //setPageInfo(pageSize: number, direction: number): void {
    //    this.setPageInfo(pageSize, direction, PdfCommand.DEFAULT_DPI);
    //}
    /**
     * ページのサイズなどのページに関する情報を設定する。<BR> このとき、ユーザ座標は、ページサイズと同じサイズに設定される。
     * @param pageSize	出力するページのサイズ
     * @param direction	ページを使用する方向（縦、または横）
     * @param dpi			1 インチ当たりのドット数
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.setPageInfo = function (pageSize, direction, dpi) {
        if (dpi === void 0) { dpi = PdfCommand.DEFAULT_DPI; }
        this.setPageInfomation(pageSize, direction, dpi);
        this.setUserCoordinate(0, 0, this.mPageWidth, this.mPageHeight);
    };
    /**
     * ページのサイズなどのページに関する情報設定の実処理を行う。
     * @param pageSize	出力するページのサイズ
     * @param direction	ページを使用する方向（縦、または横）
     * @param dpi			1 インチ当たりのドット数
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.setPageInfomation = function (pageSize, direction, dpi) {
        if (pageSize < 0 || pageSize >= PdfCommand.MEDIA_SIZE_INDEXES.length) {
            throw new Error("illegal page size");
        }
        if (direction != PdfCommand.PORTRAIT && direction != PdfCommand.LANDSCAPE) {
            throw new Error("illegal page direciton");
        }
        if (dpi <= 0) {
            throw new Error("illegal dpi");
        }
        this.mPageSize = pageSize;
        this.mDirection = direction;
        this.mDpi = dpi;
        var wh = PdfCommand.MEDIA_SIZE[this.mPageSize];
        if (this.mDirection == PdfCommand.PORTRAIT) {
            this.setPageDimension(wh[0], wh[1]);
        }
        else {
            this.setPageDimension(wh[1], wh[0]);
        }
    };
    /**
     * 現在、出力用に開かれている PDF ファイルへの出力を終了し、 PDF ファイルの作成を行う。<BR>
     * @return		ファイルが正常に作成された場合、作成されたファイル名。そうでない場合は、null
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.end = function () {
        this.endPage();
        this.closeFile();
        this.closePdf();
    };
    /**
     * 指定されたファイルを新たな PDF の出力ファイルとして開く。<BR> 事前に、openPdf を実行し、PDF コマンドが開く必要がある。
     * @param fileName	PDF を出力するファイル名
     * @see #closeFile()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfCommand.prototype.openFile = function (fileName) {
        if (fileName == null) {
            throw new Error("file name is null");
        }
        if (!this.mIsPDFOpened) {
            throw new Error("PDF is not opend");
        }
        if (this.mIsFileOpened) {
            throw new Error("file is already opend");
        }
        this.mFileName = fileName;
        this.mOutStream.println("filestart," + this.mFileName);
        this.mPageCount = 0;
        this.mIsFileOpened = true;
    };
    /**
     * 現在の出力ファイルを閉じる。<BR> このとき、出力中のページもあわせて閉じる。
     * @see #openFile(String)
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.closeFile = function () {
        if (!this.mIsFileOpened) {
            return;
        }
        this.mOutStream.println("fileend," + this.mFileName);
        this.mIsFileOpened = false;
    };
    /**
     * 指定されたページのサイズ、および印刷方向で新しいページを作成する。
     * @param pageSize	"A4", "A3" など出力するページのサイズ
     * @param direction	ページの使用方向
     * @see #endPage()
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.startPage = function (pageSize, direction) {
        if (!this.mIsFileOpened) {
            throw new Error("file is not opend");
        }
        if (this.mIsPageStarted) {
            throw new Error("page is already opend");
        }
        var sb = "pagestart,";
        sb += this.mPageCount;
        this.mOutStream.println(sb);
        sb = "pagesize,";
        sb += pageSize;
        sb += ",";
        sb += direction;
        if (this.mDpi != PdfCommand.DEFAULT_DPI) {
            sb += ",";
            sb += this.mDpi;
        }
        this.mOutStream.println(sb);
        this.setFormat();
        this.mIsPageStarted = true;
    };
    /**
     * 現在の出力中のページを閉じる。
     * @see #startPage()
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.endPage = function () {
        if (!this.mIsPageStarted) {
            return;
        }
        this.mOutStream.println("pageend," + this.mPageCount++);
        this.mIsPageStarted = false;
        this.resetViewport();
        this.setUserCoordinate(0, 0, this.mPageWidth, this.mPageHeight);
    };
    /**
     * 指定されたフォーマットを、ページのフォーマットとして設定する。
    
     * formatが省略された場合、現在、設定されているフォーマットをもとに、 PDF フォーマットコマンドを作成する。<BR>
     * フォーマットが null の場合、フォーマットコマンドは 作成されない。
     * @param format	フォーマット。null を指定した場合は、以前、設定されていたフォーマットは クリアされる。
     * @since		2004/02/05
     * @author		fujita
     * @version	2004/02/05
    **/
    PdfCommand.prototype.setFormat = function (format) {
        if (format) {
            if (this.mFormats.length > 0) {
                this.mFormats.length = 0;
            }
            this.addFormat(format);
        }
        else {
            for (var i = 0; i < this.mFormats.length; i++) {
                var f = this.mFormats[i];
                this.mOutStream.println("format," + f);
            }
        }
    };
    /**
     * 指定されたフォーマットを、現在のフォーマットに追加する。
     * @param format	フォーマット。null を指定した場合は、フォーマットは設定されない。
     * @since		2004/02/05
     * @author		fujita
     * @version	2004/02/05
    **/
    PdfCommand.prototype.addFormat = function (format) {
        if (format != null && format.length > 0) {
            this.mFormats.push(format);
        }
    };
    /**
     * PDF コマンドを出力するための Graphics を返す。
     * @return		PDF コマンド出力用 Graphics
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.getGraphics = function () {
        if (!this.mIsPageStarted) {
            this.startPage(PdfCommand.MEDIA_SIZE_INDEXES[this.mPageSize], this.mDirection);
        }
        return this.mPdfGraphics;
    };
    /**
     * 現在、書き込まれた PDF コマンドデータを返す。
     * @return		PDF コマンドデータ
     * @since		2004/01/20
     * @author		fujita
     * @version	2004/01/20
    **/
    PdfCommand.prototype.getData = function () {
        this.mTable.set("DATA", this.mOutStream.getData());
        return this.mTable;
    };
    /**
     * 現在の出力対象となるイメージ名を返す。<BR> この名前を、イメージ出力時のイメージ名として設定する。
     * @return	イメージ名
     * @since   2004/01/21
     * @author  fujita
     * @version 2004/01/21
    **/
    PdfCommand.prototype.getNextImageName = function () {
        return this.mImageCount.toString();
    };
    /**
     * ページの大きさをミリ単位で返す。
     * @return		ページの大きさ
     * @since		2004/01/23
     * @author		fujita
     * @version	2004/01/23
    **/
    PdfCommand.prototype.getPageDimension = function () {
        return new KDimension_1.KDimension(this.mPageWidth, this.mPageHeight);
    };
    /**
     * 指定された最大数でビューポートを、新たに作成する。
     * @param max	ビューポートの最大数
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.createViewport = function (max) {
        this.mViewportMax = max;
        this.mViewMinX = new Array(this.mViewportMax);
        this.mViewMinY = new Array(this.mViewportMax);
        this.mViewMaxX = new Array(this.mViewportMax);
        this.mViewMaxY = new Array(this.mViewportMax);
        this.resetViewport();
    };
    /**
     * 現在のすべてのビューポートを初期状態へ戻す。
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.resetViewport = function () {
        this.mCurrentViewport = 0;
        this.mViewMinX[0] = 0;
        this.mViewMinY[0] = 0;
        this.mViewMaxX[0] = 1;
        this.mViewMaxY[0] = 1;
        this.mViewportCount = 1;
    };
    /**
     * 指定されたサイズを、現在のビューポートのサイズとして設定する。
     * @param x1	ビューポートの x 成分の最小値
     * @param y1	ビューポートの y 成分の最小値
     * @param x2	ビューポートの x 成分の最大値
     * @param y2	ビューポートの y 成分の最大値
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.setViewport = function (x1, y1, x2, y2) {
        this.mViewMinX[this.mViewportCount] = x1;
        this.mViewMinY[this.mViewportCount] = y1;
        this.mViewMaxX[this.mViewportCount] = x2;
        this.mViewMaxY[this.mViewportCount] = y2;
        this.mCurrentViewport = this.mViewportCount;
        this.mViewportCount++;
    };
    /**
     * 指定されたサイズを、新たなユーザ座標のサイズとして設定する。
     * @param x1	ユーザ座標の x 成分の最小値
     * @param y1	ユーザ座標の y 成分の最小値
     * @param x2	ユーザ座標の x 成分の最大値
     * @param y2	ユーザ座標の y 成分の最大値
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.setUserCoordinate = function (x1, y1, x2, y2) {
        this.mUserMinX = x1;
        this.mUserMinY = y1;
        this.mUserMaxX = x2;
        this.mUserMaxY = y2;
    };
    /**
     * ユーザ座標の y 軸の伸びる方向を設定する。
     * @param dir		上から下へ伸びる場合、TOP_TO_BOTTOM。下から上へ伸びる場合は、BOTTOM_TOP_TO
     * @see #getUserCoordinateDirectionY()
     * @since		2004/02/09
     * @author		fujita
     * @version	2004/02/09
    **/
    PdfCommand.prototype.setUserCoordinateDirectionY = function (dir) {
        this.mUserCoordDirY = dir;
    };
    /**
     * 現在のユーザ座標の y 軸の伸びる方向を返す。
     * @return		dir		上から下へ伸びる場合、TOP_TO_BOTTOM。下から上へ伸びる場合は、BOTTOM_TOP_TO
     * @see #setUserCoordinateDirectionY(int)
     * @since		2004/02/09
     * @author		fujita
     * @version	2004/02/09
    **/
    PdfCommand.prototype.getUserCoordinateDirectionY = function () {
        return this.mUserCoordDirY;
    };
    /**
     * 指定されたサイズを、新たな PDF の出力ページのサイズとして設定する。
     * @param x1	PDF の出力ページの x 成分の最小値
     * @param y1	PDF の出力ページの y 成分の最小値
     * @param x2	PDF の出力ページの x 成分の最大値
     * @param y2	PDF の出力ページの y 成分の最大値
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.setPageDimension = function (width, height) {
        this.mPageWidth = width;
        this.mPageHeight = height;
    };
    /**
     * 指定されたユーザ座標の幅をページ座標の幅へ変換する。
     * @param w	ユーザ座標の幅
     * @return		ページ座標の幅
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userWidthToPageWidth = function (w) {
        var vw = this.mViewMaxX[this.mCurrentViewport] - this.mViewMinX[this.mCurrentViewport];
        return Math.abs(vw * this.mPageWidth * w / (this.mUserMaxX - this.mUserMinX));
    };
    /**
     * ユーザ座標の高さをページ座標の高さへ変換する。
     * @param h	ユーザ座標の高さ
     * @return		ページ座標の高さ
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userHeightToPageHeight = function (h) {
        var vh = this.mViewMaxY[this.mCurrentViewport] - this.mViewMinY[this.mCurrentViewport];
        return Math.abs(vh * this.mPageHeight * h / (this.mUserMaxY - this.mUserMinY));
    };
    /**
     * ユーザ座標の x 成分をページ座標の x 成分へ変換する。
     * @param x	ユーザ座標の x 成分
     * @return		ページ座標の x 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userXToPageX = function (x) {
        return this.viewXToPageX(this.userXToViewX(x));
    };
    /**
     * ユーザ座標の y 成分をページ座標の y 成分へ変換する。
     * @param y	ユーザ座標の y 成分
     * @return		ページ座標の y 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userYToPageY = function (y) {
        return this.viewYToPageY(this.userYToViewY(y));
    };
    /**
     * ユーザ座標の x 成分をビューポートの x 成分へ変換する。
     * @param x	ユーザ座標の x 成分
     * @return		ビューポートの x 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userXToViewX = function (x) {
        var s = (x - this.mUserMinX) / (this.mUserMaxX - this.mUserMinX);
        return this.mViewMinX[this.mCurrentViewport] + s * (this.mViewMaxX[this.mCurrentViewport] - this.mViewMinX[this.mCurrentViewport]);
    };
    /**
     * ユーザ座標の y 成分をビューポートの y 成分へ変換する。
     * @param y	ユーザ座標の y 成分
     * @return		ビューポートの y 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.userYToViewY = function (y) {
        if (this.getUserCoordinateDirectionY() == PdfCommand.TOP_TO_BOTTOM) {
            y = this.mUserMaxY - y;
        }
        var s = (y - this.mUserMinY) / (this.mUserMaxY - this.mUserMinY);
        return this.mViewMinY[this.mCurrentViewport] + s * (this.mViewMaxY[this.mCurrentViewport] - this.mViewMinY[this.mCurrentViewport]);
    };
    /**
     * ビューポートの x 成分をページ座標の x 成分へ変換する。
     * @param x	ビューポートの x 成分
     * @return		ページ座標の x 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.viewXToPageX = function (x) {
        return this.mPageWidth * x;
    };
    /**
     * ビューポートの y 成分をページ座標の y 成分へ変換する。
     * @param y	ビューポートの y 成分
     * @return		ページ座標の y 成分
     * @since		2004/02/04
     * @author		fujita
     * @version	2004/02/04
    **/
    PdfCommand.prototype.viewYToPageY = function (y) {
        return this.mPageHeight * y;
    };
    /**
     * 指定されたデータを PDF コマンドとして書き込む。
     * @param data	PDF コマンドデータ
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfCommand.prototype.draw = function (data) {
        if (!this.canWrite()) {
            throw new Error("graphics is not ready");
        }
        if (typeof data === "string") {
            this.mOutStream.println(data);
            this.count++;
            return null;
        }
        else {
            var token = this.mImageCount.toString();
            this.mTable.set("im" + token, data);
            this.mImageCount++;
            return token;
        }
    };
    ///**
    // * 指定されたデータをイメージとして書き込む。
    // * @param name	イメージの名前
    // * @param data	イメージのデータ
    // * @since   2004/01/20
    // * @author  fujita
    // * @version 2004/01/20
    //**/
    //draw(data: number[]): string {
    //    if (!this.canWrite()) {
    //        throw new Error("graphics is not ready");
    //    }
    //    let token = this.mImageCount.toString();
    //    this.mTable.put("im" + token, data);
    //    this.mImageCount++;
    //    return token;
    //}
    ///**
    // * 指定されたデータをイメージとして書き込む。
    // * @param name	イメージの名前
    // * @param data	イメージのデータ
    // * @since   2004/11/29
    // * @author  kawae
    // * @version 2004/11/29
    //**/
    //draw(data: number[]): string {
    //    if (!this.canWrite()) {
    //        throw new Error("graphics is not ready");
    //    }
    //    let token = this.mImageCount.toString();
    //    this.mTable.put("im" + token, data);
    //    this.mImageCount++;
    //    return token;
    //}
    /**
     * PDF コマンドへの出力できるか判定する。
     * @return		ファイルのオープン等が正常に行われ出力できる状態の場合、true。出力できない場合は、false
     * @since		2004/01/30
     * @author		fujita
     * @version	2004/01/30
    **/
    PdfCommand.prototype.canWrite = function () {
        if (!this.mIsPDFOpened || !this.mIsFileOpened) {
            return false;
        }
        if (!this.mIsPageStarted) {
            return false;
        }
        return true;
    };
    /**
     * PDF ページの印刷先として A1 用紙を使用することを示す **/
    PdfCommand.A1 = 0;
    /**
     * PDF ページの印刷先として A2 用紙を使用することを示す **/
    PdfCommand.A2 = 1;
    /**
     * PDF ページの印刷先として A3 用紙を使用することを示す **/
    PdfCommand.A3 = 2;
    /**
     * PDF ページの印刷先として A4 用紙を使用することを示す **/
    PdfCommand.A4 = 3;
    /**
     * PDF ページの印刷先として A5 用紙を使用することを示す **/
    PdfCommand.A5 = 4;
    /**
     * PDF ページの印刷先として A6 用紙を使用することを示す **/
    PdfCommand.A6 = 5;
    /**
     * PDF ページの印刷先として A7 用紙を使用することを示す **/
    PdfCommand.A7 = 6;
    /**
     * PDF ページの印刷先として A8 用紙を使用することを示す **/
    PdfCommand.A8 = 7;
    /**
     * PDF ページの印刷先として A9 用紙を使用することを示す **/
    PdfCommand.A9 = 8;
    /**
     * PDF ページの印刷先として A10 用紙を使用することを示す **/
    PdfCommand.A10 = 9;
    /**
     * PDF ページの印刷先として B1 用紙を使用することを示す **/
    PdfCommand.B1 = 10;
    /**
     * PDF ページの印刷先として B2 用紙を使用することを示す **/
    PdfCommand.B2 = 11;
    /**
     * PDF ページの印刷先として B3 用紙を使用することを示す **/
    PdfCommand.B3 = 12;
    /**
     * PDF ページの印刷先として B4 用紙を使用することを示す **/
    PdfCommand.B4 = 13;
    /**
     * PDF ページの印刷先として B5 用紙を使用することを示す **/
    PdfCommand.B5 = 14;
    /**
     * PDF ページの印刷先として B6 用紙を使用することを示す **/
    PdfCommand.B6 = 15;
    /**
     * PDF ページの印刷先として B7 用紙を使用することを示す **/
    PdfCommand.B7 = 16;
    /**
     * PDF ページの印刷先として B8 用紙を使用することを示す **/
    PdfCommand.B8 = 17;
    /**
     * PDF ページの印刷先として B9 用紙を使用することを示す **/
    PdfCommand.B9 = 18;
    /**
     * PDF ページの印刷先として B10 用紙を使用することを示す **/
    PdfCommand.B10 = 19;
    /**
     * PDF のページ印刷先の物理媒体の幅と高さ用のインデックス **/
    PdfCommand.MEDIA_SIZE_INDEXES = ["A1", "A2", "A3", "A4", "A5", "A6", "B1", "B2", "B3", "B4", "B5", "B6"];
    /**
     * PDF のページ印刷先の物理媒体の幅、および高さ（単位：mm） **/
    PdfCommand.MEDIA_SIZE = [[594, 841], [420, 594], [297, 420], [210, 297], [148, 210], [105, 148], [74, 105], [52, 74], [37, 52], [26, 37], [728, 1030], [515, 728], [364, 515], [257, 364], [182, 257], [128, 182], [91, 128], [64, 91], [45, 64], [32, 45]];
    /**
     * 用紙を縦方向に使用することを示す **/
    PdfCommand.PORTRAIT = 0;
    /**
     * 用紙を横方向に使用することを示す **/
    PdfCommand.LANDSCAPE = 1;
    /**
     * 印刷先の DPI のデフォルト値 **/
    PdfCommand.DEFAULT_DPI = 72;
    /**
     * ビューポートの最大数のデフォルト値 **/
    PdfCommand.DEFAULT_VIEWPORT_MAX = 8;
    /**
     * ユーザ座標の y 軸が上から下へ方向へ伸びることを示す（デフォルト） **/
    PdfCommand.TOP_TO_BOTTOM = 0;
    /**
     * ユーザ座標の y 軸が下から上へ方向へ伸びることを示す **/
    PdfCommand.BOTTOM_TO_TOP = 1;
    return PdfCommand;
})();
exports.PdfCommand = PdfCommand;
//# sourceMappingURL=PdfCommand.js.map