import { ObjectFolder } from '../tree/ObjectFolder';
/**
 * XML 形式データの読み込みを行う。
 * @since		2004/05/06
 * @author		fujita
 * @version	2004/05/06
 * Copyright (c) 2004 KITec Inc,.. All rights reserved.
**/
export declare class XMLReader {
    /**
     * コメントの開始／終了文字群
     **/
    static COMMENT: string[][];
    /**
     * データの読み込み元オブジェクト
     **/
    private mInput;
    /**
     * データ読み込みできるかどうかを表すフラグ
     **/
    private mCanRead;
    /**
     * データ（ファイル）の終端かどうかを表すフラグ
     **/
    private mEOF;
    /**
     * 読み込み途中の残データ
     **/
    private mRemain;
    /**
     * 読み込み用のリーダー
     **/
    private mBr;
    /**
     * 指定されたストリームを、データの読み込み元とする
     * 新たな XMLReader を作成する。
     * @param input	入力ストリーム
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    constructor(input: string);
    /**
     * 指定されたオブジェクトを、データの読み込み元とする
     * 新たな XMLReader を作成する。
     * @param input	入力ストリーム読み込み用オブジェクト
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    /**
     * 指定されたストリーム、およびファイル名をもとに
     * データ読み込みオブジェクトを設定する。<BR>
     * @param input	入力ストリーム
     * @param file	ファイル名
     * @return		データ読み込みオブジェクトが正常に
     * 設定された場合、true。
     * そうでない場合は、false
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    private setFile(input, file);
    /**
     * @param code
     * @return
     * @author kawae
     * @since 2005/12/09
    **/
    /**
     * XML 形式データ読み込み、そのデータを表現するツリー構造を
     * 持つオブジェクトを作成し、返す。
     * @return		データの読み込み、およびその内容が正常な場合、
     * ツリー構造を持つデータオブジェクト。
     * そうでない場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    read(): ObjectFolder;
    /**
     * @param code
     * @return
     * @author kawae
     * @since 2005/12/09
    **/
    /**
     * データ読み込みの前処理として、読み込み先ストリームをオープンする。
     * @return		正しくオープンできた場合、true。
     * そうでない場合は、false
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    open(): boolean;
    /**
     * データ読み込みの後処理として、読み込み先ストリームをクローズする。
     * @return		正しくクローズできた場合、true。
     * そうでない場合は、false
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    close(): boolean;
    /**
     * XML データを読み込み解析する。そして、そのデータを
     * 表現するツリー構造を持つオブジェクトを作成し、返す。
     * @return		データの読み込み、およびその内容が正常な場合、
     * ツリー構造を持つデータオブジェクト。
     * そうでない場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
     * @throws UnsupportedEncodingException
    **/
    parseXML(code: string): ObjectFolder;
    /**
     * 指定されたデータにおいて、指定されたインデックスから
     * 検索を開始し、引用符内を除き、区切り文字が最初に出現
     * したインデックスを返す。
     * @param line	検索を行うデータ
     * @param st		検索を開始するインデックス
     * @return		区切り文字が、指定されたインデックスと同じか
     * これより大きいインデックス位置にある場合は、
     * 最初に出現した位置のインデックス。
     * 区切り文字がない場合は、line の文字列の長さ
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    indexOfDelimiter(line: string, st: number): number;
    /**
     * 指定された要素を解析し、指定されたツリーオブジェクトへ
     * その要素が定義するデータを追加する。
     * @param elem	解析する要素
     * @param of		ツリーオブジェクト
     * @return		データ追加後の新しいツリーオブジェクト
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
     * @throws UnsupportedEncodingException
    **/
    parseElement(elem: string, of: ObjectFolder, code: string): ObjectFolder;
    /**
     * Base64タグが存在するか確認し、存在すればタグを外した文字列を得る
     * @param val
     * @return
     * @author kawae
     * @since 2005/08/11
    **/
    getValue(val: string, code: string): string;
    /**
     * 1 件分のXML データ（"<" から ">" まで）を読み込む。<BR>
     * #parseXML() で解析しやすいようデータの整形を主に行う。
     * @return		読み込まれた XML データ。
     * ストリームの終わりに達した場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    readXML(): string;
    /**
     * XML 形式データでのコメントを除く、1 行分のデータを読み込み。<BR>
     * 次のデータはコメントとして取り扱う。<BR>
     * (1)<!-- xxx --><BR>
     * (2)<? xxxxxx ?><BR>
     * @return		読み込まれたデータ。
     * ストリームの終わりに達した場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    readLineWithoutComment(): string;
    /**
     * 指定された行データを解析し、コメントを除くデータを返す。
     * @param line	解析を開始する行（データ）
     * @param start	コメントの開始文字
     * @param end		コメントの終了文字
     * @return		コメントを除くデータ。
     * ストリームの終わりに達した場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    trimComment(line: string, start: string, end: string): string;
    /**
     * 1 行分のデータを読み込む。
     * @return		前後の空白を除いた読み込んだデータ。
     * ストリームの終わりに達した場合は、null
     * @since		2004/05/06
     * @author		fujita
     * @version	2004/05/06
    **/
    readLine(): string;
}
