import { ObjectNode } from './ObjectNode';
/**
 * ツリーなどの階層構造における複数のノードを保持する親ノード。<BR>
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
**/
export declare class ObjectFolder extends ObjectNode {
    mChildren: Array<ObjectNode>;
    /**
     * 指定された名前と親を持つ新たな親ノードを構築する。
     * @param name		このノードの名前
     * @param parent	このノードの親
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    constructor(name?: string);
    /**
     * 階層構造内の全てのObjectFolderのノードを削除する**/
    removeAllElements(): void;
    /**
     * 指定されたパスに対応する親ノード、およびノードからなる
     * 階層構造を作成し、このノード下へ追加する。
     * @param s	パス
     * @return 最後に追加されたノード。
     * ノードが追加されたなかった場合は、このノード。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    addFolderFromPath(path: string): ObjectFolder;
    /**
     * このノードが保持する階層構造の直下の階層を対象として、指定された
     * 名前を持つノード、または親ノードを検索し、該当するノードが存在
     * する場合、そのノードを返す。
     * @param name				ノードの名前
     * @param includeFolder	親ノードを検索対象とする場合、true。
     * false を指定すると、名前が一致しても
     * 対象からは除外される。
     * @return ノード。該当するノードが存在しない場合は、null。
     * @see #getNode(String,boolean)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getNode(name: string, includeFolder?: boolean): ObjectNode;
    /**
     * このノードが保持する階層構造の全ノードを、１つの階層ごとに
     * 親ノード、ノードの順に並び替える。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    sortFolder(): void;
    /**
     * このノードが保持する階層構造のノード群（子ノード）を返す。
     * @return ノード群
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getChildren(): Array<ObjectNode>;
    /**
     * このノードの階層構造に、指定されたノードを追加する。
     * @param n			新たなに追加するノード
     * @param sameregist	このノードと同じ名前を持つノードが存在する場合、
     * 階層構造に追加するか否かを指定する。
     * 追加する場合は、true。追加しない場合は、false。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    addChild(n: ObjectNode, sameregist?: boolean): void;
    /**
     * このノードの階層構造の中から、指定された名前と同じ名前を持つ
     * すべてのノードを返す。大文字小文字は区別しない。
     * @param name ノードの名前
     * @return 指定された名前と同じ名前を持つノード群
    **/
    getNamedElement(s: string): Array<ObjectNode>;
    /**
     * このノードの階層構造のすべての階層を対象として、指定された名前と
     * 同じ名前を持つノードを検索し、そのノードを返す。
     * 同じ名前を持つノードが複数存在する場合、次の内容でノードが優先度
     * が決まる。
     * (1)所属する階層が異なる場合、階層が上のノード
     * (2)同じ階層の場合、最初に見つかった（先に追加された）ノード
     * @param name	ノードの名前
     * @return 指定された名前と同じ名前を持つノード
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    findNode(name: string): ObjectNode;
    findNodeFirst(name: string): ObjectNode;
    /**
     * ObjectFolderが持つ階層構造のすべてのノードを v に追加する。
     * @param v	ノードを設定先となるコレクション
     * @since   2010/04/12
     * @author  matsuzaki
    **/
    getAllChildNode(v: ObjectNode[]): void;
    /**
     * ObjectFolderにString値を設定
     * valがnullの場合、追加しない
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    addNode(name: string, val: string): void;
    /**
     * ObjectFolderにint値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    addNode(name: string, val: number): void;
    /**
     * ObjectFolderにboolean値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    addNode(name: string, val: boolean): void;
    /**
     * ObjectFolderにString値を設定
     * valがnullの場合、追加しない
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    static addNodeString(of: ObjectFolder, name: string, val: string): void;
    /**
     * ObjectFolderにint値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    static addNodeInt(of: ObjectFolder, name: string, val: number): void;
    /**
     * ObjectFolderにfloat値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    static addNodeFloat(of: ObjectFolder, name: string, val: number): void;
    /**
     * ObjectFolderにdouble値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    static addNodeDouble(of: ObjectFolder, name: string, val: number): void;
    /**
     * ObjectFolderにboolean値を設定
     * @param of
     * @param name
     * @param val
     * @author kawae
     * @since 2004/11/05
    **/
    static addNodeBool(of: ObjectFolder, name: string, val: boolean): void;
}
