var KFont_1 = require('../kwt/KFont');
var KColor_1 = require('../kwt/KColor');
var PdfCommand_1 = require('./PdfCommand');
//import { DecimalFormat } from '../../java/text/DecimalFormat'
var util_1 = require('./util');
//module option.pdf
/**
 * PDF コマンド・グラフィックス。<BR> PDF ファイルへの描画に必要な操作を提供する。
 * @since       2004/01/20
 * @author      fujita2004/08/18  Fujita 多角形グループ用の描画 コマンド、およびパターン、 シェーディングによる描画 コマンドを追加。 2004/09/30  Fujita クリッピング領域の設定、および 解除コマンドを追加。 Copyright (c) 2004 KITec Inc,.. All rights reserved.
**/
var PdfGraphics = (function () {
    /**
     * 描画位置などのフォーマット **/
    //mFormat: DecimalFormat;
    /**
     * 指定された親を持つ新たな PdfCommandGraphics を作成する。
     * @param pdfcmd  PDF コマンド
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    function PdfGraphics(pdfcmd) {
        /**
         * イタリック体の文字の傾き **/
        this.mFontSkew = PdfGraphics.FONT_SKEW;
        this.mPdfCmd = pdfcmd;
        //this.mFormat = new DecimalFormat("####0.####");
        this.reset();
    }
    /**
     * 描画色、フォントなどを初期状態へ設定する。
     * @since   2004/01/21
     * @author  fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.reset = function () {
        this.mColor = new KColor_1.KColor(0, 0, 0);
        this.mFontName = "MS GOTHIC";
        this.mFontStyle = KFont_1.KFont.PLAIN;
        this.mFontSize = 12;
        this.mLineWidth = 1;
        this.mLinePattern = PdfGraphics.SOLID_LINE;
        this.mOriginX = 0;
        this.mOriginY = 0;
    };
    /**
     * 現在のページへの描画を終了する。
     * @since   2004/01/21
     * @author  fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.dispose = function () {
        this.mPdfCmd.endPage();
        this.reset();
    };
    /**
     * PDF ページ上の原点を現在の座標系の点 (x, y) に変換する。
     * @param x   x 座標
     * @param y   y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.translate = function (x, y) {
        this.mOriginX += x;
        this.mOriginY += y;
    };
    /**
     * 指定した座標が指定した矩形に現在のクリップを設定する。
     * @param x       新しいクリップ矩形の x 座標
     * @param y       新しいクリップ矩形の y 座標
     * @param width   新しいクリップ矩形の幅
     * @param height  新しいクリップ矩形の高さ
     * @see #releaseClip()
     * @since       2004/10/06
     * @author      fujita
     * @version 2004/10/06
    **/
    PdfGraphics.prototype.setClip = function (x, y, width, height) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_CLIP);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.drawRect_(x, y, width, height, sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * 現在のクリッピング領域を解除し、出力先として 指定されている領域全体を描画領域とする。
     * @see #setClip(int,int,int,int)
     * @see #setClip(double,double,double,double)
     * @since       2004/09/30
     * @author      fujita
     * @version 2004/09/30
    **/
    PdfGraphics.prototype.releaseClip = function () {
        this.mPdfCmd.draw(PdfGraphics.CMD_UNCLIP);
    };
    /**
     * @see java.awt.Graphics#getColor()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.getColor = function () {
        return (this.mColor);
    };
    /**
     * @see java.awt.Graphics#setColor(Color)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.setColor = function (c) {
        if (c == null) {
            throw new Error();
        }
        this.mColor = c;
    };
    /**
     * @see java.awt.Graphics#getFont()
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.getFont = function () {
        return KFont_1.KFont.getFont(this.mFontName, this.mFontStyle, (this.mFontSize));
    };
    /**
     * 現在のフォント名を返す。
     * @return      フォント名
     * @see #setFontName(String)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.getFontName = function () {
        return this.mFontName;
    };
    /**
     * 現在のフォント・スタイルを返す。
     * @return      フォント・スタイル
     * @see #setFontStyle(int)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.getFontStyle = function () {
        return this.mFontStyle;
    };
    /**
     * 現在のフォント・サイズを返す。
     * @return      フォント・サイズ
     * @see #setFontSize(double)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.getFontSize = function () {
        return this.mFontSize;
    };
    /**
     * PDF ファイルへの出力にフォントを埋め込むか否かを判定する。 デフォルトは、フォントを埋め込まない。
     * @return  フォントを埋め込む場合、true。埋め込まない場合は、false。
     * @see #setFontEmbed(boolean)
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.isFontEmbedded = function () {
        return this.mIsFontEmbedded;
    };
    /**
     * @see java.awt.Graphics#setFont(Font)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.setFont = function (font) {
        this.setFontName(font.name());
        this.setFontStyle(font.style());
        this.setFontSize(font.size());
    };
    /**
     * 指定されたフォント名、スタイル、およびサイズを持つ フォントを設定する。
     * @param name    フォント名
     * @param style   フォント・スタイル
     * @param size    フォント・サイズ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.setFontParam = function (name, style, size) {
        this.setFontName(name);
        this.setFontStyle(style);
        this.setFontSize(size);
    };
    /**
     * 指定されたフォント名を持つフォントを設定する。
     * @param name    フォント名
     * @see #getFontName()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.setFontName = function (name) {
        if (name == null) {
            throw new Error();
        }
        this.mFontName = name;
    };
    /**
     * 指定されたスタイルを持つフォントを設定する。
     * @param style   フォント・スタイル
     * @see #getFontStyle()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.setFontStyle = function (style) {
        this.mFontStyle = style;
    };
    /**
     * 指定されたサイズを持つフォントを設定する。
     * @param size    フォント・サイズ
     * @see #getFontSize()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.setFontSize = function (size) {
        if (size < 0) {
            throw new Error();
        }
        this.mFontSize = size;
    };
    /**
     * PDF ファイルへの出力にフォントを埋め込むか否かを設定する。<BR> デフォルトは、フォントを埋め込まない。
     * @param embed   フォントを埋め込む場合、true。埋め込まない場合は、false。
     * @see #isFontEmbedded()
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.setFontEmbed = function (embed) {
        this.mIsFontEmbedded = embed;
    };
    /**
     * 描画に用いる線のパターンを設定する。<BR> double[]{ 0, 0 } の場合は、切れ目のない線となる。
     * @param dash    線のパターンとして実線と空白部分の長さを指定する
     * @see #getLinePattern()
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.setLinePattern = function (dash) {
        this.mLinePattern = new Array(dash.length);
        for (var i = 0; i < dash.length; i++) {
            if (dash[i] < 0) {
                throw new Error("negative line pattern");
            }
            this.mLinePattern[i] = dash[i];
        }
    };
    /**
     * 現在の線のパターンを取得する。
     * @return      線のパターン
     * @see     #setLinePattern(final double[])
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.getLinePattern = function () {
        var lp = new Array(this.mLinePattern.length);
        for (var i = 0; i < lp.length; i++) {
            lp[i] = this.mLinePattern[i];
        }
        return lp;
    };
    /**
     * 描画に用いる線の太さを設定する。
     * @param width   線の太さ
     * @see #getLineWidth()
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.setLineWidth = function (width) {
        if (width <= 0) {
            throw new Error("negative or 0 line width");
        }
        this.mLineWidth = width;
    };
    /**
     * 現在の線の太さを返す。
     * @return      線の太さ
     * @see     #setLineWidth(final double)
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.getLineWidth = function () {
        return this.mLineWidth;
    };
    /**
     * 最後に描画したイメージ名を返す。
     * @return      イメージ名
    **/
    PdfGraphics.prototype.getLastImageName = function () {
        return this.mLastImageName;
    };
    /**
     * @see java.awt.Graphics#drawArc(int,int,int,int,int,int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.drawArc = function (x, y, width, height, startAngle, arcAngle) {
        this.drawArc_(x, y, width, height, startAngle, arcAngle, false);
    };
    /**
     * @see java.awt.Graphics#fillArc(int,int,int,int,int,int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.fillArc = function (x, y, width, height, startAngle, arcAngle) {
        this.drawArc_(x, y, width, height, startAngle, arcAngle, true);
    };
    /**
     * 指定された矩形をカバーする円弧または楕円弧の輪郭を PDF コマンドとして出力する。
     * @param x           孤の左上隅の x 座標
     * @param y           孤の左上隅の y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param startAngle  開始角度[°]
     * @param arcAngle    開始角度に対する弧の展開角度の大きさ[°]
     * @param fill        弧の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.drawArc_ = function (x, y, width, height, startAngle, arcAngle, fill) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_ARC);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        if (this.mPdfCmd.getUserCoordinateDirectionY() == PdfCommand_1.PdfCommand.BOTTOM_TO_TOP) {
            y -= height;
        }
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x + (width / 2.0), y + (height / 2.0));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendSize(sb, PdfGraphics.SEPARATOR_GRAPHICS, width / 2.0, height / 2.0);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.format(startAngle));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.format(arcAngle));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(fill ? PdfGraphics.FILL : PdfGraphics.DRAW);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた 多角形の集合を指定されたパターンで塗りつぶす PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x座標の配列
     * @param yPoints     すべての多角形の y座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param name        パターン名
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawPatternedGroupPolygon = function (name, xPoints, yPoints, nPoints) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_PATTERN);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.drawGroupPolygon_(sb, xPoints, yPoints, nPoints, true);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた 多角形の集合を指定されたシェーディングで 塗りつぶす PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x座標の配列
     * @param yPoints     すべての多角形の y座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param name        シェーディング名
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawShadingGroupPolygon = function (name, xPoints, yPoints, nPoints) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_SHADING);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.drawGroupPolygon_(sb, xPoints, yPoints, nPoints, true);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形の 集合を PDF コマンドとして出力する。
     * @param xPoints     すべての多角形の x 座標の配列
     * @param yPoints     すべての多角形の y 座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param fill        多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawGroupPolygon = function (xPoints, yPoints, nPoints, fill) {
        var sb = new util_1.StringBuilder();
        this.drawGroupPolygon_(sb, xPoints, yPoints, nPoints, fill);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形の 集合を PDF コマンドとして、指定された sb へ出力する。
     * @param sb          PDF コマンドを出力先となるオブジェクト
     * @param xPoints     すべての多角形の x 座標の配列
     * @param yPoints     すべての多角形の y 座標の配列
     * @param nPoints     各多角形の点の総数の配列
     * @param fill        多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawGroupPolygon_ = function (sb, xPoints, yPoints, nPoints, fill) {
        sb.append(PdfGraphics.CMD_GROUP_POLYGON);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(fill ? PdfGraphics.FILL : PdfGraphics.DRAW);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(nPoints.length);
        var idx = 0;
        for (var _i = 0; _i < nPoints.length; _i++) {
            var nPoint = nPoints[_i];
            sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
            sb.append(nPoint);
            for (var j = 0; j < nPoint; j++, idx++) {
                sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
                this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, xPoints[idx], yPoints[idx]);
            }
        }
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
    };
    /**
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.drawImagePosition = function (img, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2) {
        var pw = Math.abs(dx2 - dx1);
        var ph = Math.abs(dy2 - dy1);
        var px = (dx1 < dx2) ? dx1 : dx2;
        var py = (dy1 < dy2) ? dy1 : dy2;
        var pixels = new Array(pw * ph);
        if (!img.getPixels(px, py, pw, ph, pixels, 0, pw))
            return false;
        var ox = (sx1 < sx2) ? sx1 : sx2;
        var oy = (sy1 < sy2) ? sy1 : sy2;
        var ow = Math.abs(sx2 - sx1);
        var oh = Math.abs(sy2 - sy1);
        return this.drawImagePixels(pixels, ox, oy, pw, ph, ow, oh, 30);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を出力する。<BR>
     * イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画されるイメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.drawImageRect = function (img, x, y, width, height) {
        if (img == null) {
            return false;
        }
        var size = img.getSize();
        var w = size.width;
        var h = size.height;
        return this.drawImage_2(img, x, y, w, h, width, height, 0);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたファイルをイメージとして出力する。<BR> イメージは、出力される領域の（x, y）に その左上隅を合わせて出力される。
     * @param file    描画されるイメージのファイル名
     * @param x       x 座標
     * @param y       y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param theta   出力するイメージの傾き
     * @return      ファイル名が null の場合、false。そうでない場合は、true
     * @since       2004/11/29
     * @author      Kawae
     * @version 2004/11/29
    **/
    PdfGraphics.prototype.drawImage_ = function (file, x, y, width, height, theta) {
        if (file == null) {
            return false;
        }
        return this.drawImageCommand(PdfGraphics.CMD_IMAGE, file, x, y, 0, 0, width, height, theta);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたファイルをイメージとして出力する。<BR> イメージは、出力される領域の（x, y）に その左上隅を合わせて出力される。
     * @param file    描画されるイメージのファイル名
     * @param x       x 座標
     * @param y       y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param theta   出力するイメージの傾き
     * @return      ファイル名が null の場合、false。そうでない場合は、true
     * @since       2004/11/29
     * @author      Kawae
     * @version 2004/11/29
    **/
    PdfGraphics.prototype.drawImageAsJpeg = function (file, x, y, width, height, theta) {
        if (file == null) {
            return false;
        }
        return this.drawImageCommand(PdfGraphics.CMD_JPEG, file, x, y, 0, 0, width, height, theta);
    };
    /**
     * 指定されたイメージの現在利用可能な部分だけを PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画される指定イメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.drawImage_1 = function (img, x, y) {
        if (img == null) {
            return false;
        }
        var size = img.getSize();
        var w = size.width;
        var h = size.height;
        var ow = w * PdfGraphics.PT_TO_MM;
        var oh = h * PdfGraphics.PT_TO_MM;
        return this.drawImage_2(img, x, y, w, h, ow, oh, 0);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img     描画される指定イメージ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.drawImage_2 = function (img, x, y, w, h, ow, oh, theta) {
        var iw = (w);
        var ih = (h);
        var pixels = new Array(iw * ih);
        if (!img.getPixels(0, 0, iw, ih, pixels, 0, iw)) {
            return false;
        }
        return this.drawImagePixels(pixels, x, y, w, h, ow, oh, theta);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> jpegイメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param pixels  描画される指定イメージのピクセルデータ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/11/29
     * @author      kawae
     * @version 2004/11/29
    **/
    PdfGraphics.prototype.drawImageAsJpegPixels = function (pixels, x, y, w, h, ow, oh, theta) {
        this.mLastImageName = this.mPdfCmd.draw(pixels);
        if (!this.drawImageCommand(PdfGraphics.CMD_JPEG, this.mLastImageName, x, y, w, h, ow, oh, theta)) {
            return false;
        }
        return true;
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param pixels  描画される指定イメージのピクセルデータ
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.drawImagePixels = function (pixels, x, y, w, h, ow, oh, theta) {
        this.mLastImageName = this.mPdfCmd.draw(pixels);
        if (!this.drawImageCommand(PdfGraphics.CMD_IMAGE, this.mLastImageName, x, y, w, h, ow, oh, theta)) {
            return false;
        }
        return true;
    };
    PdfGraphics.prototype.drawImageCommand = function (command, name, x, y, w, h, ow, oh, theta) {
        var sb = new util_1.StringBuilder();
        sb.append(command);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(w);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(h);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        var y1 = y;
        var y2 = y;
        if (this.mPdfCmd.getUserCoordinateDirectionY() == PdfCommand_1.PdfCommand.TOP_TO_BOTTOM) {
            y2 += oh;
        }
        else {
            y2 -= oh;
        }
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x, y1);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, (x + ow), y2);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(theta);
        this.mPdfCmd.draw(sb.toString());
        return true;
    };
    /**
     * 指定されたイメージを、指定されたフォーム上の項目として出力する。
     * @param file        描画されるイメージのファイル名
     * @param formatName  フォーム名
     * @param ItemName    項目名
     * @since       2004/03/03
     * @author      fujita
     * @version 2004/03/03
    **/
    PdfGraphics.prototype.drawItemImage = function (file, formatName, ItemName) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_ITEM_IMAGE);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(file);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(formatName);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(ItemName);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * 指定された文字列を、指定されたフォーム上の項目として出力する。
     * @param str         出力する文字列
     * @param formatName  フォーム名
     * @param ItemName    項目名
     * @since       2004/02/20
     * @author      fujita
     * @version 2004/02/20
    **/
    PdfGraphics.prototype.drawItemString = function (str, formatName, ItemName) {
        if (str == null || str.length == 0)
            return;
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_ITEM_TEXT);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(PdfGraphics.STRING_QUOTAION);
        sb.append(str);
        sb.append(PdfGraphics.STRING_QUOTAION);
        sb.append(formatName);
        sb.append(PdfGraphics.SEPARATOR_TEXT);
        sb.append(ItemName);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * 指定された矩形の内部に指定された url へのリンクを出力する。<BR>
     * @param url         リンク先の URL
     * @param x1          矩形の左上の x 座標
     * @param y1          矩形の左上の y 座標
     * @param x2          矩形の右下の x 座標
     * @param y2          矩形の右下の y 座標
     * @param boderWidth  矩形を囲む線の幅。矩形を透明とする場合は、0
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.drawLink = function (url, x1, y1, x2, y2, boderWidth) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_LINK);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(url);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x1, y1);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x2, y2);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendWidth(sb, boderWidth);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * PDF ページ上の点 (x1, y1) と点 (x2, y2) との間に 現在の色を使って線を出力する。
     * @param x1  最初の点の x 座標
     * @param y1  最初の点の y 座標
     * @param x2  2 番目の点の x 座標
     * @param y2  2 番目の点の y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.drawLine = function (x1, y1, x2, y2) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_LINE);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x1, y1);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x2, y2);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * @see java.awt.Graphics#drawPolygon(int[],int[],int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.drawPolyline = function (xPoints, yPoints, nPoints) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_POLYLINE);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(nPoints);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoints(sb, PdfGraphics.SEPARATOR_GRAPHICS, xPoints, yPoints, nPoints);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mPdfCmd.draw(sb.toString());
    };
    //drawPolyline(xPoints: number[], yPoints: number[], nPoints: number): void {
    //    let sb: StringBuilder = new StringBuilder();
    //    sb.append(PdfGraphics.CMD_POLYLINE);
    //    sb.append(PdfGraphics.SEPARATOR_COMMAND);
    //    this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
    //    sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
    //    sb.append(nPoints);
    //    sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
    //    this.appendPoints(sb, PdfGraphics.SEPARATOR_GRAPHICS, xPoints, yPoints, nPoints);
    //    sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
    //    this.appendLineWidth(sb);
    //    sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
    //    this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
    //    this.mPdfCmd.draw(sb.toString());
    //}
    /**
     * @see java.awt.Graphics#drawPolygon(int[],int[],int)
     * @since       2004/01/20
     * @author      fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.drawPolygon = function (xPoints, yPoints, nPoints) {
        this.drawPolygon_(xPoints, yPoints, nPoints, false);
    };
    /**
     * @see java.awt.Graphics#fillPolygon(int[],int[],int)
     * @since   2004/01/20
     * @author  fujita
     * @version 2004/01/20
    **/
    PdfGraphics.prototype.fillPolygon = function (xPoints, yPoints, nPoints) {
        this.drawPolygon_(xPoints, yPoints, nPoints, true);
    };
    ///**
    // * 指定された整数の配列を単精度小数点の配列へ変換する。
    // * @param array   整数の配列
    // * @return      変換された単精度小数点の配列
    // * @since       2004/02/05
    // * @author      fujita
    // * @version 2004/02/05
    //**/
    //toFloatArray(array: number[]): number[] {
    //    let fa: number[] = new Array(array.length);
    //    for (let i: number = 0; i < array.length; i++) {
    //        fa[i] = array[i];
    //    }
    //    return fa;
    //}
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を 指定されたパターンで塗りつぶす PDF コマンドを 出力する。
     * @param name    パターン名
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawPatternedPolygon = function (name, xPoints, yPoints, nPoints) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_PATTERN);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.drawPolygon__(sb, xPoints, yPoints, nPoints, true);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を 指定されたシェーディングで塗りつぶす PDF コマンドを 出力する。
     * @param name    シェーディング名
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawShadingPolygon = function (name, xPoints, yPoints, nPoints) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_SHADING);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.drawPolygon__(sb, xPoints, yPoints, nPoints, true);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を PDF コマンドとして出力する。
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/08/18  Fujita多角形の PDF コマンド 出力部分を別メソッドへ 移行。
    **/
    PdfGraphics.prototype.drawPolygon_ = function (xPoints, yPoints, nPoints, fill) {
        var sb = new util_1.StringBuilder();
        this.drawPolygon__(sb, xPoints, yPoints, nPoints, fill);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * x 座標と y 座標の配列で定義された閉じた多角形を PDF コマンドとして、指定された sb へ出力する。
     * @param sb      PDF コマンドの出力先となるオブジェクト
     * @param xPoints x 座標の配列
     * @param yPoints y 座標の配列
     * @param nPoints 点の総数
     * @param fill    多角形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/08/18
     * @author      fujita
     * @version 2004/08/18
    **/
    PdfGraphics.prototype.drawPolygon__ = function (sb, xPoints, yPoints, nPoints, fill) {
        sb.append(PdfGraphics.CMD_POLYGON);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(fill ? PdfGraphics.FILL : PdfGraphics.DRAW);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(nPoints);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoints(sb, PdfGraphics.SEPARATOR_GRAPHICS, xPoints, yPoints, nPoints);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
    };
    /**
     * 指定された矩形の輪郭を出力する。
     * @param x       矩形の左上の x 座標
     * @param y       矩形の左上の y 座標
     * @param width   矩形の幅
     * @param height  矩形の高さ
     * @param fill    矩形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/01/21
     * @author      fujita
     * @version 2004/01/21
    **/
    PdfGraphics.prototype.drawRect = function (x, y, width, height, fill) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_RECT);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        this.drawRect_(x, y, width, height, sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(fill ? PdfGraphics.FILL : PdfGraphics.DRAW);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * 指定された矩形を、PDF 用の矩形領域の形式にて 指定された buffer へ出力する。
     * @param x           矩形の左上の x 座標
     * @param y           矩形の左上の y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param buffer      データの出力先
     * @param separator   データの区切り文字
     * @since       2004/09/30
     * @author      fujita
     * @version 2004/09/30
    **/
    PdfGraphics.prototype.drawRect_ = function (x, y, width, height, buffer, separator) {
        this.appendPosition(buffer, separator, x, y);
        buffer.append(separator);
        var y2 = y;
        if (this.mPdfCmd.getUserCoordinateDirectionY() == PdfCommand_1.PdfCommand.TOP_TO_BOTTOM) {
            y2 += height;
        }
        else {
            y2 -= height;
        }
        this.appendPosition(buffer, separator, x + width, y2);
    };
    /**
     * 丸いコーナー付きの輪郭の矩形を、現在の色を使用して出力する。<BR> 出力される矩形は、左端と右端がそれぞれ x と x + width、 上端と下端がそれぞれ y と y + height で指定される。
     * @param x           描画される矩形の x 座標
     * @param y           描画される矩形の y 座標
     * @param width       描画される矩形の幅
     * @param height      描画される矩形の高さ
     * @param arcWidth    4 隅の弧の水平方向の直径
     * @param arcHeight   4 隅の弧の垂直方向の直径
     * @param fill        矩形の内部を塗りつぶす場合、true。塗りつぶさない場合は、false
     * @since       2004/02/16
     * @author      fujita
     * @version 2004/02/16
    **/
    PdfGraphics.prototype.drawRoundRect = function (x, y, width, height, arcWidth, arcHeight, isFill) {
        var x1 = x + arcWidth / 2.0;
        var x2 = x + width - arcWidth / 2.0;
        var y1;
        var y2;
        var ty;
        var by;
        var ry;
        if (this.mPdfCmd.getUserCoordinateDirectionY() == PdfCommand_1.PdfCommand.TOP_TO_BOTTOM) {
            y1 = y + arcHeight / 2.0;
            y2 = y + height - arcHeight / 2.0;
            ty = y;
            by = y + height;
            ry = y + height - arcHeight;
        }
        else {
            y1 = y - arcHeight / 2.0;
            y2 = y - height + arcHeight / 2.0;
            ty = y;
            by = y - height;
            ry = y - height + arcHeight;
        }
        this.drawArc_(x, y, arcWidth, arcHeight, 90, 180, isFill);
        this.drawArc_(x + width - arcWidth, y, arcWidth, arcHeight, 0, 90, isFill);
        this.drawArc_(x, ry, arcWidth, arcHeight, 180, 270, isFill);
        this.drawArc_(x + width - arcWidth, ry, arcWidth, arcHeight, 270, 360, isFill);
        if (isFill) {
            this.drawRect(x1, ty, width - arcWidth, height, isFill);
            this.drawRect(x, y1, width, height - arcHeight, isFill);
        }
        else {
            this.drawLine(x1, ty, x2, ty);
            this.drawLine(x1, by, x2, by);
            this.drawLine(x, y1, x, y2);
            this.drawLine(x + width, y1, x + width, y2);
        }
    };
    /**
     * 現在のフォントと色を使い、指定された文字列によって 指定されたテキストを出力する。<BR> 指定された配置方法により、文字のベースラインは、 位置 (x, y)に対して、左端、または中央となる。
     * @param str     出力する文字列
     * @param x       x 座標
     * @param y       y 座標
     * @param center  位置（x, y）を文字の中央とする場合、true。左端とする場合は、false
     * @param rotate  文字の回転角
     * @param width   テキストの出力幅
     * @param type    出力幅に対する空白の割り当て方法
     * @since       2004/02/09
     * @author      fujita
     * @version 2004/02/09
    **/
    PdfGraphics.prototype.drawString = function (str, x, y, center, rotate, width, type) {
        if (str == null || str.length == 0) {
            return;
        }
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_TEXT);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(PdfGraphics.STRING_QUOTAION);
        sb.append(str);
        sb.append(PdfGraphics.STRING_QUOTAION);
        this.appendPosition(sb, PdfGraphics.SEPARATOR_TEXT, x, y);
        sb.append(PdfGraphics.SEPARATOR_TEXT);
        this.appendFontNameAndSize(sb, PdfGraphics.SEPARATOR_TEXT);
        sb.append(PdfGraphics.SEPARATOR_TEXT);
        this.appendColor(sb, PdfGraphics.SEPARATOR_TEXT);
        var whatdraw = 0;
        if (this.isFontEmbedded()) {
            whatdraw += 32;
        }
        if (width > 0 && type != null && (type === PdfGraphics.SPACE_FOR_CHAR || type === PdfGraphics.SPACE_FOR_PITCH)) {
            whatdraw += 16;
        }
        else if (rotate != 0) {
            whatdraw += 8;
        }
        else if ((this.getFontStyle() & KFont_1.KFont.ITALIC) != 0) {
            whatdraw += 4;
        }
        else if ((this.getFontStyle() & KFont_1.KFont.BOLD) != 0) {
            whatdraw += 2;
        }
        else if (center) {
            whatdraw += 1;
        }
        if (whatdraw >= 1) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            if (center) {
                sb.append("1");
            }
            else {
                sb.append("0");
            }
        }
        if (whatdraw >= 2) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            sb.append((this.mFontStyle & KFont_1.KFont.BOLD) == 0 ? PdfGraphics.PLAIN : PdfGraphics.BOLD);
        }
        if (whatdraw >= 4) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            sb.append((this.mFontStyle & KFont_1.KFont.ITALIC) == 0 ? 0 : this.mFontSkew);
        }
        if (whatdraw >= 8) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            sb.append(rotate);
        }
        if (whatdraw >= 16) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            this.appendStringWidth(sb, width);
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            sb.append(type);
        }
        if (whatdraw >= 32) {
            sb.append(PdfGraphics.SEPARATOR_TEXT);
            sb.append(PdfGraphics.EMBED);
        }
        this.mPdfCmd.draw(sb.toString());
    };
    /**
     * 指定された点（x, y）を PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param x           x 座標
     * @param y           y 座標
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendPosition = function (sb, separator, x, y) {
        sb.append(this.format(this.mPdfCmd.userXToPageX(this.mOriginX + x)));
        sb.append(separator);
        sb.append(this.format(this.mPdfCmd.userYToPageY(this.mOriginY + y)));
    };
    /**
     * 指定された x 座標と y 座標の配列を PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param xPoints     x 座標の配列
     * @param yPoints     y 座標の配列
     * @param nPoint      点の総数
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendPoints = function (sb, separator, xPoints, yPoints, nPoints) {
        var i = 0;
        var max = nPoints - 1;
        for (; i < max; i++) {
            this.appendPosition(sb, separator, xPoints[i], yPoints[i]);
            sb.append(separator);
        }
        this.appendPosition(sb, separator, xPoints[i], yPoints[i]);
    };
    /**
     * 指定された幅と高さを PDF コマンド用に フォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @param width       幅
     * @param height      高さ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendSize = function (sb, separator, width, height) {
        this.appendWidth(sb, width);
        sb.append(separator);
        this.appendHeight(sb, height);
    };
    /**
     * 現在の色を PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendColor = function (sb, separator) {
        var c = this.getColor();
        sb.append(c.getRed());
        sb.append(separator);
        sb.append(c.getGreen());
        sb.append(separator);
        sb.append(c.getBlue());
    };
    /**
     * 現在のフォントの名前とサイズを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendFontNameAndSize = function (sb, separator) {
        sb.append(this.mFontName);
        sb.append(separator);
        this.appendWidth(sb, this.mFontSize);
    };
    /**
     * 指定された文字列の幅を PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param width   文字列の幅
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendStringWidth = function (sb, width) {
        this.appendWidth(sb, width);
    };
    /**
     * 現在の線の太さを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendLineWidth = function (sb) {
        this.appendWidth(sb, this.mLineWidth);
    };
    /**
     * 現在の線のパターンを PDF コマンド用にフォーマットして出力する。
     * @param sb          PDF コマンドの出力先
     * @param separator   コマンドの区切り文字
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendLinePattern = function (sb, separator) {
        var max = this.mLinePattern.length - 1;
        for (var i = 0; i < max; i++) {
            this.appendWidth(sb, this.mLinePattern[i]);
            sb.append(separator);
        }
        this.appendWidth(sb, this.mLinePattern[max]);
    };
    /**
     * 指定された幅を PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param width   幅
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendWidth = function (sb, width) {
        sb.append(this.format(this.mPdfCmd.userWidthToPageWidth(width)));
    };
    /**
     * 指定された高さを PDF コマンド用にフォーマットして出力する。
     * @param sb      PDF コマンドの出力先
     * @param height  高さ
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.appendHeight = function (sb, height) {
        sb.append(this.format(this.mPdfCmd.userHeightToPageHeight(height)));
    };
    /**
     * 指定された数値（座標、幅、高さなど）を PDF コマンド用にフォーマットする。
     * @param number  変換する数値
     * @return      PDF コマンド用に変換された数値
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.format = function (number) {
        return number.toFixed(4); //this.mFormat.format(number);
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param img         描画されるイメージ
     * @param x           x 座標
     * @param y           y 座標
     * @param width       矩形の幅
     * @param height      矩形の高さ
     * @param observer    イメージがさらに変換されることを通知するオブジェクト
     * @return      イメージが完全にロードされている場合、true。そうでない場合は、false
     * @since       2004/02/05
     * @author      fujita
     * @version 2004/02/05
    **/
    PdfGraphics.prototype.draw3D = function (idtf, x1, y1, x2, y2, fov) {
        if (idtf == null) {
            return false;
        }
        this.mLastImageName = this.mPdfCmd.draw(idtf);
        if (!this.draw3DName(this.mLastImageName, x1, y1, x2, y2, fov)) {
            return false;
        }
        return true;
    };
    /**
     * 指定された矩形の内部に収まるようにスケーリングして、 指定されたイメージの利用可能な部分を PDF コマンドとして出力する。<BR> イメージは、出力される領域の（x, y）にその左上隅を合わせて出力される。
     * @param name    イメージデータの名前
     * @param x       x 座標
     * @param y       y 座標
     * @param w       イメージの幅
     * @param h       イメージの高さ
     * @param ow      矩形の幅
     * @param oh      矩形の高さ
     * @param theta   出力するイメージの傾きそうでない場合は、false
     * @since       2004/02/13
     * @author      fujita
     * @version 2004/02/13
    **/
    PdfGraphics.prototype.draw3DName = function (name, x1, y1, x2, y2, fov) {
        var sb = new util_1.StringBuilder();
        sb.append(PdfGraphics.CMD_3D);
        sb.append(PdfGraphics.SEPARATOR_COMMAND);
        sb.append(name);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        var yy1 = y1;
        var yy2 = y2;
        sb.append(this.format(this.mPdfCmd.userXToPageX(this.mOriginX + x1)));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.format(this.mPdfCmd.userYToPageY(this.mOriginY + yy1)));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.format(this.mPdfCmd.userXToPageX(this.mOriginX + x2)));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.format(this.mPdfCmd.userYToPageY(this.mOriginY + yy2)));
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(0);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(fov);
        this.mPdfCmd.draw(sb.toString());
        return true;
    };
    /**
     * テストメソッド
     * @param s
     * @author kawae
     * @since 2006/02/03
    **/
    PdfGraphics.prototype.draw = function (s) {
        this.mPdfCmd.draw(s);
    };
    /**
     * 星型のパターンを表す定数 **/
    PdfGraphics.PATTERN_STAR = "Pt101d";
    /**
     * タイプ2（軸）シェーディングを表す定数 **/
    PdfGraphics.SHADING_TYPE2 = "Sh201d";
    /**
     * 線の属性として切れ目のない直線を示す **/
    PdfGraphics.SOLID_LINE = [0, 0];
    /**
     * 出力幅に対する空白を文字間隔へ割り当てることを示す **/
    PdfGraphics.SPACE_FOR_PITCH = "spaceX";
    /**
     * 出力幅に対する空白を文字幅へ割り当てることを示す **/
    PdfGraphics.SPACE_FOR_CHAR = "scaleX";
    /**
     * テキストコマンドの区切り文字 **/
    PdfGraphics.SEPARATOR_TEXT = "*";
    /**
     * グラフィックストコマンドの区切り文字 **/
    PdfGraphics.SEPARATOR_GRAPHICS = ",";
    /**
     * コマンドとパラメータの区切り文字 **/
    PdfGraphics.SEPARATOR_COMMAND = " ";
    /**
     * テキストコマンドなどでの文字列を囲む引用符 **/
    PdfGraphics.STRING_QUOTAION = "#";
    /**
     * PDF コマンドの円弧描画コマンド **/
    PdfGraphics.CMD_ARC = "arc";
    /**
     * PDF コマンドのイメージ描画コマンド **/
    PdfGraphics.CMD_IMAGE = "image";
    /**
     * PDF コマンドのJPEGイメージ描画コマンド **/
    PdfGraphics.CMD_JPEG = "jpeg";
    /**
     * PDF コマンドのイメージに対するアイテム・コマンド **/
    PdfGraphics.CMD_ITEM_IMAGE = "itemimage";
    /**
     * PDF コマンドのテキストに対するアイテム・コマンド **/
    PdfGraphics.CMD_ITEM_TEXT = "itemtext";
    /**
     * PDF コマンドの線描画コマンド **/
    PdfGraphics.CMD_LINE = "line";
    /**
     * PDF コマンドのリンク描画コマンド **/
    PdfGraphics.CMD_LINK = "link";
    /**
     * PDF コマンドのポリゴン描画コマンド **/
    PdfGraphics.CMD_POLYGON = "polygon";
    /**
     * PDF コマンドのポリライン描画コマンド **/
    PdfGraphics.CMD_POLYLINE = "polyline";
    /**
     * PDF コマンドの矩形描画コマンド **/
    PdfGraphics.CMD_RECT = "rect";
    /**
     * PDF コマンドのテキスト描画コマンド **/
    PdfGraphics.CMD_TEXT = "text";
    /**
     * PDF コマンドのグループポリゴン描画コマンド **/
    PdfGraphics.CMD_GROUP_POLYGON = "grouppolygon";
    /**
     * PDF コマンドのパターン描画コマンド **/
    PdfGraphics.CMD_PATTERN = "pattern";
    /**
     * PDF コマンドのシェーディング描画コマンド **/
    PdfGraphics.CMD_SHADING = "shading";
    /**
     * PDF コマンドのクリッピング設定コマンド **/
    PdfGraphics.CMD_CLIP = "clip";
    /**
     * PDF コマンドのクリッピング解除コマンド **/
    PdfGraphics.CMD_UNCLIP = "unclip";
    /**
     * PDF コマンドのイメージ描画コマンド **/
    PdfGraphics.CMD_3D = "3d";
    /**
     * イタリック体の文字の傾きの **/
    PdfGraphics.FONT_SKEW = 20.0;
    /**
     * ポイントをミリメートルへ変換する場合の比率 （1 pt = 1 / 2.8346 mm）**/
    PdfGraphics.PT_TO_MM = 1.0 / 2.8346;
    /**
     * PDF コマンドでの矩形などの外形のみを描くことを表す **/
    PdfGraphics.DRAW = "0";
    /**
     * PDF コマンドでの矩形などを塗りつぶすことを表す **/
    PdfGraphics.FILL = "1";
    /**
     * PDF コマンドでの装飾のない通常のフォントを表す **/
    PdfGraphics.PLAIN = "0";
    /**
     * PDF コマンドでのボールド体のフォントを表す **/
    PdfGraphics.BOLD = "2";
    /**
     * PDF コマンドでのフォントを埋め込むことを表す **/
    PdfGraphics.EMBED = "1";
    /**
     * PDF コマンドでのフォント出力幅として 指定フォントの幅をそのまま使用することを表す**/
    PdfGraphics.TEXT_WIDTH = 0;
    return PdfGraphics;
})();
exports.PdfGraphics = PdfGraphics;
//# sourceMappingURL=PdfGraphics.js.map