import { error } from '../../common';
//module jp.kitec.lib.util.tree
/**
 * ツリーなどの階層構造におけるノード。<BR>
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
**/
export class ObjectNode {
    /**
     * 指定された名前、親、およびデータを持つ新たなノードを構築する。
     * @param name		このノードの名前
     * @param parent	このノードの親
     * @param o		このノードに対応するデータ
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    constructor(name = "", o = null) {
        this.mName = name;
        this.mObject = o;
    }
    /**
     * このノードの名前を返す。
     * @return ノードの名前
     * @see #setName(String)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getName() {
        return this.mName;
    }
    /**
     * 指定された名前を、このノードの名前として設定する。
     * @param ノードの名前
     * @see #getName()
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    setName(name) {
        this.mName = name;
    }
    /**
     * このノードに対応するデータを返す。
     * @return このノードに対応するデータ
     * @see #setObject(Object)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getObject() {
        return this.mObject;
    }
    /**
     * 指定されたデータを、このノードに対応するデータとして設定する。
     * @param このノードに対応するデータ
     * @see #getObject()
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    setObject(o) {
        this.mObject = o;
    }
    /**
     * このノードの親を返す。
     * @return ノードの親
     * @see #setParent(ObjectFolder)
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getParent() {
        return this.mParent;
    }
    /**
     * 指定された ObjectFolder を、このノードの親として設定する。
     * @param ノードの親とする ObjectFolder
     * @see #getParent()
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    setParent(f) {
        this.mParent = f;
    }
    /**
     * Stringの取得
     * 存在しない場合はnullを返す
     * @param defalutValue
    **/
    getString(defalutValue = null) {
        if (this.getObject() != null && typeof this.getObject() === "string") {
            return (this.getObject());
        }
        return defalutValue;
    }
    /**
     * int値の取得
     * 存在しない場合はdefaultValueを返す
     * @param defaultValue
    **/
    getInt(defaultValue) {
        var r = NaN;
        try {
            r = parseIntStrict(this.getString());
        }
        catch (e) {
        }
        if (isNaN(r)) {
            if (defaultValue === undefined)
                throw new error.util.NoSuchElementException();
            return defaultValue;
        }
        return r;
    }
    /**
     * float値の取得
     * 存在しない場合はdefaultValueを返す
     * @param defaultValue
    **/
    getFloat(defaultValue) {
        var r = NaN;
        try {
            r = parseFloatStrict(this.getString());
        }
        catch (e) {
        }
        if (isNaN(r)) {
            if (defaultValue === undefined)
                throw new error.util.NoSuchElementException();
            return defaultValue;
        }
        return r;
    }
    /**
     * double値の取得
     * 存在しない場合はdefaultValueを返す**/
    getDouble(defaultValue) {
        var r = NaN;
        try {
            r = parseFloatStrict(this.getString());
        }
        catch (e) {
        }
        if (isNaN(r)) {
            if (defaultValue === undefined)
                throw new error.util.NoSuchElementException();
            return defaultValue;
        }
        return r;
    }
    /**
     * boolの取得
     * 存在しない場合はdefaultValueを返す
     * @param defaultValue
    **/
    getBool(defaultValue) {
        try {
            var res = this.getString().trim();
            if (res === "1")
                return true;
            if (res === "0")
                return false;
        }
        catch (e) {
        }
        if (defaultValue === undefined)
            throw new error.lang.IllegalStateException();
        return defaultValue;
    }
    /**
     * ノードオブジェクトがStringか確認
     * @return
    **/
    isStringObject() {
        return typeof this.getObject() === "string";
    }
    /**
     * ノードオブジェクトがintか確認
     * @return
    **/
    isIntObject() {
        if (!this.isStringObject())
            return false;
        var str = this.getString();
        return !isNaN(parseIntStrict(str));
    }
    /**
     * オブジェクトがBooleanか確認
     * 文字列かつ"1"or"0"か確認
     * @return
    **/
    isBoolObject() {
        if (!this.isStringObject())
            return false;
        var str = this.getString();
        if (str != null) {
            str = str.trim();
            return (str === "1" || str === "0");
        }
        return false;
    }
    /**
     * ノードオブジェクトがfloatか確認
     * @return
    **/
    isFloatObject() {
        if (!this.isStringObject())
            return false;
        var str = this.getString();
        return !isNaN(parseFloatStrict(str));
    }
    /**
     * ノードオブジェクトがdoubleか確認
     * @return
    **/
    isDoubleObject() {
        if (!this.isStringObject())
            return false;
        var str = this.getString();
        return !isNaN(parseFloatStrict(str));
    }
    /**
     * Stringの取得
     * 存在しない場合はnullを返す
     * @param on
    **/
    static getNodeString(on) {
        //    return ((on.getObject()) as string);
        return on.getString();
    }
    /**
     * int値の取得
     * @param on
    **/
    static getNodeInt(on) {
        //    var res: string = ObjectNode.getNodeString(on);
        //    if (res == null) throw new Error("NoSuchElementException");
        //    return java.lang.Integer.parseInt$(res);
        return on.getInt();
    }
    /**
     * float値の取得
     * @param on
    **/
    static getNodeFloat(on) {
        //    var res: string = ObjectNode.getNodeString(on);
        //    if (res == null) throw new Error("NoSuchElementException");
        //    return java.lang.Float.parseFloat$(res);
        return on.getFloat();
    }
    ///**
    // * double値の取得
    // * @param on
    //**/
    //static getNodeDouble(on: ObjectNode): number {
    //    var res: string = ObjectNode.getNodeString(on);
    //    return java.lang.Double.parseDouble(res);
    //}
    /**
     * boolの取得
     * 存在しない場合は0を返す
     * @param on
    **/
    static getNodeBool(on) {
        //    var res: string = ObjectNode.getNodeString(on).trim();
        //    if (res.equalsIgnoreCase("1")) return true;
        //    if (res.equalsIgnoreCase("0")) return false;
        //    throw new Error("IllegalStateException");
        return on.getBool();
    }
    /**
     * ノードオブジェクトがStringか確認
     * @param on
     * @return
     * @author kawae
     * @since 2004/11/05
    **/
    static isStringObject(on) {
        return on && on.isStringObject();
    }
    /*
     * ノードオブジェクトがintか確認
     * @param on
     * @return
     * @author kawae
     * @since 2004/11/05
     **/
    static isIntObject(on) {
        return on && on.isIntObject();
    }
    /**
     * オブジェクトがBooleanか確認
     * 文字列かつ"1"or"0"か確認
     * @param on
     * @return
     * @author kawae
     * @since 2004/11/05
     */
    static isBoolObject(on) {
        return on && on.isBoolObject();
    }
    /**
     * ノードオブジェクトがfloatか確認
     * @param on
     * @return
     * @author kawae
     * @since 2004/11/05
     **/
    static isFloatObject(on) {
        return on && on.isFloatObject();
    }
}
function parseIntStrict(number) {
    var reOut = /^([-+]?(\d+))$/.exec(number);
    return reOut ? parseInt(reOut[1]) : NaN;
}
function parseFloatStrict(number) {
    var reOut = /^([-+]?(\d+\.?\d*|\d*\.?\d+)([eE][-+]?[0-9]+)?)$/.exec(number);
    return reOut ? parseFloat(reOut[1]) : NaN;
}
//# sourceMappingURL=ObjectNode.js.map