import { KPoint2d } from '../vecmath/KPoint2d';
import { KFont } from '../kwt/KFont';
import { IRichGraphicsConst } from '../kwt/IRichGraphics';
import { KLineType } from '../kwt/KLineType';
import { RichGraphicsUtil } from '../kwt/RichGraphicsUtil';
import { K2DFigureType } from '../geom/K2DFigureType';
import { PdfGraphics } from './PdfGraphics';
import { StringBuilder } from './util';
//module option.pdf
/**
 * IRichGraphics を使用して、PdfCommandGraphics への描画を行うアダプタ。
 * @since		2005/02/23
 * @author		fujita
 * @version	2005/02/23Copyright (c) 2005 KITec Inc,.. All rights reserved.
**/
export class PdfRichGraphicsAdapter {
    /**
     * デフォルトの内容を持つ新たな PdfRichGraphicsAdapter を作成する。
     * @param g	PDF グラフィックス
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    constructor(g) {
        /**
         * ポリゴン等の描画用の x 座標のバッファ **/
        this.XBUF = new Array(PdfRichGraphicsAdapter.BUF_SIZE);
        /**
         * ポリゴン等の描画用の y 座標のバッファ **/
        this.YBUF = new Array(PdfRichGraphicsAdapter.BUF_SIZE);
        this.pdfPathMode = false;
        this.mGraphics = g;
    }
    scale(scale) {
        this.mScale = scale;
    }
    getScale() {
        return this.mScale;
    }
    /**
     * 指定された flag により、フォントをファイルに 埋め込むかどうかを指定する。
     * @param flag	フォントを埋め込む場合、true。そうでない場合は、false
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    setFontEmbed(flag) {
        this.mGraphics.setFontEmbed(flag);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#setClipArea(double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    setClipArea(x1, y1, x2, y2) {
        this.mGraphics.setClip(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1));
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#releaseClipArea()
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    releaseClipArea() {
        this.mGraphics.releaseClip();
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#setColor(java.awt.Color)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    setColor(c) {
        this.mGraphics.setColor(c);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawLine(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawLine(x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawLine(x1, y1, x2, y2);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawCircle(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawCircle(xc1, yc1, ra, ss, ee, lt) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - ra, yc1 + ra, ra * 2, ra * 2, ss, ee, false);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillCircle(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawFillCircle(xc1, yc1, ra, ss, ee, lt) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - ra, yc1 + ra, ra * 2, ra * 2, ss, ee, true);
    }
    /**
     * 指定された内容をもとに、楕円の弧を描く。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawEllipse(xc1, yc1, v, h, ss, ee, lt) {
        this.drawEllipse_(xc1, yc1, v, h, ss, ee, lt, false);
    }
    /**
     * 指定された内容をもとに、楕円の弧を描き、 その内部を塗りつぶす。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawFillEllipse(xc1, yc1, v, h, ss, ee, lt) {
        this.drawEllipse_(xc1, yc1, v, h, ss, ee, lt, true);
    }
    /**
     * 指定された内容をもとに、楕円の弧を描く。
     * @param xc1		楕円中心の x 座標
     * @param yc1		楕円中心の y 座標
     * @param v		楕円の幅
     * @param h		楕円の高さ
     * @param ss		楕円の開始角度[°]
     * @param ee		楕円の終了角度[°]
     * @param lt		線種
     * @param fill	楕円を塗りつぶす場合、true。そうでない場合は、false
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawEllipse_(xc1, yc1, v, h, ss, ee, lt, fill) {
        this.setLineType(lt);
        if (Math.abs(ss - ee) < PdfRichGraphicsAdapter.EPS) {
            ss = 0;
            ee = 360;
        }
        this.mGraphics.drawArc_(xc1 - h, yc1 + v, h * 2, v * 2, ss, ee, fill);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawRect(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawRect(x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), false);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillRect(double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawFillRect(x1, y1, x2, y2, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), true);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawRoundRect(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawRoundRect(x1, y1, x2, y2, round, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRoundRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), round * 2, round * 2, false);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillRoundRect(double,double,double,double,double,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawFillRoundRect(x1, y1, x2, y2, round, lt) {
        this.setLineType(lt);
        this.mGraphics.drawRoundRect(Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1), round * 2, round * 2, true);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawPolygon(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawPolygon(poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        this.mGraphics.drawPolygon(this.XBUF, this.YBUF, poly.length);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawFillPolygon(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawFillPolygon(poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        this.mGraphics.fillPolygon(this.XBUF, this.YBUF, poly.length);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawPolyLine(com.kt.util.RefList,jp.kitec.kwt.KLineType)
     * @since		2005/02/23
     * @author		fujita
     * @version	2005/02/23
    **/
    drawPolyLine(poly, lt) {
        this.setLineType(lt);
        this.createFloatPointArrayBuffer(poly);
        for (let i = 0; i < poly.length - 1; i++) {
            this.drawLine(this.XBUF[i], this.YBUF[i], this.XBUF[i + 1], this.YBUF[i + 1], lt);
        }
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,java.lang.String,double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawImageId(img, id, x1, y1, x2, y2) {
        this.drawImagePosition(img, x1, y1, x2, y2);
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,double,double,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    drawImagePosition(img, x1, y1, x2, y2) {
        this.mGraphics.drawImageRect(img, Math.min(x1, x2), Math.max(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1));
    }
    drawImageRect(img, cx, cy, width, height, angle) {
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawText(java.lang.String,double,double,java.lang.String,int,double,double)
     * @since		2004/09/29
     * @author		kawae
     * @version	2005/02/23	fujita文字の回転を反映するよう変更。
    **/
    drawText(s, x1, y1, fontname, fonttype, size, angle) {
        if (s == null || s.length == 0) {
            return;
        }
        this.mGraphics.setFontParam(fontname, fonttype, size);
        let f = KFont.getFont(fontname, fonttype, size);
        this.mGraphics.drawString(s, x1, y1, false, -angle, f.getFontMetrics(null).stringWidth(s), PdfGraphics.SPACE_FOR_PITCH);
    }
    ///**
    // * @see jp.kitec.kwt.IRichGraphics#drawShadePolygon(java.util.Vector,double,double,double,double,java.awt.Color,java.awt.Color)
    // * @since		2004/09/29
    // * @author		kawae
    // * @version	2004/09/29
    //**/
    //drawShadePolygon(polygon: Array<RefList<number[]>>, x1: number, y1: number, x2: number, y2: number, c1: KColor, c2: KColor): void {
    //    let pointsize: number = 0;
    //    let npts: number[] = new Array(polygon.size());
    //    for (let i: number = 0; i < polygon.size(); i++) {
    //        let r: RefList<number[]> = (polygon.get(i)) as RefList;
    //        pointsize += r.size();
    //    }
    //    let x: number[] = new Array(pointsize);
    //    let y: number[] = new Array(pointsize);
    //    let count: number = 0;
    //    for (let i: number = 0; i < polygon.size(); i++) {
    //        let r: RefList<number[]> = (polygon.get(i)) as RefList;
    //        npts[i] = r.size();
    //        for (let j: number = 0; j < r.size(); j++) {
    //            let f: number[] = (r.elementAt(j)) as Array;
    //            x[count] = f[0];
    //            y[count] = f[1];
    //            count++;
    //        }
    //    }
    //    let formatter: NumberFormat = new DecimalFormat("000");
    //    let shadename: StringBuilder = new StringBuilder("TYPE2_");
    //    shadename.append(formatter.format(c1.getRed()));
    //    shadename.append(formatter.format(c1.getGreen()));
    //    shadename.append(formatter.format(c1.getBlue()));
    //    shadename.append(formatter.format(c2.getRed()));
    //    shadename.append(formatter.format(c2.getGreen()));
    //    shadename.append(formatter.format(c2.getBlue()));
    //    this.mGraphics.drawShadingGroupPolygon(shadename.toString(), x, y, npts);
    //}
    /**
     * 線種を設定する。
     * @param lt	線種
     * @since		2004/09/29
     * @author		kawae
     * @version	2004/09/29
    **/
    setLineType(lt) {
        if (lt != null) {
            this.mGraphics.setLineWidth(lt.width);
            if (lt.stroke != null)
                this.mGraphics.setLinePattern(lt.stroke);
            else
                this.mGraphics.setLinePattern(PdfRichGraphicsAdapter.NULL_LINE.stroke);
        }
        else {
            this.mGraphics.setLineWidth(PdfRichGraphicsAdapter.NULL_LINE.width);
            this.mGraphics.setLinePattern(PdfRichGraphicsAdapter.NULL_LINE.stroke);
        }
    }
    /**
     * コレクションノードを配列に直す
     * @param pts
     * @author kawae
     * @since 2004/09/29
    **/
    createFloatPointArrayBuffer(pts) {
        if (pts.length > this.XBUF.length) {
            this.XBUF = new Array(pts.length);
            this.YBUF = new Array(pts.length);
        }
        for (let i = 0; i < pts.length; i++) {
            this.XBUF[i] = pts[i].getX();
            this.YBUF[i] = pts[i].getY();
        }
    }
    /**
     * @see jp.kitec.kwt.IRichGraphics#drawImage(java.awt.Image,java.lang.String,java.lang.String,double,double,double,double)
     * @param img
     * @param prefix
     * @param id
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @since 2006/01/24
     * @author kamimura
     * @version 2006/01/24
    **/
    drawImagePrefixId(img, prefix, id, x1, y1, x2, y2) {
        this.drawImagePosition(img, x1, y1, x2, y2);
    }
    pathInit(i) {
        this.mCurrentPoint = null;
        switch (i) {
            case IRichGraphicsConst.WIND_EVEN_ODD:
                this.mWindingRule = 0;
                break;
            case IRichGraphicsConst.WIND_NON_ZERO:
                this.mWindingRule = 1;
                break;
        }
        this.mGraphics.draw("[multiline]");
        this.mGraphics.draw("path,start");
        this.pdfPathMode = true;
    }
    pathClear() {
        this.mCurrentPoint = null;
        this.mGraphics.draw("path,end");
        this.mGraphics.draw("[multiline]");
        this.pdfPathMode = false;
    }
    pathMoveTo(x, y, la) {
        let xx = this.getGlobalX(x, y, la);
        let yy = this.getGlobalY(x, y, la);
        if (isNaN(xx) || isNaN(yy))
            throw new Error();
        if (!isFinite(xx) || !isFinite(yy))
            throw new Error();
        let p = this.mCurrentPoint;
        if (p != null && RichGraphicsUtil.isSame2D(xx, yy, p.getX(), p.getY()))
            return;
        let sb = new StringBuilder("moveto");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.updateCurrentPoint(xx, yy);
        this.appendPoint(sb, xx, yy);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    }
    pathLineTo(x, y, la) {
        let sb = new StringBuilder("lineto");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        let xx = this.getGlobalX(x, y, la);
        let yy = this.getGlobalY(x, y, la);
        if (isNaN(xx) || isNaN(yy))
            throw new Error();
        if (!isFinite(xx) || !isFinite(yy))
            throw new Error();
        this.updateCurrentPoint(xx, yy);
        this.appendPoint(sb, xx, yy);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    }
    pathQuadTo(x1, y1, x2, y2, la) {
    }
    pathCurveTo(x1, y1, x2, y2, x3, y3, la) {
        let xa = this.getGlobalX(x1, y1, la);
        let ya = this.getGlobalY(x1, y1, la);
        let xb = this.getGlobalX(x2, y2, la);
        let yb = this.getGlobalY(x2, y2, la);
        let xc = this.getGlobalX(x3, y3, la);
        let yc = this.getGlobalY(x3, y3, la);
        if (isNaN(xa) || isNaN(ya))
            throw new Error();
        if (!isFinite(xa) || !isFinite(ya))
            throw new Error();
        if (isNaN(xb) || isNaN(yb))
            throw new Error();
        if (!isFinite(xb) || !isFinite(yb))
            throw new Error();
        if (isNaN(xc) || isNaN(yc))
            throw new Error();
        if (!isFinite(xc) || !isFinite(yc))
            throw new Error();
        this.updateCurrentPoint(xc, yc);
        let sb = new StringBuilder("bezier1");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xa, ya);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xb, yb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.appendPoint(sb, xc, yc);
        this.mGraphics.draw(sb.toString());
        if (!this.pdfPathMode) {
            console.warn("pathmode error!");
        }
    }
    getGlobalX(x, y, la) {
        if (la == null) {
            return x;
        }
        else {
            return la.getGlobalX(x, y);
        }
    }
    getGlobalY(x, y, la) {
        if (la == null) {
            return y;
        }
        else {
            return la.getGlobalY(x, y);
        }
    }
    appendPoint(sb, x, y) {
        this.mGraphics.appendPosition(sb, PdfGraphics.SEPARATOR_GRAPHICS, x, y);
    }
    pathClose() {
        this.mGraphics.draw("close");
    }
    pathAdd(col, la) {
        if (col.length < 2)
            return;
        let f1 = col[0];
        let f2 = col[1];
        let x1;
        let y1;
        if (RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX1(), f2.getY1()) || RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX2(), f2.getY2())) {
            x1 = f1.getX2();
            y1 = f1.getY2();
        }
        else {
            x1 = f1.getX1();
            y1 = f1.getY1();
        }
        this.pathMoveTo(x1, y1, la);
        for (let i = 0; i < col.length; i++) {
            f1 = col[i];
            f2 = col[(i + 1) % col.length];
            let startflip = false;
            if (RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX1(), f2.getY1()) || RichGraphicsUtil.isSame2D(f1.getX1(), f1.getY1(), f2.getX2(), f2.getY2()))
                startflip = true;
            let type = f1.getType();
            if (type == K2DFigureType.ARC_CW || K2DFigureType.ARC_CCW) {
                let arc = f1;
                let r = arc.getRadius();
                let cx = arc.getCX();
                let cy = arc.getCY();
                let p1a = RichGraphicsUtil.getAngle2D(cx, cy, arc.getX1(), arc.getY1());
                let p2a = RichGraphicsUtil.getAngle2D(cx, cy, arc.getX2(), arc.getY2());
                let angl = p2a - p1a;
                while (angl < 0)
                    angl += 360;
                if (arc.isCw())
                    angl = -(360 - angl);
                if (startflip)
                    angl = -angl;
                this.pathArc(cx, cy, r, r, startflip ? p2a : p1a, angl, la);
            }
            else {
                if (startflip) {
                    this.pathLineTo(f1.getX1(), f1.getY1(), la);
                }
                else {
                    this.pathLineTo(f1.getX2(), f1.getY2(), la);
                }
            }
        }
    }
    pathArc(cx, cy, r, h, startAngle, arcAngle, la) {
        let div = 4;
        let width = r * 2.0;
        let height = h * 2.0;
        let ath = arcAngle * Math.PI / 180.0;
        let theta = ath / div;
        let cos = Math.cos(theta);
        let sin = Math.sin(theta);
        let x = new Array(4);
        let y = new Array(4);
        x[0] = x[1] = 1.0;
        y[0] = 0.0;
        x[3] = cos;
        y[3] = sin;
        x[2] = (8.0 * Math.cos(theta / 2) - (x[0] + 3 * x[1] + x[3])) / 3;
        y[2] = y[3] - (-x[2] + x[3]) / (-Math.tan(theta));
        y[1] = (8.0 * Math.sin(theta / 2) - (y[0] + 3 * y[2] + y[3])) / 3;
        let stheta = startAngle * Math.PI / 180.0;
        let scos = Math.cos(stheta);
        let ssin = Math.sin(stheta);
        let xtmp = new Array(4);
        let ytmp = new Array(4);
        for (let j = 0; j < 4; j++) {
            xtmp[j] = x[j] * scos - y[j] * ssin;
            ytmp[j] = x[j] * ssin + y[j] * scos;
            x[j] = xtmp[j];
            y[j] = ytmp[j];
        }
        let aspect = height / width;
        if (isNaN(aspect))
            aspect = 1;
        this.pathMoveTo(r * x[0] + cx, r * y[0] * aspect + cy, la);
        for (let i = 0; i < div; i++) {
            this.pathCurveTo(r * x[1] + cx, r * y[1] * aspect + cy, r * x[2] + cx, r * y[2] * aspect + cy, r * x[3] + cx, r * y[3] * aspect + cy, la);
            for (let j = 1; j < 4; j++) {
                xtmp[j] = x[j] * cos - y[j] * sin;
                ytmp[j] = x[j] * sin + y[j] * cos;
                x[j] = xtmp[j];
                y[j] = ytmp[j];
            }
        }
    }
    fillPath(lt) {
        this.setLineType(lt);
        let sb = new StringBuilder("fill");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(this.mWindingRule);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    }
    updateCurrentPoint(xx, yy) {
        if (this.mCurrentPoint == null)
            this.mCurrentPoint = new KPoint2d(xx, yy);
        else
            this.mCurrentPoint.setXY(xx, yy);
    }
    drawPath(lt) {
        this.setLineType(lt);
        let sb = new StringBuilder("stroke");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendColor(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    }
    shadePath(lt, x1, y1, x2, y2, c1, c2, cyclic) {
        let SEP = "_";
        //let fRgb: NumberFormat = new DecimalFormat("000");
        //let fAxis: NumberFormat = new DecimalFormat("00.00");
        let shadename = new StringBuilder("TYPE2");
        shadename.append(SEP);
        shadename.append(formatRgb(c1.getRed()));
        shadename.append(formatRgb(c1.getGreen()));
        shadename.append(formatRgb(c1.getBlue()));
        shadename.append(formatRgb(c2.getRed()));
        shadename.append(formatRgb(c2.getGreen()));
        shadename.append(formatRgb(c2.getBlue()));
        shadename.append(SEP);
        shadename.append(formatAxis(x1));
        shadename.append(":");
        shadename.append(formatAxis(y1));
        shadename.append(":");
        shadename.append(formatAxis(x2));
        shadename.append(":");
        shadename.append(formatAxis(y2));
        shadename.append(SEP);
        shadename.append(cyclic ? "1" : "0");
        let sb = new StringBuilder("shadestroke");
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        sb.append(shadename.toString());
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLineWidth(sb);
        sb.append(PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.appendLinePattern(sb, PdfGraphics.SEPARATOR_GRAPHICS);
        this.mGraphics.draw(sb.toString());
    }
    //pathArc(xc: number, yc: number, ra: number, startangl: number, angl: number, la: KLocalAxis): void {
    //    this.pathArc(xc, yc, ra, ra, startangl, angl, la);
    //}
    drawMark(x, y, r, type) {
    }
    getDrawArea() {
        return null;
    }
    getDotPerLen() {
        return 0;
    }
    setQuality(quality) {
    }
    getQuality() {
        return false;
    }
}
/**
 * ポリゴン等の描画用のバッファのサイズ **/
PdfRichGraphicsAdapter.BUF_SIZE = 128;
/**
 * 線の属性が指定されていない（null）の場合の線種 **/
PdfRichGraphicsAdapter.NULL_LINE = new KLineType("", [0, 0], 0.1, KLineType.TYPE_DRAWING);
PdfRichGraphicsAdapter.EPS = 1.0e-3;
PdfRichGraphicsAdapter.D2R = Math.PI / 180;
function formatRgb(n) {
    let s = n.toFixed(0);
    while (s.length < 3)
        s = "0" + s;
    return s;
}
function formatAxis(n) {
    return n.toFixed(2);
}
//# sourceMappingURL=PdfRichGraphicsAdapter.js.map