import { error } from '../common';
//module jp.kitec.kwt
/**
 * 線種指定クラス
 * @author kawae
**/
export class KLineType {
    /**
     * コンストラクタ
     * @param name
     * @param str
     * @param w
     * @param widType
    **/
    constructor(name = "", str = null, w = 1, widType = KLineType.TYPE_DRAWING, cap = KLineType_Cap.BUTT, join = KLineType_Join.MITER, miterLimit = 1) {
        /**
         * 幅 **/
        this.mWidth = 0;
        if (cap == null || join == null)
            throw new error.lang.NullPointerException();
        this.mName = name;
        this.mStroke = str;
        this.mWidth = w;
        this.mWidthType = widType;
        this.mCap = cap;
        this.mJoin = join;
        this.mMiterLimit = miterLimit;
    }
    static get GEOMNAME() { return "LINETYPE"; }
    /**
     * 実座標系での幅指定 **/
    static get TYPE_DRAWING() { return 0; }
    /**
     * デバイス座標系での幅指定 **/
    static get TYPE_DEVICE() { return 1; }
    /**
     * 名称を返す
     * @return
    **/
    get name() {
        return this.mName;
    }
    /**
     * ストロークを返す
     * @return
    **/
    get stroke() {
        return this.mStroke;
    }
    /**
     * 幅を返す
     * @return
    **/
    get width() {
        return this.mWidth;
    }
    /**
     * 幅指定オプションを返す
     * @return
    **/
    get widthType() {
        return this.mWidthType;
    }
    /**
     * 両端部オプションを返す
     * @return
    **/
    get cap() {
        return this.mCap;
    }
    /**
     * 接合部オプションを返す
     * @return
    **/
    get join() {
        return this.mJoin;
    }
    /**
     * 接合トリミングの制限を返す
     * @return
    **/
    get miterLimit() {
        return this.mMiterLimit;
    }
    /**
     * 現在のKLineTypeに新しいストロークを適用した複製を返す
     * @param stroke
     * @return
    **/
    deriveStroke(stroke) {
        return new KLineType(this.mName, stroke, this.mWidth, this.mWidthType, this.mCap, this.mJoin, this.mMiterLimit);
    }
    /**
     * 現在のKLineTypeに新しい幅を適用した複製を返す
     * @param width
     * @return
    **/
    deriveWidth(width) {
        return new KLineType(this.mName, this.mStroke, width, this.mWidthType, this.mCap, this.mJoin, this.mMiterLimit);
    }
    /**
     * 現在のKLineTypeに新しい幅指定オプションを適用した複製を返す
     * @param widthType
     * @return
    **/
    deriveWidthType(widthType) {
        return new KLineType(this.mName, this.mStroke, this.mWidth, widthType, this.mCap, this.mJoin, this.mMiterLimit);
    }
    /**
     * 現在のKLineTypeに新しい両端部オプションを適用した複製を返す
     * @param cap
     * @return
    **/
    deriveCap(cap) {
        return new KLineType(this.mName, this.mStroke, this.mWidth, this.mWidthType, cap, this.mJoin, this.mMiterLimit);
    }
    /**
     * 現在のKLineTypeに新しい接合部オプションを適用した複製を返す
     * @param join
     * @return
    **/
    deriveJoin(join) {
        return new KLineType(this.mName, this.mStroke, this.mWidth, this.mWidthType, this.mCap, join, this.mMiterLimit);
    }
    /**
     * 現在のKLineTypeに新しい接合トリミング制限を適用した複製を返す
     * @param miterLimit
     * @return
    **/
    deriveMiterLimit(miterLimit) {
        return new KLineType(this.mName, this.mStroke, this.mWidth, this.mWidthType, this.mCap, this.mJoin, miterLimit);
    }
    save(savenode) {
        savenode.setName(KLineType.GEOMNAME);
        if (this.mName != null)
            savenode.addNode("name", this.mName);
        if (this.mStroke != null) {
            var sb = "";
            for (var i = 0; i < this.mStroke.length; i++) {
                sb += this.mStroke[i];
                if (i != this.mStroke.length - 1)
                    sb += ",";
            }
            var s = sb;
            if (s.length > 0)
                savenode.addNode("stroke", s);
        }
        savenode.addNode("width", this.mWidth);
        savenode.addNode("cap", KLineType_Cap[this.mCap]);
        savenode.addNode("join", KLineType_Join[this.mJoin]);
        savenode.addNode("miterlimit", this.mMiterLimit);
    }
    static readData(of, l) {
        var l = l || new KLineType(null, null, 0);
        var on;
        if ((on = of.getNode("name")) != null)
            l.mName = on.getString();
        if ((on = of.getNode("stroke")) != null) {
            var str = on.getString();
            var st = str.split(",");
            l.mStroke = new Array(st.length);
            for (var i = 0; i < l.mStroke.length; i++)
                l.mStroke[i] = parseFloat(st[i]);
        }
        if ((on = of.getNode("width")) != null)
            l.mWidth = on.getFloat();
        if ((on = of.getNode("cap")) != null)
            l.mCap = KLineType_Cap[on.getString()];
        if ((on = of.getNode("join")) != null)
            l.mJoin = KLineType_Join[on.getString()];
        if ((on = of.getNode("miterlimit")) != null)
            l.mMiterLimit = on.getFloat();
        return l;
    }
}
export var KLineType_Cap;
(function (KLineType_Cap) {
    //enum constant declarations
    KLineType_Cap[KLineType_Cap["BUTT"] = 0] = "BUTT";
    KLineType_Cap[KLineType_Cap["ROUND"] = 1] = "ROUND";
    KLineType_Cap[KLineType_Cap["SQUARE"] = 2] = "SQUARE";
})(KLineType_Cap || (KLineType_Cap = {}));
export var KLineType_Join;
(function (KLineType_Join) {
    //enum constant declarations
    KLineType_Join[KLineType_Join["BEVEL"] = 0] = "BEVEL";
    KLineType_Join[KLineType_Join["MITER"] = 1] = "MITER";
    KLineType_Join[KLineType_Join["ROUND"] = 2] = "ROUND";
})(KLineType_Join || (KLineType_Join = {}));
//# sourceMappingURL=KLineType.js.map