import { XMLReader } from './XMLReader';
//module jp.kitec.lib.util
function equal(expect, actual, eps) {
    QUnit.ok(Math.abs(expect - actual) <= eps);
}
/**
 * コメントを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXML01()", function () {
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <!-- コメントのみの行  -->
  <bbb value="a" />
  <!-- 行頭のコメント  --><bbb value="b" />
  <bbb value="c" />
  <bbb value="d" /><!-- 行末のコメント -->
  <bbb value="e" />
  <!-- 行頭と行末のコメント1  --><bbb value="f" /><!-- 行頭と行末のコメント2  -->
  <bbb value="g" />
</aaa>`;
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    for (let i = 0; i < 14; i++)
        console.log(i + ":" + testXMLreader.readLine());
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 連続したコメントを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXML02()", function () {
    //String testFilePath = XML_PATH_02;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <!-- コメントのみの行1  -->  <!-- コメントのみの行2  -->  <!-- コメントのみの行3  -->
  <bbb value="a" />
  <!-- 行頭のコメント1  --><!-- 行頭のコメント2  --><!-- 行頭のコメント3  --><bbb value="b" />
  <bbb value="c" />
  <bbb value="d" /><!-- 行末のコメント1 --><!-- 行末のコメント2 --><!-- 行末のコメント3 -->
  <bbb value="e" />
  <!-- 行頭と行末のコメント1  --><!-- 行頭と行末のコメント2  --><!-- 行頭と行末のコメント3  --><bbb value="f" /><!-- 行頭と行末のコメント4  --><!-- 行頭と行末のコメント5  --><!-- 行頭と行末のコメント6  -->
  <bbb value="g" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 改行を含むコメントを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXML03()", function () {
    //String testFilePath = XML_PATH_03;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <!-- コメントのみの行1
   コメントのみの行2
   コメントのみの行3 -->
  <bbb value="a" />
  <!-- 行頭のコメント1
    行頭のコメント2
    行頭のコメント3 --><bbb value="b" />
  <bbb value="c" />
  <bbb value="d" /><!-- 行末のコメント1
  行末のコメント2
  行末のコメント3 -->
  <bbb value="e" />
  <!-- 行頭と行末のコメント1
  行頭と行末のコメント2
  行頭と行末のコメント3 --><bbb value="f" /><!-- 行頭と行末のコメント4
  行頭と行末のコメント5
  行頭と行末のコメント6 -->
  <bbb value="g" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 1行に開始タグと終了タグがあるXMLの読み込み<br/>
 */
QUnit.test("testReadXML04()", function () {
    //String testFilePath = XML_PATH_04;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb> value="c" </bbb>
  <bbb value="d" />
  <bbb value="e" />
  <bbb value="f" />
  <bbb value="g" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb> value=\"c\" </bbb>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 1行に終了タグが複数存在するXMLの読み込みテスト<br/>
 * 読み込みに失敗する<br/>
 * 現状としては、この動作を許容している<br/>
 */
QUnit.test("testReadXML05()", function () {
    //String testFilePath = XML_PATH_05;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb> value="c" </bbb><bbb> value="d" </bbb>
  <bbb value="e" />
  <bbb value="f" />
  <bbb value="g" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb> value=\"c\" </bbb><bbb> value=\"d\" </bbb>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 1行に空要素タグが複数存在するXMLの読み込みテスト<br/>
 * 読み込みに失敗する<br/>
 * 現状としては、この動作を許容している<br/>
 */
QUnit.test("testReadXML06()", function () {
    //String testFilePath = XML_PATH_06;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" /><bbb value="d" />
  <bbb value="e" />
  <bbb value="f" />
  <bbb value="g" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\" /><bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"e\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"f\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"g\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 左側括弧が欠落したコメントの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid01()", function () {
    //String testFilePath = XML_PATH_INVALID_01;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  !-- 左側括弧が欠落したコメント-->
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.notEqual(null, testXMLreader.readXML());
});
/**
 * 左側ハイフンが欠落したコメントの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid02()", function () {
    //String testFilePath = XML_PATH_INVALID_02;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <!- 左側ハイフンが欠落したコメント -->
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.notEqual(null, testXMLreader.readXML());
});
/**
 * 右側括弧が欠落したコメントの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid03()", function () {
    //String testFilePath = XML_PATH_INVALID_03;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <!-- 右側括弧が欠落したコメント --
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 右側ハイフンが欠落したコメントの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid04()", function () {
    //String testFilePath = XML_PATH_INVALID_04;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <!-- 右側ハイフンが欠落したコメント ->
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 右側にスラッシュを誤記したコメントの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid05()", function () {
    //String testFilePath = XML_PATH_INVALID_05;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <!-- 右側にスラッシュを誤記したコメント --/>
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 開始タグの欠落したXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid06()", function () {
    //String testFilePath = XML_PATH_INVALID_06;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 開始タグの左側括弧が欠落したXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid07()", function () {
    //String testFilePath = XML_PATH_INVALID_07;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 開始タグの右側括弧が欠落したXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid08()", function () {
    //String testFilePath = XML_PATH_INVALID_08;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.notEqual("<aaa", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 終了タグが欠落したXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid09()", function () {
    //String testFilePath = XML_PATH_INVALID_09;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 左側括弧が欠落した終了タグを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid10()", function () {
    //String testFilePath = XML_PATH_INVALID_10;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
/aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("/aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 右側括弧が欠落した終了タグを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid11()", function () {
    //String testFilePath = XML_PATH_INVALID_11;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</aaa`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 開始タグと終了タグが一致しないXMLの読み込みテスト<br/>
 * 開始タグ名のObjectFolderが生成される<br/>
 */
QUnit.test("testReadXMLInvalid12()", function () {
    //String testFilePath = XML_PATH_INVALID_12;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</ccc>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("</ccc>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 左側括弧が欠落した空要素タグを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid13()", function () {
    //String testFilePath = XML_PATH_INVALID_13;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  bbb value="b" />
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("bbb value=\"b\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * 右側括弧が欠落した空要素タグを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid14()", function () {
    //String testFilePath = XML_PATH_INVALID_14;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" /
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"b\"  /", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.notEqual("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.notEqual("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
/**
 * スラッシュが欠落した空要素タグを含むXMLの読み込みテスト<br/>
 */
QUnit.test("testReadXMLInvalid15()", function () {
    //String testFilePath = XML_PATH_INVALID_15;
    const content = `<?xml version="1.0" encoding="UTF-8" ?>
<aaa>
  <bbb value="a" />
  <bbb value="b" >
  <bbb value="c" />
  <bbb value="d" />
</aaa>`;
    //log.debug("testFilePath[" + testFilePath + "]");
    var testXMLreader = new XMLReader(content);
    testXMLreader.open();
    QUnit.equal("<aaa>", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"a\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"b\">", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"c\"  />", testXMLreader.readXML());
    QUnit.equal("<bbb value=\"d\"  />", testXMLreader.readXML());
    QUnit.equal("</aaa>", testXMLreader.readXML());
    QUnit.equal(null, testXMLreader.readXML());
});
//QUnit.test("testRead()", function () {
//    const content =
//        `<?xml version="1.0" encoding="UTF-8" ?>
//<aaa>
//  <!-- コメントのみの行  -->
//  <bbb value="a" />
//  <!-- 行頭のコメント  --><bbb value="b" />
//  <bbb value="c" />
//  <bbb value="d" /><!-- 行末のコメント -->
//  <bbb value="e" />
//  <!-- 行頭と行末のコメント1  --><bbb value="f" /><!-- 行頭と行末のコメント2  -->
//  <bbb value="g" />
//</aaa>`;
//    var testXMLreader = new XMLReader(content);
//    var of = testXMLreader.read();
//    console.log(JSON.stringify(of));
//    QUnit.ok(of);
//}); 
//# sourceMappingURL=XMLReaderTest.js.map