import { error } from '../common';
//module jp.kitec.lib.util
/**
 * 履歴を管理する。<BR>
 * @since 　2002/12/20
 * @author　Kawae
 * @version 2002/12/20
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
**/
export class History {
    /**
     * 指定された数の履歴を保存する新たな履歴を構築する。
     * @param hisnum	履歴の最大保存数
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    constructor(hisnum, converter) {
        /**
         * 保存する履歴の最大数 **/
        this._maxHistoryNum = 8;
        this.setMaxHistory(hisnum);
        this._history = [];
        this._current = null;
        this._converter = converter;
    }
    /**
     * 指定された数を、履歴の最大保存数として設定する。
     * @param num	履歴の最大保存数
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
     * @throws IllegalArgumentException
    **/
    setMaxHistory(num) {
        if (num < 0)
            throw new error.lang.IllegalArgumentException();
        this._maxHistoryNum = num;
    }
    /**
     * 現在、設定されている履歴の最大保存数を返す。
     * @return	履歴の最大保存数
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getMaxHistory() {
        return this._maxHistoryNum;
    }
    /**
     * 現在、保存されているすべての履歴を削除する。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    flushHistory() {
        this._current = null;
        this._history.length = 0;
    }
    /**
     * 現在、保存されている履歴を数を返す。
     * @return 履歴の数
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getHistorySize() {
        return this._history.length;
    }
    /**
     * 新たに履歴を追加する。
     * 保存している履歴の数が、履歴の最大保存数を超えた場合、
     * 古い履歴は、順次削除される。
     * @param o 追加する履歴
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    addHistory(o) {
        var b = this._converter.writeHistory(o);
        if (b == null)
            return;
        if (this._current != null) {
            var num = this._history.indexOf(this._current);
            if (num >= 0 && num < this._history.length - 1) {
                while (num + 1 <= this._history.length - 1)
                    this._history.pop();
            }
        }
        this._current = b;
        this._history.push(b);
        while (this._history.length > this._maxHistoryNum)
            this._history.shift();
    }
    /**
     * 保存されている履歴からアンドゥを行えるか否か判定する。
     * 保存されている履歴が１つ以上存在する、かつ前回のアンドゥ、
     * またはリドゥなどで処理された履歴が、最古の履歴でない場合のみ
     * アンドゥ可能となる。
     * @return アンドゥできる場合、true。できない場合は、false
     * @since   2010/04/08
     * @author  Kawae
    **/
    hasUndo() {
        if (this._current == null)
            return false;
        if (this._history.length == 0)
            return false;
        var num = this._history.indexOf(this._current);
        return num > 0;
    }
    /**
     * 保存されている履歴からリドゥを行えるか否か判定する。
     * 保存されている履歴が１つ以上存在する、かつ前回のアンドゥ、
     * またはリドゥなどで処理された履歴が、最新の履歴でない場合のみ
     * リドゥ可能となる。
     * @return リドゥできる場合、true。できない場合は、false
     * @since   2010/04/08
     * @author  Kawae
     * @version 2002/12/20
    **/
    hasRedo() {
        if (this._current == null)
            return false;
        var num = this._history.indexOf(this._current);
        return num >= 0 && num < this._history.length - 1;
    }
    /**
     * 現在、保存されている履歴から最新の履歴を返す。
     * @return 最新の履歴。履歴が存在しない場合は、null
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getLastHistory() {
        if (this._history.length == 0)
            return null;
        return this._history[this._history.length - 1];
    }
    /**
     * 現在の履歴を返す。
     * @return 現在の履歴。
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    getCurrentHistory() {
        return this._current;
    }
    /**
     * 現在の履歴の１世代前の履歴を返す。
     * @return １世代前の履歴。
     * @see #redo()
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    undo() {
        if (this._current == null)
            return null;
        var num = this._history.indexOf(this._current);
        if (num < 0)
            return null;
        if (num > 0) {
            this._current = this._history[num - 1];
            return this._converter.readHistory(this._current);
        }
        return null;
    }
    /**
     * 現在の履歴の１世代後の履歴を返す。
     * @return １世代後の履歴。
     * @see #undo()
     * @since   2002/12/20
     * @author  Kawae
     * @version 2002/12/20
    **/
    redo() {
        if (this._current == null)
            return null;
        var num = this._history.indexOf(this._current);
        if (num < 0)
            return null;
        if (num >= 0 && num < this._history.length - 1) {
            this._current = this._history[num + 1];
            return this._converter.readHistory(this._current);
        }
        return null;
    }
    getConverter() {
        return this._converter;
    }
}
//# sourceMappingURL=History.js.map