package nethome5.geom.primitive;

import java.io.Serializable;

import jp.kitec.lib.kwt.IRichGraphics;
import jp.kitec.lib.kwt.KArea;
import jp.kitec.lib.kwt.KColor;
import jp.kitec.lib.kwt.KLineType;
import jp.kitec.lib.util.RefList;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import nethome5.geom.AreaControl;
import nethome5.geom.GeomName;
import nethome5.geom.GeomParser;
import nethome5.geom.LocalAxis2Df;
import nethome5.geom.Ltype;

/**
 * すべての幾何オブジェクトの基底クラス
 *
 * @author kawae
 */
public abstract class Geometry implements Serializable {

	/** コントロールノードのステータス */
	/** 通常移動ステータス */
	public static final short CTRL_NODE_STATE_DEF = 0;
	/** フリー移動ステータス */
	public static final short CTRL_NODE_STATE_FREE = 1;

	/** 幾何ID */
	protected int mGid;

	/** 線種 */
	protected Ltype lineType = null;

	/** 形状色 */
	protected KColor mGeomColor = null;

	/** 領域 */
	protected KArea area = new KArea();

	/** 領域のコントロールオブジェクト */
	protected AreaControl areaControl = null;

	/** X方向へのスケーリングの許可 */
	public boolean scaleEnX = false;

	/** Y方向へのスケーリングの許可 */
	public boolean scaleEnY = false;

	/** アスペクト比一定のフラグ */
	public boolean mKeepAspect = false;

	/** スケーリングの基点X */
	public double scaleBaseX = 0;

	/** スケーリングの基点Y */
	public double scaleBaseY = 0;

	/**
	 * コンストラクタ
	 */
	public Geometry() {
	}

	/**
	 * 幾何名の取得
	 *
	 * @return Returns the mGeomName.
	 * @author kawae
	 * @since 2003/01/01
	 */
	public String getGeomName() {
		GeomName name = getClass().getAnnotation(GeomName.class);
		if (name == null)
			return null;
		return name.value();
	}

	/**
	 * 幾何IDの設定
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void setID(int id) {
		mGid = id;
	}

	/**
	 * 幾何IDの取得
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public int getID() {
		return mGid;
	}

	/**
	 * エリアの取得
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public KArea getArea() {
		return area;
	}

	/**
	 * 最大・最小領域の更新
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void updateMinMax(KArea area) {
	}

	/**
	 * エリアコントロールの取得
	 *
	 * @return
	 * @author kawae
	 * @since 2003/01/01
	 */
	public AreaControl getAreaControl() {
		return areaControl;
	}

	/**
	 * コピーインスタンスの取得
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public Geometry copy() {
		return null;
	}

	/**
	 * コピーインスタンスの取得
	 *
	 * 引数インスタンスに対し属性をコピーする
	 * @param g　コピー属性設定対象オブジェクト
	 * @return
	 * @author kawae
	 * @since 2003/01/01
	 */
	protected Geometry copyTo(Geometry g) {
		if (g == null)
			return null;
		g.lineType = lineType;
		g.mGeomColor = mGeomColor;
		g.scaleEnX = scaleEnX;
		g.scaleEnY = scaleEnY;
		g.scaleBaseX = scaleBaseX;
		g.scaleBaseY = scaleBaseY;
		g.mKeepAspect = mKeepAspect;
		g.mGid = mGid;
		return g;
	}

	/**
	 * 一番近い構成点の取得
	 *
	 * @param	x	対象計測点X座標
	 * @param	y	対象計測点Y座標
	 * @author kawae
	 * @since 2003/01/01
	 */
	public GPoint2Df getNearNode(double x, double y) {
		return null;
	}

	/**
	 * 幾何形状までの距離の取得
	 *
	 * @param	x	距離計測点X座標
	 * @param	y	距離計測点Y座標
	 *
	 * @return	幾何形状までの距離。計測不能な場合は負を返す
	 * @author kawae
	 * @since 2003/01/01
	 */
	public double getDistanceToPoint(double x, double y) {
		return -1;
	}

	/**
	 * 幾何データ共通属性データの読み込み
	 *
	 * @param datanode		読み込み対象ノード
	 * @param geom			設定対象幾何インスタンス
	 * @throws Exception
	 * @author kawae
	 * @since 2003/01/01
	 */
	public static void readData(ObjectFolder datanode, Geometry geom, GeomParser ps) throws Exception {
		ObjectNode on;
		if ((on = datanode.getNode("id")) != null)
			geom.mGid = on.getInt();
		if ((on = datanode.getNode("forec")) != null)
			geom.mGeomColor = KColor.getColorFromRGBString(on.getString());
	}

	/**
	 * 幾何データの保存
	 *
	 * @param savenode		保存対象ノード
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		savenode.setName(getGeomName());
		savenode.addNode("id", getID());
		if (mGeomColor != null)
			savenode.addNode("forec", mGeomColor.getRed() + "," + mGeomColor.getGreen() + "," + mGeomColor.getBlue());
	}

	/**
	 * 幾何の移動
	 * @param	x	X方向移動量
	 * @param	y	Y方向移動量
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void offset(double x, double y) {
	}

	/**
	 * 幾何の回転
	 * @param	xc		回転中心X座標
	 * @param	yc		回転中心Y座標
	 * @param	angl	回転角度
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void rotate(double xc, double yc, double angl) {
	}

	/**
	 * 幾何の回転
	 * @param	xc		回転中心X座標
	 * @param	yc		回転中心Y座標
	 * @param	ofsx
	 * @param	ofsy
	 * @param	angl	回転角度
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void rot(double x, double y, double ofsx, double ofsy) {
	}

	/**
	 * X軸方向反転
	 *
	 * @param x		反転軸X座標
	 * @param ofsx
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void flipx(double x, double ofsx) {
	}

	/**
	 * Y軸方向反転
	 *
	 * @param y		反転軸Y座標
	 * @param ofsx
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void flipy(double y, double ofsy) {
	}

	/**
	 * 領域内に幾何が存在するか確認
	 *
	 * @param	minx	領域左下X座標
	 * @param	miny	領域左下Y座標
	 * @param	maxx	領域右上X座標
	 * @param	maxy	領域右上Y座標
	 * @author kawae
	 * @since 2003/01/01
	 */
	public boolean isInner(double minx, double miny, double maxx, double maxy) {
		return false;
	}

	/**
	 * ポリゴン内に存在するか確認
	 *
	 * @param	v	ポリゴンのノードリスト
	 * @author kawae
	 * @since 2003/01/01
	 */
	public boolean isInner(RefList<GPoint2Df> v) {
		return false;
	}

	/**
	 * 形状色の設定
	 *
	 * @param	c	設定する色
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void setGeomColor(KColor c) {
		mGeomColor = c;
	}

	/**
	 * 形状色の取得
	 *
	 * @return		幾何色
	 * @author kawae
	 * @since 2003/01/01
	 */
	public KColor getGeomColor() {
		return mGeomColor;
	}

	/**
	 * 描画
	 *
	 * @param	d	描画インターフェース
	 * @author kawae
	 * @since 2003/01/01
	 */
	public abstract void drawGeom(IRichGraphics d);

	/**
	 * 指定色で描画
	 *
	 * @param	d	描画インターフェース
	 * @param	fore	前景色
	 * @param	back	背景色
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawGeom(IRichGraphics d, KColor fore, KColor back) {
	}

	/**
	 * ローカル座標系での通常描画
	 *
	 * @param d	描画インターフェース
	 * @param la	描画座標
	 * @author kawae
	 * @since 2004/09/24
	 */
	public void drawAbstDevice(IRichGraphics d, LocalAxis2Df la) {
	}

	/**
	 * 最終描画
	 *
	 * @param d	描画インターフェース
	 * @param fore	前景色
	 * @param back	背景色
	 * @param la	描画座標
	 * @author kawae
	 * @since 2004/09/24
	 */
	public void drawAbstDevice(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
	}

	/**
	 * ハイライト描画
	 *
	 * @param	rg	描画インターフェース
	 * @param	c		幾何描画色
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawHighLight(IRichGraphics rg, KColor c, boolean outline) {
	}

	/**
	 * 遷移状態の描画
	 *
	 * @param rg	描画インターフェース
	 * @param lt	描画座標
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawTransition(IRichGraphics rg, KColor col, KLineType lt) {
	}

	/**
	 * タイトルの描画
	 *
	 * @param	d	描画インターフェース
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawTitle(IRichGraphics d) {
	}

	/**
	 * 指定色でタイトルを描画
	 *
	 * @param	d	描画インターフェース
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawTitle(IRichGraphics d, KColor c) {
	}


	/**
	 * ローカル座標系での描画
	 *
	 * @param rg	描画インターフェース
	 * @param fore	前景色
	 * @param back	背景色
	 * @param la	描画座標
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void drawAlias(IRichGraphics rg, KColor fore, KColor back, LocalAxis2Df la) {
	}

	/**
	 * 幾何形状の更新
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	protected void updateShape() {
	}

	/**
	 * 一番近いコントロールポイントの取得
	 * @return Object	一番近いコントロールポイント
	 *
	 * @author kawae
	 * @since 2003/01/01
	 */
	public Object getNearControlPoint(double x, double y) {
		return null;
	}

	/**
	 * コントロールポイントの移動
	 *
	 * @param cp　コントロールポイント
	 * @param x	移動先ｘ
	 * @param y	移動先ｙ
	 * @author kawae
	 * @since 2003/01/01
	 */
	public void moveControlPoint(Object cp, double x, double y) {
	}

	/**
	 * コントロールノードのステータスの取得
	 *
	 * @param o
	 * @return
	 * @author kawae
	 * @since 2003/01/01
	 */
	public short getCtrlNodeStatus(Object o) {
		return CTRL_NODE_STATE_DEF;
	}
}
