package nethome5.geom.primitive;
import jp.kitec.lib.kwt.IRichGraphics;
import jp.kitec.lib.kwt.KArea;
import jp.kitec.lib.kwt.KColor;
import jp.kitec.lib.util.RefList;
import jp.kitec.lib.util.Vector2;
import jp.kitec.lib.util.tree.ObjectFolder;
import nethome5.geom.GeomName;
import nethome5.geom.LocalAxis2Df;
import nethome5.geom.util.ToolMath;
import nethome5.geom.util.ToolMathEx;
import nethome5.geom.util.ToolPolygon;

@GeomName("RECT")
public class GRect extends Geometry implements Primitive, IFillableGeometry {

	/** 点群 */
	protected RefList<GPoint2Df> mPoints;

	/** 線の太さ */
	public double mLineWidth;

	/** 描画用ポイントバッファ */
	protected transient RefList<GPoint2Df> _pbuf;

	/** 塗り潰しフラグ */
	protected boolean mFill = false;

	/** 塗りつぶし色 */
	protected KColor mFillColor;

	/**
	 * コンストラクタ
	 *
	 * @param	x1	始点X座標3
	 * @param	y1	始点X座標
	 * @param	x2	始点X座標
	 * @param	y2	始点X座標
	 * @param	c	描画色
	 */
	public GRect() {
		this(0, 0, 0, 0, false);
	}

	public GRect(double x1, double y1, double x2, double y2, boolean fill) {
		this(x1, y1, x2, y1, x2, y2, x1, y2, fill);
	}

	public GRect(double x1, double y1, double x2, double y2, double x3, double y3, double x4, double y4, boolean fill) {
		mGeomColor = KColor.BLACK;
		mFillColor = KColor.BLACK;
		mFill = fill;
		mLineWidth = 0;
		mPoints = new RefList<GPoint2Df>();
		mPoints.addElement(new GPoint2Df(x1, y1));
		mPoints.addElement(new GPoint2Df(x2, y2));
		mPoints.addElement(new GPoint2Df(x3, y3));
		mPoints.addElement(new GPoint2Df(x4, y4));
		area.maximize();
		updateMinMax(area);
	}

	public GPoint2Df getP1() {
		return mPoints.elementAt(0);
	}
	public GPoint2Df getP2() {
		return mPoints.elementAt(2);
	}

	public void setRect(double x1, double y1, double x2, double y2) {
		double xa, ya, xb, yb;
		xa = x1 < x2 ? x1 : x2;
		ya = y1 < y2 ? y1 : y2;
		xb = x1 > x2 ? x1 : x2;
		yb = y1 > y2 ? y1 : y2;
		for (int i = 0; i < 4; i++) {
			GPoint2Df p = mPoints.elementAt(i);
			p.x = (i == 0 || i == 3) ? xa : xb;
			p.y = (i == 0 || i == 1) ? ya : yb;
		}
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public boolean getSnapNode(double x1, double y1, double gx, double gy, double[] min, double[] res, double eps, short limit) {
		boolean flag = false;
		for (int i = 0; i < mPoints.size(); i++) {
//			GPoint2Df p1 = (GPoint2Df)_points.elementAt(i);
//			GPoint2Df p2 = (GPoint2Df)_points.elementAt(i + 1);
//			boolean r = Snapping.utilSnapNodeToLine(x1, y1, p1, p2, gx, gy, min, res, eps, limit);
			boolean r = false;
			if (r)
				flag = r;
		}
		return flag;
	}

	@Override
	public void getSnapLoop(Vector2<? super GPoint2Df> v) {
		for (int i = 0; i <= mPoints.size(); i++)
			v.addElement(mPoints.elementAt(i));
	}

	@Override
	public void getCenter(GPoint2Df p) {
		double x = 0;
		double y = 0;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			x += p1.x;
			y += p1.y;
		}

		p.x = x / mPoints.size();
		p.y = y / mPoints.size();
	}

	@Override
	public GPoint2Df getNearNode(GPoint2Df p1, GPoint2Df p2, double eps) {
		double len, min = Float.MAX_VALUE;
		GPoint2Df res = null;
		for (int k = 0; k < mPoints.size(); k++) {
			GPoint2Df pa = mPoints.elementAt(k);
			if ((len = ToolMath.getLengthLimitLineToPoint2D(pa.x, pa.y, p1.x, p1.y, p2.x, p2.y)) >= 0 && len < eps && len < min) {
				min = len;
				res = pa;
			}
		}
		return res;
	}

	@Override
	public boolean getNearLine(double x, double y, double eps, GPoint2Df[] pts) {
		double len;
		boolean exist = false;
//		double min = Float.MAX_VALUE;
		for (int k = 0; k < mPoints.size(); k++) {
			GPoint2Df pa = mPoints.elementAt(k), pb = mPoints.elementAt(k + 1);
			if ((len = ToolMath.getLengthLimitLineToPoint2D(x, y, pa.x, pa.y, pb.x, pb.y)) >= 0 && len < eps) {
				pts[0] = pa;
				pts[1] = pb;
//				min = len;
				exist = true;
			}
		}
		return exist;
	}

	@Override
	public GPoint2Df getRotbasePoint() {
		return mPoints.elementAt(0);
	}

	@Override
	public void updateMinMax(KArea area) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p = mPoints.elementAt(i);
			area.updateMinMax(p.x, p.y);
		}
	}

	@Override
	public Geometry copy() {
		return copyTo(new GRect());
	}

	@Override
	protected Geometry copyTo(Geometry g) {
		if (g == null || !(g instanceof GRect))
			return null;
		GRect pl = (GRect)g;
		super.copyTo(pl);
		double x1 = (mPoints.elementAt(0)).x;
		double y1 = (mPoints.elementAt(0)).y;
		double x3 = (mPoints.elementAt(2)).x;
		double y3 = (mPoints.elementAt(2)).y;
		pl.setRect(x1, y1, x3, y3);
		pl.area.maximize();
		pl.setFillMode(mFill);
		pl.setFillColor(mFillColor);
		pl.updateMinMax(pl.area);
		return pl;
	}

	@Override
	public GPoint2Df getNearNode(double x, double y) {
		double min = Float.MAX_VALUE;
		GPoint2Df gp = null;
		double tmp;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p = mPoints.elementAt(i);
			if ((tmp = ToolMath.getLength2D(x, y, p.x, p.y)) < min) {
				gp = p;
				min = tmp;
			}
		}
		return gp;
	}

	@Override
	public double getDistanceToPoint(double x, double y) {
		double min = Float.MAX_VALUE;
		double len;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			GPoint2Df p2 = mPoints.elementAt(i + 1);
			len = ToolMath.getLengthLimitLineToPoint2D(x, y, p1.x, p1.y, p2.x, p2.y);
			if (len >= 0 && len < min)
				min = len;
		}
		return min;
	}

	@Override
	public void offset(double x, double y) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			p1.offset(x, y);
		}
		area.maximize();
		updateMinMax(area);

	}

	@Override
	public void rotate(double xc, double yc, double angl) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			p1.rotate(xc, yc, angl);
		}
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public boolean isInner(double minx, double miny, double maxx, double maxy) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			if (!(p1.x > minx && p1.x < maxx && p1.y > miny && p1.y < maxy))
				return false;
		}
		return true;
	}

	/**
	 * ポリゴン内に存在するか
	 */
	@Override
	public boolean isInner(RefList<GPoint2Df> v) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			GPoint2Df p2 = mPoints.elementAt(i + 1);
			boolean b1 = ToolPolygon.isInnerPolygon(v, p1.x, p1.y);
			boolean b2 = ToolPolygon.isInnerPolygon(v, p2.x, p2.y);
			if (!b1 || !b2)
				return false;
		}
		return true;
	}


	/**
	 * データの保存
	 * @param savenode
	 */
	@Override
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		super.save(savenode);

		for (int i = 0; i < this.mPoints.size(); i++) {
			GPoint2Df p = mPoints.elementAt(i);
			savenode.addNode("x" + (i + 1), p.x);
			savenode.addNode("y" + (i + 1), p.y);
		}
		savenode.addNode("fill", mFill);
		savenode.addNode("width", mLineWidth);
	}


	@Override
	public void drawGeom(IRichGraphics d) {
		drawGeom(d, KColor.BLACK, null);
	}

	@Override
	public void drawGeom(IRichGraphics d, KColor fore, KColor back) {
		if (back == null)
			back = mFillColor;
		if (fore == null)
			fore = mGeomColor;
		drawAbstDevice(d, fore, back, null);
	}


	@Override
	public void drawAlias(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
		drawAbstDevice(d, fore, back, la);
	}

	@Override
	public void drawAbstDevice(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
		if (mFill) {
			if (la != null) {
				if (_pbuf == null)
					_pbuf = new RefList<GPoint2Df>();
				if (_pbuf.size() != mPoints.size()) {
					_pbuf.removeAllElements();
					for (int i = 0; i < mPoints.size(); i++)
						_pbuf.addElement(new GPoint2Df());
				}
				for (int i = 0; i < mPoints.size(); i++) {
					GPoint2Df p1 = mPoints.elementAt(i);
					GPoint2Df p2 = _pbuf.elementAt(i);
					p2.x = la.getGlobalX(p1.x, p1.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
					p2.y = la.getGlobalY(p1.x, p1.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
				}
				d.setColor(back);
				d.drawFillPolygon(_pbuf, null);
			} else {
				d.setColor(back);
				d.drawFillPolygon(mPoints, null);
			}
		}
		d.setColor(fore);
		d.drawFillPolygon(mPoints, null);
	}


	@Override
	public void drawHighLight(IRichGraphics d, KColor c, boolean outline) {
		if (outline) {
			d.setColor(c);
			d.drawPolygon(mPoints, IRichGraphics.LINE_DASH_1);
		}
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			p1.drawHighLight(d, c, false);
		}
	}

	/**
	 * 一番近いコントロールポイントの取得
	 * @return Object	一番近いコントロールポイント
	 */
	@Override
	public Object getNearControlPoint(double x, double y) {
		double min = Double.POSITIVE_INFINITY, len;
		GPoint2Df p = null;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			len = p1.getDistanceToPoint(x, y);
			if (len < min) {
				min = len;
				p = p1;
			}
		}
		return p;
	}


	/**
	 * コントロールポイントの移動
	 * @param cp　コントロールポイント
	 * @param x	移動先ｘ
	 * @param y	移動先ｙ
	 */
	@Override
	public void moveControlPoint(Object cp, double x, double y) {
		GPoint2Df pa, pb, pc;
		GPoint2Df p = (GPoint2Df)cp;
		Object o;
		pa = ((o = mPoints.itemAt(p).nextItem.getObject()) != null && o instanceof GPoint2Df) ? (GPoint2Df)o : null;
		pb = ((o = mPoints.itemAt(o).nextItem.getObject()) != null && o instanceof GPoint2Df) ? (GPoint2Df)o : null;
		pc = ((o = mPoints.itemAt(o).nextItem.getObject()) != null && o instanceof GPoint2Df) ? (GPoint2Df)o : null;
		if (pa != null && pb != null && pb != null) {
			if (!ToolMathEx.isPointPickUporDown(x, y, pa, pb) || !ToolMathEx.isPointPickUporDown(x, y, pb, pc))
				return;
		}
		double len;
		len = ToolMath.getLengthLimitlessLineToPoint2D(x,y, pa.x, pa.y, pb.x, pb.y);
		if (len >= 0 && len < ToolMath.EPS)
			return;
		len = ToolMath.getLengthLimitlessLineToPoint2D(x,y, pb.x, pb.y, pc.x, pc.y);
		if (len >= 0 && len < ToolMath.EPS)
			return;
		double[] _tmpf = new double[2];
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			if (p1 == cp) {
				GPoint2Df pt = mPoints.elementAt(i - 2);
				GPoint2Df p0 = mPoints.elementAt(i - 1);
				GPoint2Df p2 = mPoints.elementAt(i + 1);
				p1.x = x;
				p1.y = y;
				ToolMath.isCrossPointVertLimitlessLineToPoint2D(x, y, pt.x, pt.y, p0.x, p0.y, _tmpf);
				p0.x = _tmpf[0];
				p0.y = _tmpf[1];
				ToolMath.isCrossPointVertLimitlessLineToPoint2D(x, y, pt.x, pt.y, p2.x, p2.y, _tmpf);
				p2.x = _tmpf[0];
				p2.y = _tmpf[1];
				break;
			}
		}
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void flipx(double x, double ofsx) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).flipx(x, ofsx);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void flipy(double y, double ofsy) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).flipy(y, ofsy);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void rot(double x, double y, double ofsx, double ofsy) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).rot(x, y, ofsx, ofsy);

		area.maximize();
		updateMinMax(area);
	}

	/**
	 * 幾何形状の更新
	 */
	@Override
	protected void updateShape() {
		GPoint2Df p1 = mPoints.elementAt(0);
		GPoint2Df p2 = mPoints.elementAt(1);
		GPoint2Df p3 = mPoints.elementAt(2);
		GPoint2Df p4 = mPoints.elementAt(3);
		p2.x = p3.x;
		p2.y = p1.y;
		p4.x = p1.x;
		p4.y = p3.y;
	}

	@Override
	public void setFillMode(boolean b) {
	    mFill = b;
	}

	@Override
	public boolean isFillMode() {
		return mFill;
	}

	@Override
	public KColor getFillColor() {
		return mFillColor;
	}

	@Override
	public void setFillColor(KColor fillColor) {
		mFillColor = fillColor;
	}
}
