package nethome5.text;

import java.io.Serializable;

import jp.kitec.lib.kwt.KColor;
import jp.kitec.lib.kwt.KFont;
import jp.kitec.lib.util.UtilString;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;



/**
 * テキストフォーマット
 * @since 2010/06/28
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
@SuppressWarnings("serial")
public class GenericTextFormat implements ITextFormat, Serializable{

	/** フォント名 */
	private String fontName;
	/**  */
	private boolean italic = false;
//	/** フォントスタイル */
//	private int mFontStyle;

	/** フォントサイズ */
	private double fontSize;

	/** 下線 */
	private boolean underLine = false;

	/** テキスト色 */
	private KColor textColor;

	/** 背景色 */
	private KColor backColor;

	/**
	 * コンストラクタ
	 */
	public GenericTextFormat(){
		fontName = "RyuminPro-Light";
		fontSize = 6;
		textColor = KColor.BLACK;
		backColor = KColor.WHITE;
	}

	/**
	 * getter
	 */
	@Override
	public KColor getBackColor(){
		return backColor;
	}

	@Override
	public KColor getTextColor(){
		return textColor;
	}

	@Override
	public String getFontName(){
		return fontName;
	}

	@Override
	public int getFontStyle(){
		if(italic)
			return KFont.ITALIC;
		return 0;
	}

	@Override
	public double getFontSize(){
		return fontSize;
	}

	@Override
	public boolean isUnderLine(){
		return underLine;
	}

	/**
	 * setter
	 */
	@Override
	public void setBackColor(KColor c){
		backColor = c;
	}

	@Override
	public void setTextColor(KColor c){
		textColor = c;
	}

	@Override
	public void setFontName(String name){
		fontName = name;
	}

	@Override
	public void setFontStyle(int style){
		this.italic = (KFont.ITALIC & style) ==KFont.ITALIC;
	}

	@Override
	public void setFontSize(double size){
		fontSize = size;
	}

	@Override
	public void setUnderLine(boolean state){
		underLine = state;
	}

	/**
	 * テキストフォーマットの指定
	 * @param f
	 * @author kawae
	 * @since 2004/09/22
	 * @see nethome.text.ITextFormat#setFormat(nethome.text.ITextFormat)
	 */
	@Override
	public void setFormat(ITextFormat f) {
		if(f==null)
			return;
		setBackColor(f.getBackColor());
		setTextColor(f.getTextColor());
		setFontName(f.getFontName());
		setFontSize(f.getFontSize());
		setFontStyle(f.getFontStyle());
		setUnderLine(f.isUnderLine());
	}

	/**
	 * @param tf
	 * @return
	 * @author kawae
	 * @since 2004/09/26
	 * @see nethome.text.ITextFormat#isSameTextFormat(nethome.text.ITextFormat)
	 */
	@Override
	public boolean isSameTextFormat(ITextFormat tf) {
		if (UtilString.isDiff(getFontName(), tf.getFontName()))
			return false;
		if (getFontStyle() != tf.getFontStyle())
			return false;
		if (Math.abs(getFontSize() - tf.getFontSize()) > 1e-3)
			return false;
		if (KColor.isDiff(getTextColor(), tf.getTextColor()))
			return false;
		if (KColor.isDiff(getBackColor(), tf.getBackColor()))
			return false;
		if (isUnderLine() != tf.isUnderLine())
			return false;
		return true;
	}

	public static void readData(ObjectFolder node, GenericTextFormat t) throws Exception {
		if (node == null)
			return;
		ObjectNode on;
		if (ObjectNode.isStringObject(on = node.getNode("textfont")))
			t.fontName = ObjectNode.getNodeString(on);
		if (ObjectNode.isDoubleObject(on = node.getNode("textsize")))
			t.fontSize = ObjectNode.getNodeDouble(on);
		if (ObjectNode.isBoolObject(on = node.getNode("underline")))
			t.underLine = ObjectNode.getNodeBool(on);
		if (ObjectNode.isIntObject(on = node.getNode("textstyle")))
			t.setFontStyle(Integer.valueOf((String)on.getObject()).intValue());
		if ((on = node.getNode("textcolor")) != null)
			t.textColor = KColor.decode(ObjectNode.getNodeString(on));
		if ((on = node.getNode("backcolor")) != null)
			t.backColor = KColor.decode(ObjectNode.getNodeString(on));
	}

	/*
	 * @see nethome.text.ITextFormat#save(jp.kitec.lib.util.tree.ObjectFolder)
	 * @since 2010/06/28
	 */
	@Override
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		savenode.setName("TEXTFORMAT");
		ObjectFolder.addNodeDouble(savenode, "textsize", this.fontSize);
		ObjectFolder.addNodeString(savenode, "textfont", this.fontName);
		ObjectFolder.addNodeInt(savenode, "textstyle", this.getFontStyle());
		ObjectFolder.addNodeBool(savenode, "underline", this.underLine);
		if(textColor!=null)
			ObjectFolder.addNodeString(savenode, "textcolor", textColor.encode());

		if(backColor!=null)
			ObjectFolder.addNodeString(savenode, "backcolor", backColor.encode());
	}
}
