package nethome5.doc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import nethome5.geom.ILinkLine;
import nethome5.geom.primitive.Geometry;


/**
 * プロジェクトの基本単位
 *
 * @author kawae
 */
public class DefaultProject<T extends DefaultProject<T>> implements Project {

	/** ヒストリー */
	protected ProjectHistoryManager<T> mHistoryManager;

	/**　アクティブな階 */
	protected Layer mActiveLayer;

	/** レイヤーマネージャのコレクション */
	protected ArrayList<Layer> mLayers;

	/** 保存フラグ */
	protected boolean mSaved;

	/** 各種情報 */
	protected HashMap<Object, Object> mProjectInfo;

	/** リスナー */
	protected ArrayList<ProjectListener> mListeners;

	/**
	 * コンストラクタ
	 */
	public DefaultProject(ProjectHistoryManager<T> history) {
		mLayers = new ArrayList<Layer>();
		mProjectInfo = new HashMap<Object, Object>();
		mHistoryManager = history;
		mHistoryManager.setProject((T)this);
		mListeners = new ArrayList<ProjectListener>();
		setSaved(true);
	}

	/**
	 * データの保存
	 * @return
	 */
	@Override
	public ObjectFolder save() {
		ObjectFolder of = new ObjectFolder("DATA");
		saveProjectInformation(of);
		//データ書き出し
		for (int i = 0; i < getLayers().size(); i++) {
			Layer l = getLayers().get(i);

			ObjectFolder lf = new ObjectFolder("");
			lf.addNode("NAME", l.getName());
			l.save(lf);

			of.addChild(lf, true);
		}
		return of;
	}

	@Override
	public void load(ObjectFolder rootf, DocumentFactory docFactory) {
		Object tmp = rootf.getNode("DATA");
		if (tmp != null && tmp instanceof ObjectFolder) {
			ObjectFolder of = (ObjectFolder)tmp;
			loadProjectInformation(of);
			List<ObjectNode> v = of.getNamedElement("LAYER");
			if (v != null) {
				for (int i = 0; i < v.size(); i++) {
					of = (ObjectFolder)v.get(i);

					Layer lm = loadLayer(of, docFactory);
					addLayer(lm);
				}
			}
		}
	}


	/**
	 * レイヤーノードの読み込み
	 *
	 * @param layerf
	 * @param docFactory
	 * @author kawae
	 * @since 2005/10/07
	 */
	@Override
	public Layer loadLayer(ObjectFolder layerf, DocumentFactory docFactory) {
		return loadLayer(layerf, docFactory, true);
	}


	/**
	 * レイヤーノードの読み込み
	 *
	 * @param layerf
	 * @param docFactory
	 * @author kawae
	 * @since 2005/11/02
	 */
	@Override
	public Layer loadLayer(ObjectFolder layerf, DocumentFactory docFactory, boolean recover) {
		String layerName = (String)layerf.getNode("NAME").getObject();

		Layer l = createLoadLayer(docFactory, layerName);
		for (int j = 0; j < layerf.getChildren().size(); j++) {
			Object o = layerf.getChildren().get(j);
			if (o instanceof ObjectFolder) {
				ObjectFolder geom = (ObjectFolder)o;
				Geometry ng = docFactory.getGeomParser().getGeometry(geom);
				addGeomToLayer(l, ng);
			}
		}
		if (recover)
			recoverAfterLoad(l);
		return l;
	}


	/**
	 * ロード時のLayer
	 * @param app
	 * @param layerName
	 * @return
	 * @author kawae
	 * @since 2004/12/13
	 */
	protected Layer createLoadLayer(DocumentFactory app, String layerName) {
		Layer lm = app.createLayer(this, layerName);
		return lm;
	}

	/**
	 * プロジェクト情報の書き込み
	 * @param savenode
	 * @author kawae
	 * @since 2004/11/06
	 */
	protected void saveProjectInformation(ObjectFolder savenode) {
	}

	/**
	 * プロジェクト情報の読み込み
	 * @param savenode
	 * @author kawae
	 * @since 2004/11/28
	 */
	protected void loadProjectInformation(ObjectFolder node) {
	}


	/**
	 * 読み込んだ幾何をレイヤーに追加
	 * @param l
	 * @param g
	 * @author kawae
	 * @since 2004/11/06
	 */
	protected void addGeomToLayer(Layer l, Geometry g) {
		if (g != null)
			l.addPrimitive(g);
	}

	/**
	 * レイヤー単位で読み込み完了後に必要な処理の実行
	 * @param l
	 * @author kawae
	 * @since 2004/11/06
	 */
	protected void recoverAfterLoad(Layer l) {
		for (int j = 0; j < l.getPrims().size(); j++) {
			Object o = l.getPrims().get(j);
			if (o instanceof ILinkLine) {
				((ILinkLine)o).recoverLink(l.getPrims());
			}
		}
	}

	/**
	 * プロジェクト情報の取得
	 * @return
	 */
	@Override
	public Map<Object, Object> getProjectInfo() {
		return mProjectInfo;
	}

	/**
	 * プロジェクト情報の特定のオブジェクトの取得
	 * @param key
	 * @return
	 */
	@Override
	public Object getProjectInfo(Object key) {
		if (mProjectInfo == null)
			return null;
		return mProjectInfo.get(key);
	}

	/**
	 * レイヤーの追加
	 * @param lm
	 */
	@Override
	public void addLayer(Layer l) {
		mLayers.add(l);
	}

	/**
	 * 現在のレイヤーの設定
	 * @param i	レイヤーID
	 */
	@Override
	public void setCurrentLayer(int i) {
		if (i >= 0 && i <= mLayers.size() - 1)
			mActiveLayer = mLayers.get(i);
	}

	/**
	 * 現在レイヤーの設定
	 * @param s	レイヤーの名前
	 */
	@Override
	public void setCurrentLayer(String s) {
		for (int i = 0; i < mLayers.size(); i++) {
			Layer l = mLayers.get(i);
			String ss = l.getName();
			if (ss != null && ss.equalsIgnoreCase(s)) {
				mActiveLayer = l;
				return;
			}
		}
	}

	@Override
	public List<Layer> getLayers() {
		return mLayers;
	}

	@Override
	public void setSaved(boolean changed) {
		mSaved = changed;
	}

	@Override
	public boolean isSaved() {
		return mSaved;
	}

	/**
	 * 現在のアクティブレイヤーの取得
	 * @return アクティブなレイヤー
	 */
	@Override
	public Layer getCurrentLayer() {
		return mActiveLayer;
	}

	/**
	 * ドキュメントのクリア
	 */
	@Override
	public void documentRefresh() {
		mActiveLayer = null;
		for (int i = 0; i < mLayers.size(); i++) {
			Layer l = mLayers.get(i);
			l.deleteAll();
			l.clearSelectBuffer();
		}
		mLayers.clear();
		mHistoryManager.flushHistory();
	}




	/**
	 * ヒストリのクリア
	 * @param o
	 */
	@Override
	public void flushHistory() {
		mHistoryManager.flushHistory();
	}

	/**
	 * ヒストリに設定
	 * @param o
	 */
	public void setHistoryObject(Object o) {
		mHistoryManager.setHistoryObject(o);
	}

	/**
	 * カレントフロアの最後のヒストリの取得
	 * @return
	 */
	@Override
	public Object getLastHistory() {
		return mHistoryManager.getLastHistory();
	}

	/**
	 * カレントフロアの現在のヒストリの取得
	 * @return
	 */
	@Override
	public Object getCurrentHistory() {
		return mHistoryManager.getCurrentHistory();
	}

	/**
	 * アンドゥ
	 */
	@Override
	public boolean undo() {
		return mHistoryManager.undo();
	}

	/**
	 * リドゥ
	 */
	@Override
	public boolean redo() {
		return mHistoryManager.redo();
	}

	/**
	 * ヒストリに追加
	 *
	 * @author kawae
	 * @since 2004/09/22
	 */
	@Override
	public void addHistory() {
		mHistoryManager.addHistory();
	}

	/**
	 * UNDOヒストリを持つか確認
	 *
	 * @return
	 * @author kawae
	 * @since 2004/11/23
	 */
	@Override
	public boolean hasUndo() {
		return mHistoryManager.hasUndo();
	}

	/**
	 * REDOヒストリを持つか確認
	 * @return
	 * @author kawae
	 * @since 2004/11/23
	 */
	@Override
	public boolean hasRedo() {
		return mHistoryManager.hasRedo();
	}

	/**
	 * 指定された数の履歴を保存する新たな履歴を構築する。
	 * @param num
	 */
	@Override
	public void setMaxHistory(int num) {
		mHistoryManager.setMaxHistory(num);
	}

	@Override
	public void addProjectListener(ProjectListener l) {
		if (l == null)
			return;
		if (mListeners.contains(l))
			return;
		mListeners.add(l);
	}

	@Override
	public void removeProjectListener(ProjectListener l) {
		mListeners.remove(l);
	}

	@Override
	public void fireProjectChanged(Object source) {
		ProjectEvent e = new ProjectEvent(source, this);
		int size = mListeners.size();
		for (int i = size - 1; i >= 0; i--) {
			mListeners.get(i).projectChanged(e);
		}
	}

}
