package nethome.geom.primitive;

import jp.kitec.lib.kwt.IRichGraphics;
import jp.kitec.lib.kwt.KArea;
import jp.kitec.lib.kwt.KColor;
import jp.kitec.lib.util.RefList;
import jp.kitec.lib.util.Vector2;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import nethome.geom.GeomName;
import nethome.geom.GeomParser;
import nethome.geom.LocalAxis2Df;
import nethome.geom.util.ToolMath;
import nethome.geom.util.ToolPolygon;

@GeomName("POLYLINE")
public class GPolyLine extends Geometry implements Primitive {

	/** 構成点コレクション */
	protected RefList<GPoint2Df> mPoints;

	/** 描画用ポイントバッファ */
	protected RefList<GPoint2Df> _pbuf = new RefList<GPoint2Df>();

	/** 線の幅 */
	public double mLineWidth;

	/** 線種 */
	public int mType;

	/** スムーズオプション */
	public int mSmooth;

	/**
	 * コンストラクタ
	 */
	public GPolyLine() {
		this(new RefList<GPoint2Df>(), 0, 0);
	}

	/**
	 * コンストラクタ
	 *
	 * @param	p1		ポリライン構成点コレクション
	 * @param	type	始点X座標
	 * @param	c		描画色
	 */
	public GPolyLine(RefList<GPoint2Df> p1, int type, int smooth) {
		super();
		mGeomColor = KColor.BLACK;
		mPoints = p1;
		mType = type;
		mSmooth = smooth;
		area.maximize();
		updateMinMax(area);
	}

	public void addPoint(GPoint2Df p) {
		mPoints.addElement(p);
		if (areaControl != null)
			areaControl.setArea(mPoints);
	}

	@Override
	public boolean getSnapNode(double x1, double y1, double gx, double gy, double[] min, double[] res, double eps, short limit) {
		return false;
	}

	@Override
	public void getSnapLoop(Vector2<? super GPoint2Df> v) {
		for (int i = 0; i < mPoints.size(); i++)
			v.addElement(mPoints.elementAt(i));
	}

	@Override
	public void getCenter(GPoint2Df p) {
		double x = 0;
		double y = 0;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			x += p1.x;
			y += p1.y;
		}

		p.x = x / mPoints.size();
		p.y = y / mPoints.size();
	}

	@Override
	public GPoint2Df getNearNode(GPoint2Df p1, GPoint2Df p2, double eps) {
		double len, min = Float.MAX_VALUE;
		GPoint2Df res = null;
		for (int k = 0; k < mPoints.size(); k++) {
			GPoint2Df pa = mPoints.elementAt(k);
			if ((len = ToolMath.getLengthLimitLineToPoint2D(pa.x, pa.y, p1.x, p1.y, p2.x, p2.y)) >= 0 && len < eps && len < min) {
				min = len;
				res = pa;
			}
		}
		return res;
	}

	@Override
	public boolean getNearLine(double x, double y, double eps, GPoint2Df[] pts) {
		double len;
		boolean exist = false;
		for (int k = 0; k < mPoints.size(); k++) {
			GPoint2Df pa = mPoints.elementAt(k), pb = mPoints.elementAt(k + 1);
			if ((len = ToolMath.getLengthLimitLineToPoint2D(x, y, pa.x, pa.y, pb.x, pb.y)) >= 0 && len < eps) {
				pts[0] = pa;
				pts[1] = pb;
				exist = true;
			}
		}
		return exist;
	}

	@Override
	public GPoint2Df getRotbasePoint() {
		return mPoints.elementAt(0);
	}

	@Override
	public void updateMinMax(KArea area) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p = mPoints.elementAt(i);
			area.updateMinMax(p.x, p.y);
		}
	}

	@Override
	public Geometry copy() {
		return copyTo(new GPolyLine());
	}

	@Override
	protected Geometry copyTo(Geometry g) {
		if (g == null || !(g instanceof GPolyLine))
			return null;
		GPolyLine np = (GPolyLine)g;
		super.copyTo(np);
		RefList<GPoint2Df> v = new RefList<GPoint2Df>();
		for (int i = 0; i < mPoints.size(); i++)
			v.addElement((GPoint2Df)(mPoints.elementAt(i)).copy());
		np.mPoints = v;
		np.mType = mType;
		np.mSmooth = mSmooth;

		np.area.maximize();
		np.updateMinMax(np.area);
		return np;
	}

	@Override
	public GPoint2Df getNearNode(double x, double y) {
		double min = Float.MAX_VALUE;
		GPoint2Df gp = null;
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			double len = ToolMath.getLength2D(x, y, p1.x, p1.y);
			if (len < min) {
				min = len;
				gp = p1;
			}
		}
		return gp;
	}

	@Override
	public double getDistanceToPoint(double x, double y) {
		double min = Float.MAX_VALUE;
		for (int i = 0; i < mPoints.size() - 1; i++) {
			GPoint2Df p1 = mPoints.elementAt(i);
			GPoint2Df p2 = mPoints.elementAt(i + 1);
			double len = ToolMath.getLengthLimitLineToPoint2D(x, y, p1.x, p1.y, p2.x, p2.y);
			if (len < 0)
				continue;
			else if (len < min)
				min = len;
		}
		return min;
	}

	@Override
	public void offset(double x, double y) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).offset(x, y);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void rotate(double xc, double yc, double angl) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).rotate(xc, yc, angl);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public boolean isInner(double minx, double miny, double maxx, double maxy) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1, p2;
			p1 = mPoints.elementAt(i);
			if (i == mPoints.size() - 1)
				p2 = mPoints.elementAt(0);
			else
				p2 = mPoints.elementAt(i + 1);
			boolean b1 = false, b2 = false;
			if (p1.x > minx && p1.x < maxx && p1.y > miny && p1.y < maxy)
				b1 = true;
			if (p2.x > minx && p2.x < maxx && p2.y > miny && p2.y < maxy)
				b2 = true;
			if (!b1 || !b2)
				return false;
		}
		return true;
	}

	@Override
	public boolean isInner(RefList<GPoint2Df> v) {
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p1, p2;
			p1 = mPoints.elementAt(i);
			if (i == mPoints.size() - 1)
				p2 = mPoints.elementAt(0);
			else
				p2 = mPoints.elementAt(i + 1);
			boolean b1 = ToolPolygon.isInnerPolygon(v, p1.x, p1.y);
			boolean b2 = ToolPolygon.isInnerPolygon(v, p2.x, p2.y);
			if (!b1 || !b2)
				return false;
		}
		return true;
	}

	public RefList<GPoint2Df> getPoints() {
		return mPoints;
	}

	public static void readData(ObjectFolder of, Geometry g, GeomParser ps) throws Exception {
		ObjectNode on;
		GPolyLine gp = null;
		if (!(g instanceof GPolyLine))
			return;
		Geometry.readData(of, g, ps);

		gp = (GPolyLine)g;
		gp.mPoints = new RefList<GPoint2Df>();
		int num = 0;
		if ((on = of.getNode("smooth")) != null)
			gp.mSmooth = Integer.valueOf((String)on.getObject()).intValue();
		if ((on = of.getNode("type")) != null)
			gp.mType = Integer.valueOf((String)on.getObject()).intValue();
		if ((on = of.getNode("pointsize")) != null)
			num = Integer.valueOf((String)on.getObject()).intValue();

		for (int i = 0; i < num; i++) {
			double x = 0, y = 0;
			if ((on = of.getNode("x" + (i + 1))) != null)
				x = Float.valueOf((String)on.getObject()).doubleValue();
			if ((on = of.getNode("y" + (i + 1))) != null)
				y = Float.valueOf((String)on.getObject()).doubleValue();
			gp.mPoints.addElement(new GPoint2Df(x, y));
		}
	}

	/**
	 * データの保存
	 * @param savenode
	 */
	@Override
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		super.save(savenode);

		savenode.addNode("smooth", mSmooth);
		savenode.addNode("type", mType);
		savenode.addNode("pointsize", mPoints.size());
		for (int i = 0; i < mPoints.size(); i++) {
			GPoint2Df p = mPoints.elementAt(i);
			savenode.addNode("x" + (i + 1), p.x);
			savenode.addNode("y" + (i + 1), p.y);
		}
	}

	@Override
	public void drawGeom(IRichGraphics d) {
		drawGeom(d, null, null);
	}

	@Override
	public void drawGeom(IRichGraphics d, KColor fore, KColor back) {
		if (fore == null)
			fore = mGeomColor;
		drawAbstDevice(d, fore, back, null);
	}

	@Override
	public void drawHighLight(IRichGraphics d, KColor c, boolean outline) {
		if (outline) {
			d.setColor(c);
			d.drawPolyLine(mPoints, IRichGraphics.LINE_DASH_1);
		}
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).drawHighLight(d, c, false);

	}

	@Override
	public void drawAlias(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
		if (fore == null)
			fore = mGeomColor;
		drawAbstDevice(d, fore, back, la);
	}
	@Override
	public void drawAbstDevice(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
		RefList<GPoint2Df> poly = null;
		_pbuf.removeAllElements();
		if (la != null) {
			for (int i = 0; i < mPoints.size(); i++) {
				GPoint2Df p = mPoints.elementAt(i);
				_pbuf.addElement(new GPoint2Df(la.getGlobalX(p.x, p.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY), la.getGlobalY(p.x, p.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY)));
			}
			poly = _pbuf;
		} else {
			poly = mPoints;
		}

		d.setColor(fore);
		d.drawPolyLine(poly, null);
		_pbuf.removeAllElements();
	}


	@Override
	public void flipx(double x, double ofsx) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).flipx(x, ofsx);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void flipy(double y, double ofsy) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).flipy(y, ofsy);
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void rot(double x, double y, double ofsx, double ofsy) {
		for (int i = 0; i < mPoints.size(); i++)
			(mPoints.elementAt(i)).rot(x, y, ofsx, ofsy);

		area.maximize();
		updateMinMax(area);
	}
}

//
//public RefList getControlPoints() {
//	return mPoints;
//}

//
//public static Object readData(AbstFile r, double version) throws Exception {
//	GPolyLine gp = null;
//	RefList pts = new RefList();
//	int smooth = Integer.valueOf(r.readLineFromBuffer().trim()).intValue();
//	int type = Integer.valueOf(r.readLineFromBuffer().trim()).intValue();
//	int num = Integer.valueOf(r.readLineFromBuffer().trim()).intValue();
//	for (int i = 0; i < num; i++) {
//		StringTokenizer st = new StringTokenizer(r.readLineFromBuffer(), " ,");
//		double x = Float.valueOf((String)st.nextElement()).doubleValue();
//		double y = Float.valueOf((String)st.nextElement()).doubleValue();
//		pts.addElement(new GPoint2Df(x, y));
//	}
//	gp = new GPolyLine(pts, type, smooth);
//	while (true) {
//		String str = r.readLineFromBuffer();
//		if (str == null || str.trim().equalsIgnoreCase("</POLYLINE>"))
//			break;
//	}
//	return gp;
//}

//public String toString() {
//	StringBuffer sb = new StringBuffer();
//	sb.append("polyline " + mType + "," + mSmooth);
//	for (int i = 0; i < mPoints.size(); i++) {
//		GPoint2Df gp = (GPoint2Df)mPoints.elementAt(i);
//		sb.append("," + gp.x + "," + gp.y);
//	}
//	return sb.toString();
//}