package nethome.geom.primitive;

import java.io.Serializable;

import jp.kitec.kwt.IRichGraphics;
import jp.kitec.kwt.KArea;
import jp.kitec.kwt.KColor;
import jp.kitec.lib.geom.K2DFigureType;
import jp.kitec.lib.geom.K2DPathFigureLine;
import jp.kitec.lib.util.RefList;
import jp.kitec.lib.util.Vector2;
import jp.kitec.lib.util.tree.ObjectFolder;
import nethome.geom.GeomName;
import nethome.geom.LocalAxis2Df;
import nethome.geom.Ltype;
import nethome.geom.util.ToolMath;
import nethome.geom.util.ToolPolygon;

@GeomName("LINE")
public class GLine extends Geometry implements K2DPathFigureLine, Primitive, Serializable {

	/** 端点1 */
	public GPoint2Df pA;

	/** 端点2 */
	public GPoint2Df pB;

	/** 線の太さ */
//	protected double mLineWidth;

//	/** コントロールポイント群 */
//	protected RefList mConPts;

	/**
	 * コンストラクタ
	 */
	public GLine() {
		this(new GPoint2Df(), new GPoint2Df());
	}

	/**
	 * コンストラクタ
	 */
	public GLine(GPoint2Df p1, GPoint2Df p2) {
		this(p1, p2, null);
	}

	/**
	 * コンストラクタ
	 */
	public GLine(GPoint2Df p1, GPoint2Df p2, Ltype ltype) {
		super();
		mGeomColor = KColor.BLACK;
		pA = p1;
		pB = p2;
//		mConPts = new RefList();
//		mConPts.addElement(pA);
//		mConPts.addElement(pB);
		lineType = ltype;
		area.maximize();
		updateMinMax(area);
//		mLineWidth = 0;
	}

	/**
	 * コンストラクタ
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 */
	public GLine(double x1, double y1, double x2, double y2) {
		this(new GPoint2Df(x1, y1), new GPoint2Df(x2, y2));
	}

	@Override
	public boolean getSnapNode(double x1, double y1, double gx, double gy, double[] min, double[] res, double eps, short limit) {
//		return Snapping.utilSnapNodeToLine(x1, y1, pA, pB, gx, gy, min, res, eps, limit);
		return false;
	}

	@Override
	public void getSnapLoop(Vector2<? super GPoint2Df> v) {
		v.addElement(pA);
		v.addElement(pB);
	}

	@Override
	public void getCenter(GPoint2Df p) {
		p.x = (pA.x + pB.x) / 2;
		p.y = (pA.y + pB.y) / 2;
	}

	@Override
	public GPoint2Df getNearNode(GPoint2Df p1, GPoint2Df p2, double eps) {
		double len, min = Double.MAX_VALUE;
		GPoint2Df res = null;
		for (int k = 0; k < 2; k++) {
			GPoint2Df pa = k == 0 ? pA : pB;
			if ((len = ToolMath.getLengthLimitLineToPoint2D(pa.x, pa.y, p1.x, p1.y, p2.x, p2.y)) >= 0 && len < eps && len < min) {
				min = len;
				res = pa;
			}
		}
		return res;
	}

	@Override
	public boolean getNearLine(double x, double y, double eps, GPoint2Df[] pts) {
		double len;
		GPoint2Df pa = pA, pb = pB;
		if ((len = ToolMath.getLengthLimitLineToPoint2D(x, y, pa.x, pa.y, pb.x, pb.y)) >= 0 && len < eps) {
			pts[0] = pa;
			pts[1] = pb;
			return true;
		}
		return false;
	}

	@Override
	public GPoint2Df getRotbasePoint() {
		return pA;
	}

	@Override
	public void updateMinMax(KArea area) {
		area.updateMinMax(pA.x, pA.y);
		area.updateMinMax(pB.x, pB.y);
	}

	@Override
	public Geometry copy() {
		return copyTo(new GLine());
	}

	@Override
	protected Geometry copyTo(Geometry g) {
		if (g == null || !(g instanceof GLine))
			return null;
		GLine gl = (GLine)g;
		super.copyTo(gl);
		gl.pA.setXY(pA);
		gl.pB.setXY(pB);
		gl.area.maximize();
		gl.updateMinMax(gl.area);
		return gl;
	}

	/**
	 * 一番近くの構成点を探す
	 * @param x
	 * @param y
	 * @return
	 */
	@Override
	public GPoint2Df getNearNode(double x, double y) {
		double min = Double.MAX_VALUE;
		GPoint2Df gp = null;
		double tmp;

		if ((tmp = ToolMath.getLength2D(x, y, pA.x, pA.y)) < min) {
			gp = pA;
			min = tmp;
		}

		if ((tmp = ToolMath.getLength2D(x, y, pB.x, pB.y)) < min)
			gp = pB;
		return gp;
	}

	@Override
	public double getDistanceToPoint(double x, double y) {
		double len;
		len = ToolMath.getLengthLimitLineToPoint2D(x, y, pA.x, pA.y, pB.x, pB.y);

		if (len < 0)
			return -1;
		return len;
	}

	@Override
	public void offset(double x, double y) {
		pA.x += x;
		pA.y += y;
		pB.x += x;
		pB.y += y;
		updateShape();
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void rotate(double xc, double yc, double angl) {
		pA.rotate(xc, yc, angl);
		pB.rotate(xc, yc, angl);
		updateShape();
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public boolean isInner(double minx, double miny, double maxx, double maxy) {
		boolean p1 = false;
		boolean p2 = false;
		if (pA.x > minx && pA.x < maxx && pA.y > miny && pA.y < maxy)
			p1 = true;
		if (pB.x > minx && pB.x < maxx && pB.y > miny && pB.y < maxy)
			p2 = true;
		return p1 && p2;
	}

	@Override
	public boolean isInner(RefList<GPoint2Df> v) {
		return ToolPolygon.isInnerPolygon(v, pA.x, pA.y) && ToolPolygon.isInnerPolygon(v, pB.x, pB.y);
	}

	/**
	 * 幾何データの保存
	 *
	 * @param savenode		保存対象ノード
	 * @author kawae
	 * @since 2003/01/01
	 */
	@Override
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		super.save(savenode);
		savenode.addNode("x1", this.pA.x);
		savenode.addNode("y1", this.pA.y);
		savenode.addNode("x2", this.pB.x);
		savenode.addNode("y2", this.pB.y);
//		savenode.addNode("width", this.mLineWidth);
	}

	/**
	 * 線種の設定
	 * @param ltype　線種
	 */
	public void setLType(Ltype ltype) {
		lineType = ltype;
	}

	public Ltype getLType() {
		return lineType;
	}

	/**
	 * 描画
	 *
	 * @param	d	描画インターフェース
	 * @author kawae
	 * @since 2003/01/01
	 */
	@Override
	public void drawGeom(IRichGraphics d) {
		drawGeom(d, mGeomColor, null);
	}

	/**
	 * ハイライト描画
	 *
	 * @param	d	描画インターフェース
	 * @param	c		幾何描画色
	 * @author kawae
	 * @since 2003/01/01
	 */
	@Override
	public void drawHighLight(IRichGraphics d, KColor c, boolean outline) {
		if (outline) {
			drawAbstDevice(d, c, c, null);
		}
		pA.drawHighLight(d, c, false);
		pB.drawHighLight(d, c, false);
	}

	@Override
	public void drawGeom(IRichGraphics d, KColor fore, KColor back) {
//		if (back == null)
//			back = _fillColor;
		if (fore == null)
			fore = mGeomColor;
		drawAbstDevice(d, fore, back, null);
	}

//	public void writeCadData(CadWriter cw, LocalAxis2Df la) {
//		drawAbstDevice(null, geomColor, null, la);
//	}

	@Override
	public void drawAlias(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
//		if (back == null)
//			back = _fillColor;
		if (fore == null)
			fore = mGeomColor;
		drawAbstDevice(d, fore, back, la);
	}

	/**
	 * 最終描画
	 *
	 * @param d	描画インターフェース
	 * @param fore	前景色
	 * @param back	背景色
	 * @param la	描画座標
	 * @author kawae
	 * @since 2004/09/24
	 */
	@Override
	public void drawAbstDevice(IRichGraphics d, KColor fore, KColor back, LocalAxis2Df la) {
		double x1, y1, x2, y2;
		if (la != null) {
			x1 = la.getGlobalX(pA.x, pA.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
			y1 = la.getGlobalY(pA.x, pA.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
			x2 = la.getGlobalX(pB.x, pB.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
			y2 = la.getGlobalY(pB.x, pB.y, mKeepAspect, scaleEnX, scaleEnY, scaleBaseX, scaleBaseY);
		} else {
			x1 = pA.x;
			y1 = pA.y;
			x2 = pB.x;
			y2 = pB.y;
		}
//		if (device instanceof Graphics){
		d.setColor(fore);
		d.drawLine(x1, y1, x2, y2, null);
//		} else
//			((CadWriter)device).writeLine(x1, y1, x2, y2, lineType, mLineWidth, fore);
	}

//	public void drawTransition(IRichGraphics g, KColor c, KLineType lt) {
//		g.setColor(c);
//
//		g.drawLine(pA.x, pA.y, pB.x, pB.y, lt);
//		pA.drawTransition(g, c, lt);
//		pB.drawTransition(g, c, lt);
//	}

//	/**
//	 * コントロールポイントの取得
//	 * @return RefList　コントロールポイント群
//	 */
//	public RefList getControlPoints() {
//		return mConPts;
//	}

	/**
	 * 一番近いコントロールポイントの取得
	 * @return Object	一番近いコントロールポイント
	 */
	@Override
	public Object getNearControlPoint(double x, double y) {
		double min = Float.POSITIVE_INFINITY;
		GPoint2Df pt = null;

		for (int i = 0; i < 2; i++) {
			GPoint2Df p = i == 0 ? pA : pB;
			double len = p.getDistanceToPoint(x, y);
			if (len < min) {
				min = len;
				pt = p;
			}
		}

//
//		for (int i = 0; i < mConPts.size(); i++) {
//			GPoint2Df p = (GPoint2Df)mConPts.elementAt(i);
//			double len = p.getDistanceToPoint(x, y);
//			if (len < min) {
//				min = len;
//				pt = p;
//			}
//		}
		return pt;
	}
//
//	/**
//	 * スナップポイントの取得
//	 * @param x	スナップしたい点のｘ座標
//	 * @param y	スナップしたい点のｙ座標
//	 * @param pt	結果の移動先バッファ
//	 * @return boolean	移動可能ならTRUE
//	 */
//	public boolean getNearSanpPoint(double x, double y, double[] pt, double eps) {
//		double len1 = pA.getDistanceToPoint(x, y);
//		double len2 = pB.getDistanceToPoint(x, y);
//		if (len1 < len2 && len1 < eps) {
//			pt[0] = pA.x;
//			pt[1] = pA.y;
//			return true;
//		} else if (len2 < len1 && len2 < eps) {
//			pt[0] = pB.x;
//			pt[1] = pB.y;
//			return true;
//		}
//		return false;
//	}
//
//	/**
//	 * スナップポイントの取得
//	 * @param x	スナップしたい点のｘ座標
//	 * @param y	スナップしたい点のｙ座標
//	 * @param pt	結果の移動先バッファ
//	 * @return boolean	移動可能ならTRUE
//	 */
//	public boolean getNearSanpBody(double x, double y, double[] pt, double eps) {
//		double len = (double)ToolMath.getLengthLimitLineToPoint2D(x, y, pA.x, pA.y, pB.x, pB.y);
//		if (len >= 0 && len < eps)
//			return ToolMathEx.isVerticalLimitLineToPoint2D(x, y, pA.x, pA.y, pB.x, pB.y, pt);
//		return false;
//	}

	/**
	 * コントロールポイントの移動
	 * @param cp　コントロールポイント
	 * @param x	移動先ｘ
	 * @param y	移動先ｙ
	 */
	@Override
	public void moveControlPoint(Object cp, double x, double y) {
		if (cp == pA) {
			pA.x = x;
			pA.y = y;
		} else if (cp == pB) {
			pB.x = x;
			pB.y = y;
		}
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void flipx(double x, double ofsx) {
		pA.flipx(x, ofsx);
		pB.flipx(x, ofsx);
		updateShape();
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void flipy(double y, double ofsy) {
		pA.flipy(y, ofsy);
		pB.flipy(y, ofsy);
		updateShape();
		area.maximize();
		updateMinMax(area);
	}

	@Override
	public void rot(double x, double y, double ofsx, double ofsy) {
		pA.rot(x, y, ofsx, ofsy);
		pB.rot(x, y, ofsx, ofsy);
		updateShape();
		area.maximize();
		updateMinMax(area);
	}

	/**
	 * @return
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigure#getType()
	 */
	@Override
	public int getType() {
		return K2DFigureType.LINE;
	}

	/**
	 * @return
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#getX1()
	 */
	@Override
	public double getX1() {
		return pA.x;
	}

	/**
	 * @return
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#getX2()
	 */
	@Override
	public double getX2() {
		return pB.x;
	}

	/**
	 * @return
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#getY1()
	 */
	@Override
	public double getY1() {
		return pA.y;
	}

	/**
	 * @return
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#getY2()
	 */
	@Override
	public double getY2() {
		return pB.y;
	}

	/**
	 * @param x
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#setX1(double)
	 */
	@Override
	public void setX1(double x) {
		pA.setX(x);
	}

	/**
	 * @param x
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#setX2(double)
	 */
	@Override
	public void setX2(double x) {
		pB.setX(x);
	}

	/**
	 * @param y
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#setY1(double)
	 */
	@Override
	public void setY1(double y) {
		pA.setY(y);
	}

	/**
	 * @param y
	 * @author kawae
	 * @since 2006/01/18
	 * @see com.kt.geom.K2DPathFigureLine#setY2(double)
	 */
	@Override
	public void setY2(double y) {
		pB.setY(y);
	}
}