package nethome.doc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


/**
 * ライブラリ用のドキュメントなど、各種ドキュメントを管理する
 * ドキュメント管理クラス。
 *
 * @since 　2003/03/12
 * @author　Kawae
 * @version 2003/03/12
 *
 * Copyright (c) 2002 KITec Inc,.. All rights reserved.
 */
public class Document implements ProjectListener {

	/** プロジェクトドキュメント */
	private Project mProject;

	/** プロジェクトドキュメント */
	private ArrayList<Project> mProjects;

	/** リスナー */
	private ArrayList<DocumentListener> mListeners;

	/**
	 * すべてのドキュメントを空として、新たな Document を構築する。
	 *
	 * @since   2003/03/12
	 * @author  Kawae
	 * @version 2003/03/12
	 */
	public Document() {
		mProject = null;
		mProjects = new ArrayList<Project>();
		mListeners = new ArrayList<DocumentListener>();
	}

	/**
	 * この Document が管理する NetHomeドキュメントを返す。
	 *
	 * @return NetHomeドキュメント
	 * @see	#setNh3Doc(DocumentNh3)
	 * @since   2003/03/12
	 * @author  Kawae
	 * @version 2003/03/12
	 */
	public Project getProject() {
		return mProject;
	}

	/**
	 * 指定されたプロジェクトを、この Document が管理する
	 * NetHome ドキュメントとして設定する。
	 *
	 * @param prj	NetHome ドキュメント
	 * @see	#getProject()
	 * @since   2003/03/12
	 * @author  Kawae
	 * @version 2003/03/12
	 */
	public void setProject(Project prj) {
		if (!mProjects.contains(prj))
			addProject(prj);
		mProject = prj;
	}

	/**
	 * 指定されたプロジェクトを、この Document が管理するプロジェクトに追加する。
	 * @param prj
	 */
	public void addProject(Project prj) {
		mProjects.add(prj);
		prj.addProjectListener(this);
	}

	/**
	 * 指定されたプロジェクトを、この Document が管理するプロジェクトから削除する。
	 * @param prj
	 */
	public void removeProject(Project prj) {
		mProjects.remove(prj);
	}

	/**
	 * この Document が管理するプロジェクトのリストを返す。
	 */
	public List<Project> getProjects() {
		return Collections.unmodifiableList(mProjects);
	}

	/**
	 * この Document が管理するすべてのドキュメントをリフレッシュする。
	 *
	 * @since   2003/03/12
	 * @author  Kawae
	 * @version 2003/03/12
	 */
	public void refreshDocument() {
		for (Project prj : mProjects) {
			prj.documentRefresh();
		}
	}

	public void addDocumentListener(DocumentListener l) {
		if (l == null)
			return;
		if (mListeners.contains(l))
			return;
		mListeners.add(l);
	}

	public void removeDocumentListener(DocumentListener l) {
		mListeners.remove(l);
	}

	public void fireDocumentChanged(Object source) {
		DocumentEvent e = new DocumentEvent(source, this);
		int size = mListeners.size();
		for (int i = size - 1; i >= 0; i--) {
			mListeners.get(i).documentChanged(e);
		}
	}

	@Override
	public void projectChanged(ProjectEvent e) {
		fireDocumentChanged(e.getSource());
	}
}
