/*
 * Created on 2004/11/13
 */
package nethome.geom.rich;

import java.io.Serializable;

import jp.kitec.kwt.KColor;
import jp.kitec.kwt.KFont;
import jp.kitec.kwt.KFontMetrics;
import jp.kitec.lib.util.UtilString;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;
import nethome.NethomeCoreResource;
import nethome.text.ITextFormat;


/**
 * テキストの最小単位
 *
 * @author kawae
 * @since 2004/11/13
 */
public class KTextElement implements ITextFormat, Serializable {

	public static final String GEOMNAME = "TEXT_ELEMENT";

	private String mText;
	private String mFontname;
	private double mFontsize;
	private boolean mIsUnderline;
	private int mFontstyle;
	private KColor mTextColor;
	private KColor mBackColor;

	/**
	 * コンストラクタ
	 *
	 * @author kawae
	 * @since 2004/11/14
	 */
	public KTextElement() {
	}

	/**
	 * テキストの設定
	 * @param text
	 * @author kawae
	 * @since 2004/11/13
	 */
	public void setText(String text) {
		mText = UtilString.tab2space(text);
	}

	public String getText() {
		return mText;
	}

	@Override
	public KColor getBackColor() {
		return mBackColor;
	}

	@Override
	public String getFontName() {
		return mFontname;
	}

	@Override
	public double getFontSize() {
		return mFontsize;
	}

	@Override
	public int getFontStyle() {
		return mFontstyle;
	}

	@Override
	public KColor getTextColor() {
		return mTextColor;
	}

	@Override
	public boolean isSameTextFormat(ITextFormat tf) {
		if (UtilString.isDiff(getFontName(), tf.getFontName()))
			return false;
		if (getFontStyle() != tf.getFontStyle())
			return false;
		if (Math.abs(getFontSize() - tf.getFontSize()) > 1e-3)
			return false;
		if (KColor.isDiff(getTextColor(), tf.getTextColor()))
			return false;
		if (KColor.isDiff(getBackColor(), tf.getBackColor()))
			return false;
		if (isUnderLine() != tf.isUnderLine())
			return false;
		return true;
	}

	@Override
	public boolean isUnderLine() {
		return mIsUnderline;
	}

	@Override
	public void setBackColor(KColor c) {
		mBackColor = c;
	}

	@Override
	public void setFontName(String s) {
		mFontname = s;
	}

	@Override
	public void setFontSize(double f) {
		mFontsize = f;
	}

	@Override
	public void setFontStyle(int style) {
		mFontstyle = style;
	}

	@Override
	public void setFormat(ITextFormat f) {
		setBackColor(f.getBackColor());
		setTextColor(f.getTextColor());
		setFontName(f.getFontName());
		setFontSize(f.getFontSize());
		setFontStyle(f.getFontStyle());
		setUnderLine(f.isUnderLine());
	}

	@Override
	public void setTextColor(KColor c) {
		mTextColor = c;
	}

	@Override
	public void setUnderLine(boolean flag) {
		mIsUnderline = flag;
	}

	/**
	 * 保存
	 * @param savenode
	 * @author kawae
	 * @since 2004/11/17
	 * @see nethome.text.ITextFormat#save(jp.kitec.lib.util.tree.ObjectFolder)
	 */
	@Override
	public void save(ObjectFolder savenode) {
		if (savenode == null)
			return;
		savenode.setName(GEOMNAME);
		savenode.addNode("text", mText);
		savenode.addNode("textsize", mFontsize);
		savenode.addNode("textfont", mFontname);
		savenode.addNode("textstyle", mFontstyle);
		savenode.addNode("underline", mIsUnderline);
		if (mTextColor != null)
			savenode.addChild(new ObjectNode("textcolor", KColor.getColorRGBString(mTextColor)));
		if (mBackColor != null)
			savenode.addChild(new ObjectNode("backcolor", KColor.getColorRGBString(mBackColor)));
	}

	/**
	 * 読み込み
	 * @param of
	 * @param tl
	 * @return
	 * @author kawae
	 * @since 2004/11/17
	 */
	public static KTextElement readData(ObjectFolder of, KTextElement tl) {
		ObjectNode on = null;
		if (ObjectNode.isStringObject(on = of.getNode("text")))
			tl.mText = UtilString.tab2space(ObjectNode.getNodeString(on));
		if (ObjectNode.isStringObject(on = of.getNode("textfont")))
			tl.mFontname = ObjectNode.getNodeString(on);
		if (ObjectNode.isFloatObject(on = of.getNode("textsize")))
			tl.mFontsize = ObjectNode.getNodeFloat(on);
		if (ObjectNode.isBoolObject(on = of.getNode("underline")))
			tl.mIsUnderline = ObjectNode.getNodeBool(on);
		if (ObjectNode.isIntObject(on = of.getNode("textstyle")))
			tl.mFontstyle = Integer.valueOf((String)on.getObject()).intValue();
		if ((on = of.getNode("textcolor")) != null)
			tl.mTextColor = KColor.getColorFromRGBString((String)on.getObject());
		if ((on = of.getNode("backcolor")) != null)
			tl.mBackColor = KColor.getColorFromRGBString((String)on.getObject());
		return tl;
	}

	/**
	 * フォントの取得
	 * @return
	 * @author kawae
	 * @since 2004/11/17
	 */
	public KFont getFont() {
		return KFont.getFont(getFontName(), getFontStyle(), getFontSize());
	}
	/**
	 * 最大幅の取得
	 * @param hor
	 * @return
	 * @author kawae
	 * @since 2004/11/14
	 * @version 2006/01/05縦書き対応
	 */
	public double getMaxWidth(boolean hor, ITextDpi dpi) {
		double total = 0;
		String str = getText();
		KFont f = KFont.getFont(getFontName(), getFontStyle(), dpi.getDpiScaleSize(getFontSize()));
		KFontMetrics fm = f.getFontMetrics(null);
		if (hor) {
			return (Math.ceil(fm.stringWidth(str) * 1000) / 1000);
		} else {
			for (int i = 0; i < str.length(); i++) {
				String s = str.substring(i, i + 1);
				double ww = 0;
				if (NethomeCoreResource.TATE_STRING.indexOf(s) >= 0)
					ww = (Math.ceil((fm.getDescent() + fm.getAscent()) * 1000) / 1000);
				else
					ww = (Math.ceil(fm.stringWidth(s) * 1000) / 1000);
				total = total < ww ? ww : total;
			}
		}
		return total;
	}

	/**
	 * 最大高さの取得
	 * @param hor
	 * @return
	 * @author kawae
	 * @since 2004/11/14
	 * @version 2006/01/05縦書き対応
	 */
	public double getMaxHeight(boolean hor, ITextDpi dpi) {
		double height = 0;
		double dpiScaleSize = dpi.getDpiScaleSize(getFontSize());
		KFont f = KFont.getFont(getFontName(), getFontStyle(), dpiScaleSize);
		KFontMetrics fm = f.getFontMetrics(null);
		if (hor) {
			height = Math.ceil((fm.getDescent() + fm.getAscent()) * 1000) / 1000;
			if (height < dpiScaleSize / 5) {//FIXME
				height = dpiScaleSize;
			}
		} else {
			String str = getText();
			for (int i = 0; i < str.length(); i++) {
				String s = str.substring(i, i + 1);
				if (NethomeCoreResource.TATE_STRING.indexOf(s) >= 0)
					height += (Math.ceil(fm.stringWidth(s) * 1000) / 1000);
				else
					height += (Math.ceil((fm.getDescent() + fm.getAscent()) * 1000) / 1000);
			}
		}
		return height;
	}
}
