/**
 * Copyright (c) 2002, 2010 KITec Inc,.. All rights reserved.
 * Created on 2002/11/28
 */
package nethome.geom;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.vecmath.Point2d;

import jp.kitec.kwt.KArea;
import jp.kitec.lib.util.Vector2;
import jp.kitec.vecmath.KPoint2d;
import nethome.geom.util.ToolMath;


/**
 * @author kawae
 */
public class WorkSheet implements java.io.Serializable {

	public static final int NEAR_GRID2_THRESHOLD_DIVIDE = 30;

	public static final int DEF_NEAR_GRID_DIVIDE = 50;

	/** ワークシート名 */
	private String mName;

	/** 基本モジュール */
	private double mModule;

	/** 領域 */
	private final KArea mArea;

	/**
	 * グリッド分割数
	 */
	private int mGridDiv = 1;

	/**
	 * グリッドを通過する点
	 */
	private double mGridCenterX, mGridCenterY;

	/**
	 * 追加グリッド
	 */
	private Vector2<Double> mExtraGridX;
	private Vector2<Double> mExtraGridY;


	/** エリアポリゴン */
	private final List<KPoint2d> mAreaPolygon;

	/**
	 * Constructor for Shikichi.
	 */
	public WorkSheet() {
		mArea = new KArea();
		mAreaPolygon = new ArrayList<KPoint2d>();
	}

	/**
	 * コピーインスタンスの作成
	 * @return
	 * @author kawae
	 * @since 2004/10/02
	 */
	public WorkSheet getCopy() {
		WorkSheet ws = new WorkSheet();
		ws.setName(getName());
		ws.setArea(mArea.minX, mArea.minY, mArea.maxX, mArea.maxY);
		ws.mGridDiv = mGridDiv;
		ws.mModule = mModule;
		ws.mGridCenterX = mGridCenterX;
		ws.mGridCenterY = mGridCenterY;
//		for (int i = 0; i < mAreaPolygon.size(); i++)
//			ws.mAreaPolygon.addElement(((Point2d)mAreaPolygon.elementAt(i)).getCopy());
		return ws;

	}
	/**
	 * ワークシート名の設定
	 * @param s
	 * @author kawae
	 * @since 2004/10/02
	 */
	public void setName(String s) {
		mName = s;
	}

	/**
	 * ワークシート名の取得
	 * @return
	 * @author kawae
	 * @since 2004/10/02
	 */
	public String getName() {
		return mName;
	}

	/**
	 * 領域の設定
	 * @param minx
	 * @param miny
	 * @param maxx
	 * @param maxy
	 * @author kawae
	 * @since 2004/10/02
	 */
	public void setArea(double minx, double miny, double maxx, double maxy) {
		mArea.maximize();
		mArea.updateMinMax(minx, miny);
		mArea.updateMinMax(maxx, maxy);
		mAreaPolygon.clear();
		mAreaPolygon.add(new KPoint2d(minx, miny));
		mAreaPolygon.add(new KPoint2d(maxx, miny));
		mAreaPolygon.add(new KPoint2d(maxx, maxy));
		mAreaPolygon.add(new KPoint2d(minx, maxy));
	}

	/**
	 * 領域の取得
	 * @return
	 * @author kawae
	 * @since 2004/10/02
	 */
	public KArea getArea() {
		return mArea;
	}

	public void updateArea() {
		mArea.maximize();
		for (int i = 0; i < mAreaPolygon.size(); i++) {
			Point2d p = mAreaPolygon.get(i);
			mArea.updateMinMax(p.x, p.y);
		}
	}

	public List<KPoint2d> getAreaPolygon() {
		return Collections.unmodifiableList(mAreaPolygon);
	}

	/**
	 * モジュールの取得
	 * @return
	 */
	public double getModule() {
		return mModule;
	}

	/**
	 * モジュールの設定
	 * @param f
	 */
	public void setModule(double f) {
		if (f < 0)
			throw new IllegalArgumentException();
		mModule = f;
	}
	/**
	 * グリッドのベースモジュールに対する分割数の設定
	 * @param i
	 */
	public void setGridDivide(int i) {
		if (i <= 0)
			throw new IllegalArgumentException();
		mGridDiv = i;
	}

	public int getGridDivide() {
		return mGridDiv;
	}

	public double getCurrentGrid() {
		return mModule / mGridDiv;
	}

	public void setGridCenter(double x, double y) {
		mGridCenterX = x;
		mGridCenterY = y;
	}

	public double getGridCenterX() {
		return mGridCenterX;
	}

	public double getGridCenterY() {
		return mGridCenterY;
	}


	/**
	 * 最も近いXグリッドを探す
	 * @param x
	 * @param extra 追加グリッドも含めて探すかのフラグ。
	 * @return
	 */
	public double getNearGridX(double x, boolean extra) {
		return getNearGrid(x, extra, getCurrentGrid(), getGridCenterX(), mExtraGridX);
	}

	/**
	 * 最も近いYグリッドを探す
	 * @param y
	 * @param extra 追加グリッドも含めて探すかのフラグ。
	 * @return
	 */
	public double getNearGridY(double y, boolean extra) {
		return getNearGrid(y, extra, getCurrentGrid(), getGridCenterY(), mExtraGridY);
	}

	private double getNearGrid(double v, boolean extra, double currentGrid, double gridCenter, Vector2<Double> extraGrid) {
		if (currentGrid == 0)
			return v;
		double near = Double.MAX_VALUE;
		int step = v <= gridCenter ? -1:1;
		int c = (int)Math.floor(((v - gridCenter) / currentGrid) * step);

		while (true) {
			double len = Math.abs(v - (gridCenter + currentGrid * c * step));
			if (near > len)
				near = len;
			else
				break;
			c++;
		}
		double gridcandidate = step * (c - 1) * currentGrid + gridCenter;
		if (extra && extraGrid != null) {
			for (int i = 0; i < extraGrid.size(); i++) {
				double f = extraGrid.elementAt(i).doubleValue();
				double len = Math.abs(v - f);
				if (len < near) {
					gridcandidate = f;
					near = len;
				}
			}
		}
		return gridcandidate;
	}
	/**
	 * デフォルトのスナップX
	 * @param x
	 * @return double
	 */
	public double getDefNearGridX(double x) {
		double near = Double.MAX_VALUE;
		double div = this.getModule() / DEF_NEAR_GRID_DIVIDE;

		int c = (int)((x - x % div) / div);
		int res = 0;
		for (int i = c - 1; i <= c + 1; i++) {
			if (Math.abs(div * i - x) < near) {
				near =  Math.abs(div * i - x);
				res = i;
			}
		}
		return res * div;
	}
	public double getDefNearGridY(double y) {
		double near = Double.MAX_VALUE;
		double div = this.getModule() / DEF_NEAR_GRID_DIVIDE;

		int c = (int)((y - y % div) / div);
		int res = 0;
		for (int i = c - 1; i <= c + 1; i++) {
			if (Math.abs(div * i - y) < near) {
				near =  Math.abs(div * i - y);
				res = i;
			}
		}
		return res * div;
	}

	public boolean isOnGridX(double x) {
		double xx = this.getNearGridX(x, true);
		if (Math.abs(xx - x) < ToolMath.EPS)
			return true;
		return false;
	}
	public boolean isOnGridY(double y) {
		double yy = this.getNearGridY(y, true);
		if (Math.abs(yy - y) < ToolMath.EPS)
			return true;
		return false;
	}
	public boolean isOnBaseGridX(double x) {
		double c = Math.abs(x % mModule);
		if (c < ToolMath.EPS || Math.abs(c - mModule) < ToolMath.EPS) {
			return true;
		}
		return false;
	}
	public boolean isOnBaseGridY(double y) {
		double c = Math.abs(y % mModule);
		if (c < ToolMath.EPS || Math.abs(c - mModule) < ToolMath.EPS) {
			return true;
		}
		return false;
	}

	public double getNearGridX2(double x, boolean dir) {
		if (getCurrentGrid() == 0)
			return x;

		int ss = (int)Math.floor(((x - getGridCenterX()) / getCurrentGrid()));

		double candx = getGridCenterX() + ss * getCurrentGrid();
		double candx1, candx2;
		if (candx < x) {
			candx1 = candx;
			candx2 = getGridCenterX() + (ss + 1) * getCurrentGrid();
		} else {
			candx1 = getGridCenterX() + (ss - 1) * getCurrentGrid();
			candx2 = candx;
		}
		if (Math.abs(candx1 - x) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE) {
			return 	candx1;
		}
		if (Math.abs(candx2 - x) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE) {
			return 	candx2;
		}
		if (mExtraGridX != null) {
			for (int i = 0; i < mExtraGridX.size(); i++) {
				double f = mExtraGridX.elementAt(i).doubleValue();
				if (Math.abs(f - x) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE)
					return 	f;
				if (candx1 < f && x > f)
					candx1 = f;
				if (x < f && candx2 > f)
					candx2 = f;
			}
		}

		if (dir)
			return candx2;
		return candx1;
	}

	public double getNearGridY2(double y, boolean dir) {
		if (getCurrentGrid() == 0)
			return y;

		int ss = (int)Math.floor(((y - getGridCenterY()) / getCurrentGrid()));

		double candy = getGridCenterY() + ss * getCurrentGrid();
		double candy1, candy2;
		if (candy < y) {
			candy1 = candy;
			candy2 = getGridCenterY() + (ss + 1) * getCurrentGrid();
		} else {
			candy1 = getGridCenterY() + (ss - 1) * getCurrentGrid();
			candy2 = candy;
		}
		if (Math.abs(candy1 - y) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE) {
			return 	candy1;
		}
		if (Math.abs(candy2 - y) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE) {
			return 	candy2;
		}
		if (mExtraGridY != null) {
			for (int i = 0; i < mExtraGridY.size(); i++) {
				double f = mExtraGridY.elementAt(i).doubleValue();
				if (Math.abs(f - y) < getCurrentGrid() / NEAR_GRID2_THRESHOLD_DIVIDE)
					return 	f;
				if (candy1 < f && y > f)
					candy1 = f;
				if (y < f && candy2 > f)
					candy2 = f;
			}
		}

		if (dir) {
			return candy2;
		}
		return candy1;
	}

	/**
	 * 追加グリッドを抜いたグリッドの取得
	 * @param x
	 * @param bx
	 * @param module
	 * @return
	 */
	public int getNearNotExtraGrid(double x, double bx, double module) {
		double near = Double.MAX_VALUE;
		int step = x <= bx ? -1:1;
		int c = (int)Math.floor(((x - bx) / module) * step);
		while (true) {
			double len = Math.abs(x - (bx + module * c * step));
			if (near > len)
				near = len;
			else
				break;
			c++;
		}
		return step * (c - 1);
	}

	public Vector2<Double> getExtraGridX() {
		return mExtraGridX;
	}

	public Vector2<Double> getExtraGridY() {
		return mExtraGridY;
	}

	public boolean addExtraGridX(double x) {
		if (Math.abs(getNearGridX(x, true) - x) < ToolMath.EPS)
			return false;
		if (mExtraGridX == null)
			mExtraGridX = new Vector2<Double>();

		for (int i = 0; i < mExtraGridX.size(); i++) {
			Double f = mExtraGridX.elementAt(i);
			if (f.doubleValue() == x)
				return false;
		}
		mExtraGridX.addElement(new Double(x));
		return true;
	}

//	public void removeExtraGridX(double x) {
//		if (mExtraGridX == null)
//			return;
//		for (int i = 0; i < mExtraGridX.size(); i++) {
//			Double f = (Double)mExtraGridX.elementAt(i);
//			if (Math.abs(f.doubleValue() - x) < ToolMath.EPS) {
//				mExtraGridX.removeElement(f);
//				i--;
//			}
//		}
//	}

	public boolean addExtraGridY(double y) {
		if (Math.abs(getNearGridY(y, true) - y) < ToolMath.EPS)
			return false;
		if (mExtraGridY == null)
			mExtraGridY = new Vector2<Double>();
		for (int i = 0; i < mExtraGridY.size(); i++) {
			Double f = mExtraGridY.elementAt(i);
			if (Math.abs(f.doubleValue() - y) < ToolMath.EPS)
				return false;
		}
		mExtraGridY.addElement(new Double(y));
		return true;
	}
//
//	public void removeExtraGridY(double y) {
//		for (int i = 0; i < mExtraGridY.size(); i++) {
//			Double f = (Double)mExtraGridY.elementAt(i);
//			if (Math.abs(f.doubleValue() - y) < ToolMath.EPS) {
//				mExtraGridX.removeElement(f);
//				i--;
//			}
//		}
//	}
}
