/*
 * Copyright (c) 2007 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;



/**
 * 時間ユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class TimeUtil {

	//------------------------------------------------------------------
	//- constants
	//------------------------------------------------------------------

	/** １日あたりの秒数 */
	public static final int SECOND_OF_DAY = 86400;

	/** １日あたりのミリ秒数 */
	public static final long MILLISECOND_OF_DAY = SECOND_OF_DAY * 1000;

	/** YYYYMMDD形式のフォーマット */
	public static final String FORMAT_YYYYMMDD = "yyyyMMdd";

	/** YYYY/MM/DD形式のフォーマット */
	public static final String FORMAT_YYYY_MM_DD = "yyyy/MM/dd";

	/** YYYYMMDDhhmmss形式のフォーマット */
	public static final String FORMAT_YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

	/** YYYY/MM/DD hh:mm:ss形式のフォーマット */
	public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy/MM/dd HH:mm:ss";



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	private TimeUtil() {
	}



	//-----------------------------------------------------------------------
	//- methods
	//-----------------------------------------------------------------------

	/**
	 * Date オブジェクトで表される、1970 年 1 月 1 日 00:00:00 GMT からのミリ秒数を返します。
	 *
	 * @return この日付で表される、1970 年 1 月 1 日 00:00:00 GMT からのミリ秒数
	 */
	public static long getTime() {
		return (new Date()).getTime();
	}

	/**
	 * Timestampを生成する。
	 *
	 * @return Timestamp
	 */
	public static Timestamp createTimestamp() {
		return createTimestamp(getTime());
	}

	/**
	 * Timestampを生成する。
	 *
	 * @param millis 現在時刻
	 * @return Timestamp
	 */
	public static Timestamp createTimestamp(long millis) {
		return new Timestamp(millis);
	}

	/**
	 * 文字列を指定したフォーマットで解析しTimestampを生成します。<br/>
	 * フォーマットの詳細はSimpleDateFormatの説明を参照してください。<br/>
	 *
	 * @param format 解析用フォーマット
	 * @param src 解析対象文字列
	 * @return 解析できた場合はTimestamp。できなかった場合はnullを返す。
	 */
	public static Timestamp parseTimestamp(String format, String src) {
		Timestamp result = null;
		try {
			if (src == null || src.equals("")) return null;

			SimpleDateFormat sdf = new SimpleDateFormat(format);
			sdf.setLenient(false);
			result = new Timestamp(sdf.parse(src).getTime());
		} catch (Exception e) {
			result = null;
		}
		return result;
	}

	/**
	 * 現在実行中のスレッドを、指定されたミリ秒数の間、スリープさせます。
	 *
	 * @param millis ミリ秒単位の停止時間
	 */
	public static void sleep(long millis) {
		try {
			Thread.sleep(millis);
		} catch (InterruptedException e) {
		}
	}

	/**
	 * 時間に加減算を行います。<br/>
	 * fieldはCalendarクラスで定義されている、年月日時分秒を表す値です。<br/>
	 *
	 * @param src 対象時間
	 * @param field 対象フィールド
	 * @param amount 加減算値
	 * @return 加減算後の時間
	 */
	public static long add(long src, int field, int amount) {
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(src);
		cal.add(field, amount);
		return cal.getTimeInMillis();
	}

	/**
	 * Timestampに加減算を行います。<br/>
	 * fieldはCalendarクラスで定義されている、年月日時分秒を表す値です。<br/>
	 *
	 * @param src 対象のTimestamp
	 * @param field 対象フィールド
	 * @param amount 加減算値
	 * @return 加減算後のTimestamp
	 */
	public static Timestamp add(Timestamp timestamp, int field, int amount) {
		return createTimestamp(add(timestamp.getTime(), field, amount));
	}

	/**
	 * Timestampに任意の日数を加えます。
	 *
	 * @param timestamp 加工対象のTimestamp
	 * @param amount 加工する日数
	 * @return 加工されたTimestamp
	 */
	public static Timestamp addDay(Timestamp timestamp, int amount) {
		return add(timestamp, Calendar.DATE, amount);
	}

	/**
	 * Timestampに任意の月数を加えます。
	 *
	 * @param timestamp 加工対象のTimestamp
	 * @param amount 加工する月数
	 * @return 加工されたTimestamp
	 */
	public static Timestamp addMonth(Timestamp timestamp, int amount) {
		return add(timestamp, Calendar.MONTH, amount);
	}

	/**
	 * 日付・時刻の切り捨てを行ないます。<br/>
	 * fieldはCalendarクラスで定義されている、年月日時分秒を表す値です。<br/>
	 * fieldには「どの値までを有効とするか」を指定して下さい。<br/>
	 * <br/>
	 * 例: fieldにCalendar.HOUR_OF_DAYを指定した場合は<br/>
	 * 　　    「時」までが有効となり、分・秒・ミリ秒が切り捨てられます。<br/>
	 *
	 * @param src 対象時間(エポックからの UTC ミリ秒値)
	 * @param field 対象フィールド
	 * @return 変換後の時間(エポックからの UTC ミリ秒値)
	 */
	public static long truncate(long src, int field) {
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(src);

		if (field == Calendar.MILLISECOND) return cal.getTimeInMillis();
		cal.set(Calendar.MILLISECOND, 0);

		if (field == Calendar.SECOND) return cal.getTimeInMillis();
		cal.set(Calendar.SECOND, 0);

		if (field == Calendar.MINUTE) return cal.getTimeInMillis();
		cal.set(Calendar.MINUTE, 0);

		if (field == Calendar.HOUR_OF_DAY) return cal.getTimeInMillis();
		cal.set(Calendar.HOUR_OF_DAY, 0);

		if (field == Calendar.DAY_OF_MONTH) return cal.getTimeInMillis();
		cal.set(Calendar.DAY_OF_MONTH, 1);

		if (field == Calendar.MONTH) return cal.getTimeInMillis();
		cal.set(Calendar.MONTH, 0);

		return cal.getTimeInMillis();
	}

	/**
	 * 日付・時刻の切り捨てを行ないます。
	 *
	 * @param timestamp 対象のTimestamp
	 * @param field 対象フィールド
	 * @return 変換後のTimestamp
	 * @see #truncate(long, int)
	 */
	public static Timestamp truncate(Timestamp timestamp, int field) {
		return createTimestamp(truncate(timestamp.getTime(), field));
	}

	/**
	 * 時刻の切り捨てを行ないます。
	 *
	 * @param timestamp 対象のTimestamp
	 * @return 変換後のTimestamp
	 */
	public static Timestamp truncateTime(Timestamp timestamp) {
		return createTimestamp(truncate(timestamp.getTime(), Calendar.DAY_OF_MONTH));
	}

	/**
	 * 指定フィールドまでの時刻フィールドでの時刻比較を行ないます。<br/>
	 * fieldはCalendarクラスで定義されている、年月日時分秒を表す値です。<br/>
	 * <br/>
	 * 例: fieldにCalendar.HOUR_OF_DAYを指定した場合は<br/>
	 * 　    　「時」までが有効となり、年月日時での比較となります。<br/>
	 *
	 * @param t1 比較時刻1
	 * @param t2 比較時刻2
	 * @param field 対象フィールド
	 * @return 同等の値ならば0、t1が大きいならば-1、t2が大きいならば1を返す。
	 */
	public static int diff(long t1, long t2, int field) {
		long truncateTime1 = truncate(t1, field);
		long truncateTime2 = truncate(t2, field);

		int result = 0;
		if (truncateTime1 != truncateTime2) {
			result = (truncateTime1 < truncateTime2)? 1: -1;
		}
		return result;
	}

	/**
	 * 指定フィールドまでの時刻フィールドでの時刻比較を行ないます。<br/>
	 * fieldはCalendarクラスで定義されている、年月日時分秒を表す値です。<br/>
	 * <br/>
	 * 例: fieldにCalendar.HOUR_OF_DAYを指定した場合は<br/>
	 * 　    　「時」までが有効となり、年月日時での比較となります。<br/>
	 *
	 * @param t1 比較時刻1
	 * @param t2 比較時刻2
	 * @param field 対象フィールド
	 * @return 同等の値ならば0、t1が大きいならば-1、t2が大きいならば1を返す。
	 */
	public static int diff(Timestamp t1, Timestamp t2, int field) {
		long epochTime1 = (t1 == null)? 0: t1.getTime();
		long epochTime2 = (t2 == null)? 0: t2.getTime();
		return diff(epochTime1, epochTime2, field);
	}



} // end-class
