/*
 * Copyright (c) 2008-2010 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * リフレクションユーティリティクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class ReflectUtil {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log _log = LogFactory.getLog(ReflectUtil.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	protected ReflectUtil() {
	}



	//-----------------------------------------------------------------------
	//- methods
	//-----------------------------------------------------------------------

	/**
	 * 指定のクラスを生成します。<br/>
	 * private/protected/packageのクラスも生成できます。<br/>
	 *
	 * @param className 生成するクラス名
	 * @param params コンストラクタに渡すパラメータ
	 * @return コンストラクタ
	 */
	@SuppressWarnings("unchecked")
	public static <T> T newInstance(final String className, final Object... params) {
		return (T)newInstance(getClass(className), params);
	}

	/**
	 * 指定のクラスを生成します。<br/>
	 * private/protected/packageのクラスも生成できます。<br/>
	 *
	 * @param clazz 生成するクラス
	 * @param params コンストラクタに渡すパラメータ
	 * @return コンストラクタ
	 */
	@SuppressWarnings("unchecked")
	public static <T> T newInstance(final Class<T> clazz, Object... params) {
		T result = null;
		try {
			Class<?>[] paramTypes = getParameterTypes(params);
			Constructor<?> constructor = getConstructor(clazz, paramTypes);
			if (!constructor.isAccessible()) constructor.setAccessible(true);

			result = (T)constructor.newInstance(params);
		} catch (Exception e) {
			_log.info("", e);
		}
		return result;
	}

	/**
	 * 指定のメソッドを実行します。<br/>
	 * private/protected/packageのメソッドも実行できます。<br/>
	 * staticメソッドのみ実行できます。<br/>
	 *
	 * @param className 実行するクラス名
	 * @param methodName メソッド名
	 * @param params メソッドに渡すパラメータ
	 * @return メソッドからの戻り値
	 */
	public static Object invokeMethod(String className, String methodName, Object... params) {
		return invokeMethod(getClass(className), methodName, params);
	}

	/**
	 * 指定のメソッドを実行します。<br/>
	 * private/protected/packageのメソッドも実行できます。<br/>
	 * staticメソッドのみ実行できます。<br/>
	 *
	 * @param clazz 実行するクラス
	 * @param methodName メソッド名
	 * @param params メソッドに渡すパラメータ
	 * @return メソッドからの戻り値
	 */
	public static Object invokeMethod(Class<?> clazz, String methodName, Object... params) {
		Class<?>[] paramTypes = getParameterTypes(params);
		Method method = getMethod(clazz, methodName, paramTypes);
		return invokeMethod(clazz, method, params);
	}

	/**
	 * 指定のメソッドを実行します。<br/>
	 * private/protected/packageのメソッドも実行できます。<br/>
	 * instance/staticメソッドともに実行できます。<br/>
	 *
	 * @param instance 実行するクラスのインスタンス。
	 * @param methodName メソッド名
	 * @param params メソッドに渡すパラメータ
	 * @return メソッドからの戻り値
	 */
	public static Object invokeMethod(Object instance, String methodName, Object... params) {
		Class<?>[] paramTypes = getParameterTypes(params);
		Method method = getMethod(instance, methodName, paramTypes);
		return invokeMethod(instance, method, params);
	}

	/**
	 * 指定のメソッドを実行します。<br/>
	 * private/protected/packageのメソッドも実行できます。<br/>
	 * instance/staticメソッドともに実行できます。<br/>
	 *
	 * @param instance 実行対象クラスのインスタンス。staticの場合null。
	 * @param method 実行するメソッド
	 * @param parameters メソッドに渡すパラメータ
	 * @return メソッドからの戻り値
	 */
	public static Object invokeMethod(Object instance, Method method, Object... values) {
		if (instance == null) return null;
		if (method == null) return null;

		Object result = null;
		try {
			if (!method.isAccessible()) method.setAccessible(true);
			result = method.invoke(instance, values);
		} catch (Exception e) {
			_log.info("", e);
		}
		return result;
	}

	/**
	 * 指定のフィールドに値を設定します。<br/>
	 * private/protected/packageのフィールドも設定できます。<br/>
	 * staticフィールドのみ設定できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param fieldName フィールド名
	 * @param value 設定値
	 * @return 設定できた場合true。
	 */
	public static boolean setFieldValue(Class<?> clazz, String fieldName, Object value) {
		return setFieldValue(null, getField(clazz, fieldName), value);
	}

	/**
	 * 指定のフィールドに値を設定します。<br/>
	 * private/protected/packageのフィールドも設定できます。<br/>
	 * instance/staticフィールドともに設定できます。<br/>
	 *
	 * @param instance 対象インスタンス。
	 * @param fieldName フィールド名
	 * @param value 設定値
	 * @return 設定できた場合true。
	 */
	public static boolean setFieldValue(Object instance, String fieldName, Object value) {
		if (instance == null) return false;
		return setFieldValue(instance, getField(instance, fieldName), value);
	}

	/**
	 * 指定のフィールドに値を設定します。<br/>
	 * private/protected/packageのフィールドも設定できます。<br/>
	 * instance/staticフィールドともに設定できます。<br/>
	 *
	 * @param instance 対象インスタンス。staticの場合はnull。
	 * @param field フィールド
	 * @param value 設定値
	 * @return 設定できた場合true。
	 */
	public static boolean setFieldValue(Object instance, Field field, Object value) {
		if (field == null) return false;
		if (field.getType().isPrimitive() && value == null) return false;

		boolean result = false;
		try {
			if (!field.isAccessible()) field.setAccessible(true);
			field.set(instance, value);
			result = true;
		} catch (Exception e) {
			_log.info("", e);
		}
		return result;
	}

	/**
	 * 指定のフィールドから値を取得します。<br/>
	 * private/protected/packageのフィールドも取得できます。<br/>
	 * staticフィールド値のみ取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param fieldName フィールド名
	 * @return フィールド値
	 */
	public static Object getFieldValue(Class<?> clazz, String fieldName) {
		return getFieldValue(null, getField(clazz, fieldName));
	}

	/**
	 * 指定のフィールドから値を取得します。<br/>
	 * private/protected/packageのフィールドも取得できます。<br/>
	 * instance/staticフィールド値ともに取得できます。<br/>
	 *
	 * @param instance 対象インスタンス
	 * @param fieldName フィールド名
	 * @return フィールド値
	 */
	public static Object getFieldValue(Object instance, String fieldName) {
		if (instance == null) return null;
		return getFieldValue(instance, getField(instance, fieldName));
	}

	/**
	 * 指定のフィールドから値を取得します。<br/>
	 * private/protected/packageのフィールドも取得できます。<br/>
	 * instance/staticフィールド値ともに取得できます。<br/>
	 *
	 * @param instance 対象インスタンス。staticの場合null。
	 * @param field フィールド
	 * @return フィールド値
	 */
	public static Object getFieldValue(final Object instance, final Field field) {
		if (field == null) return null;

		Object result = null;
		try {
			if (!field.isAccessible()) field.setAccessible(true);
			result = field.get(instance);
		} catch (Exception e) {
			_log.info("", e);
		}
		return result;
	}

	/**
	 * 指定のクラス型を取得します。<br/>
	 *
	 * @param clazzName クラス名
	 * @return クラス型
	 */
	public static Class<?> getClass(String clazzName) {
		if (clazzName == null) return null;
		ClassInfo clazzInfo = ClassInfoManager.getInstance().getClassInfo(clazzName);
		if (clazzInfo == null) throw new IllegalStateException(clazzName + " not found.");
		return clazzInfo.getType();
	}

	/**
	 * 指定のコンストラクタを取得します。<br/>
	 * private/protected/packageのコンストラクタも取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param paramTypes コンストラクタのパラメータ型
	 * @return コンストラクタ
	 */
	public static Constructor<?> getConstructor(Class<?> clazz, Class<?>... paramTypes) {
		if (clazz == null) return null;
		ClassInfo clazzInfo = ClassInfoManager.getInstance().getClassInfo(clazz);
		if (clazzInfo == null) throw new IllegalStateException(clazz + " not found.");
		Constructor<?> constructor = clazzInfo.getConstructor(paramTypes);
		if (constructor == null) throw new IllegalStateException(clazz + "'s constructor not found.");
		return constructor;
	}

	/**
	 * 指定のメソッドを取得します。<br/>
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param instance 対象インスタンス
	 * @param methodName メソッド名
	 * @param paramTypes メソッドのパラメータ型
	 * @return メソッド
	 */
	public static Method getMethod(Object instance, String methodName, Class<?>... paramTypes) {
		if (instance == null) return null;
		return getMethod(instance.getClass(), methodName, paramTypes);
	}

	/**
	 * 指定のメソッドを取得します。<br/>
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param methodName メソッド名
	 * @param paramTypes メソッドのパラメータ型
	 * @return メソッド
	 */
	public static Method getMethod(Class<?> clazz, String methodName, Class<?>... paramTypes) {
		if (clazz == null) return null;
		if (methodName == null) return null;
		return ClassInfoManager.getInstance().getClassInfo(clazz).getMethod(methodName, paramTypes);
	}

	/**
	 * 全メソッドを取得します。
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param instance 対象インスタンス
	 * @return メソッドコレクション
	 */
	public static Collection<Method> getMethods(Object instance) {
		if (instance == null) return null;
		return ClassInfoManager.getInstance().getClassInfo(instance.getClass()).getMethods();
	}

	/**
	 * 全メソッドを取得します。
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @return メソッドコレクション
	 */
	public static Collection<Method> getMethods(Class<?> clazz) {
		if (clazz == null) return null;
		return ClassInfoManager.getInstance().getClassInfo(clazz).getMethods();
	}

	/**
	 * 指定のフィールドを取得します。<br/>
	 * private/protected/package/staticのフィールドも取得できます。<br/>
	 *
	 * @param clazz 対象インスタンス
	 * @param fieldName フィールド名
	 * @return フィールド
	 */
	public static Field getField(Object instance, String fieldName) {
		if (instance == null) return null;
		return getField(instance.getClass(), fieldName);
	}

	/**
	 * 指定のフィールドを取得します。<br/>
	 * private/protected/package/staticのフィールドも取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param fieldName フィールド名
	 * @return フィールド
	 */
	public static Field getField(Class<?> clazz, String fieldName) {
		if (clazz == null) return null;
		if (StringUtil.isEmpty(fieldName)) return null;
		return ClassInfoManager.getInstance().getClassInfo(clazz).getField(fieldName);
	}

	/**
	 * 全フィールドを取得します。
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param instance 対象インスタンス
	 * @return フィールドコレクション
	 */
	public static Collection<Field> getFields(Object instance) {
		return ClassInfoManager.getInstance().getClassInfo(instance.getClass()).getFields();
	}

	/**
	 * 全フィールドを取得します。
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param clazz 対象クラス
	 * @return フィールドコレクション
	 */
	public static Collection<Field> getFields(Class<?> clazz) {
		return ClassInfoManager.getInstance().getClassInfo(clazz).getFields();
	}

	/**
	 * コンストラクタが存在するかチェックします。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param paramTypes パラメータ型配列
	 * @return 存在する場合true。
	 */
	public static boolean existConstructor(final Class<?> clazz, final Class<?>... paramTypes) {
		return ClassInfoManager.getInstance().getClassInfo(clazz).existConstructor(paramTypes);
	}

	/**
	 * メソッドが存在するかチェックします。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param methodName メソッド名
	 * @param paramTypes パラメータ型配列
	 * @return 存在する場合true。
	 */
	public static boolean existMethod(final Class<?> clazz, final String methodName, final Class<?>... paramTypes) {
		return ClassInfoManager.getInstance().getClassInfo(clazz).existMethod(methodName, paramTypes);
	}

	/**
	 * フィールドが存在するかチェックします。<br/>
	 *
	 * @param clazz 対象クラス
	 * @param fieldName フィールド名
	 * @return 存在する場合true。
	 */
	public static boolean existField(final Class<?> clazz, final String fieldName) {
		return ClassInfoManager.getInstance().getClassInfo(clazz).existField(fieldName);
	}

	/**
	 * パラメータ値の型を取得します。
	 *
	 * @param parameters パラメータ値
	 * @return パラメータ型
	 */
	public static Class<?>[] getParameterTypes(final Object... parameters) {
		if (parameters == null) return null;

		Class<?>[]  parameterTypes = new Class[parameters.length];
		for (int i = 0; i < parameters.length; i++) {
			if (parameters[i] == null) continue;
			parameterTypes[i] = parameters[i].getClass();
		}
		return parameterTypes;
	}

	/**
	 * 対象インスタンスに定義されている、ジェネリックス型コレクションを取得する。
	 *
	 * @param targetInstance インスタンス
	 * @return ジェネリックス型コレクション
	 */
	public static Collection<Class<?>> getGenericParameterClasses(Object targetInstance) {
		return getGenericParameterClasses(targetInstance.getClass());
	}

	/**
	 * 対象クラスに定義されている、ジェネリックス型コレクションを取得する。
	 *
	 * @param targetClazz 型
	 * @return ジェネリックス型コレクション
	 */
	public static Collection<Class<?>> getGenericParameterClasses(Class<?> targetClazz) {
		if (targetClazz == null) return null;
		return ClassInfoManager.getInstance().getClassInfo(targetClazz).getGenericParameterTypes();
	}



} // end-class
