package jp.kitec.lib.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.kitec.lib.io.FileIOUtil;

public class MessageUtil {

	private static final Log log = LogFactory.getLog(MessageUtil.class);

	public static final int MAGIC = 0x6e683435;

	public static final int MAGIC_0 = 0x6e;
	public static final int MAGIC_1 = 0x68;
	public static final int MAGIC_2 = 0x34;
	public static final int MAGIC_3 = 0x35;

	/**
	 * オブジェクト受信
	 * @param  is 入力ストリーム
	 * @return 受信データ（圧縮されていれば解凍）
	 */
	public static byte[] receiveBytes(InputStream is) {
		return receiveBytes(is, true, true);
	}

	/**
	 * オブジェクト受信
	 * @param  is 入力ストリーム
	 * @param decompress データを解凍するか
	 * @return 受信データ（圧縮されていれば解凍）
	 */
	public static byte[] receiveBytes(InputStream is, boolean decompress, boolean checkMagic) {
		if (checkMagic) {
			byte[] h = new byte[4];
			try {
				int r = is.read(h);
				if (r != 4) {
					log.error("read header error " + r);
					return null;
				}
			} catch (IOException e) {
				log.error("read header error", e);
				return null;
			}
			if (h[0] != MAGIC_0
					|| h[1] != MAGIC_1
					|| h[2] != MAGIC_2
					|| h[3] != MAGIC_3) {
				log.error("bad header:"
					+ Integer.toString(0xff & h[0]) + "/"
					+ Integer.toString(0xff & h[1]) + "/"
					+ Integer.toString(0xff & h[2]) + "/"
					+ Integer.toString(0xff & h[3]));
				return null;
			}
		}
		byte[] res = FileIOUtil.readBytes(is);
		if (decompress)
			res = CompressData.decompress(res);
		return res;
	}


	/**
	 * オブジェクト送信
	 * @param res HTTPレスポンス
	 * @param oo  送信オブジェクト
	 */
	public static void sendBytes(OutputStream out, byte[] o) {
		sendBytes(out, o, true, true);
	}

	/**
	 * オブジェクト送信
	 *
	 * @param out 出力ストリーム
	 * @param o 送信オブジェクト
	 * @param compress データを圧縮する
	 */
	public static void sendBytes(OutputStream out, byte[] o, boolean compress, boolean addmagic) {
		BufferedOutputStream os = null;
		try {
			os = new BufferedOutputStream(out);
			if (addmagic) {
				os.write(MAGIC_0);
				os.write(MAGIC_1);
				os.write(MAGIC_2);
				os.write(MAGIC_3);
			}
			if (compress)
				os.write(CompressData.compress(o));
			else
				os.write(o);
			os.flush();
			os.close();
		} catch (IOException e) {
			log.error("", e);
		} finally {
			FileIOUtil.close(os);
		}
	}

	/**
	 * オブジェクト受信
	 * @param  in 入力ストリーム
	 * @return 受信データ（圧縮されていれば解凍）
	 */
	public static Object recvDataUnsafe(InputStream in) {
		ObjectInputStream ois = null;
		try {
			ois = new ObjectInputStream(new BufferedInputStream(in));
			Object res = ois.readObject();
			ois.close();
			if (res != null && res instanceof byte[]) {
				Object o =  CompressData.decompressUnsafe(res);
				return o;
			}
			return res;
		} catch (IOException e) {
			log.error("", e);
		} catch (ClassNotFoundException e) {
			log.error("", e);
		}
		return null;
	}

	/**
	 * オブジェクト送信
	 * @param out 出力ストリーム
	 * @param o  送信オブジェクト
	 */
	public static void sendDataUnsafe(OutputStream out, Object o) throws IOException {
		sendDataUnsafe(out, o, false);
	}

	/**
	 *
	 * @param out 出力ストリーム
	 * @param o 送信オブジェクト
	 * @param compressedData 圧縮済みデータかのフラグ
	 * @author kawae
	 * @since 2005/11/21
	 */
	public static void sendDataUnsafe(OutputStream out, Object o, boolean compressedData) {
		ObjectOutputStream os = null;
		try {
			os = new ObjectOutputStream(new BufferedOutputStream(out));
			if (compressedData) {
				os.writeObject(o);
			} else {
				os.writeObject(CompressData.compressUnsafe(o));
			}
			os.flush();
			os.close();
		} catch (IOException e) {
			log.error("", e);
		} finally {
			FileIOUtil.close(os);
		}
	}
}
