/*
 * Copyright (c) 2011 KITec Inc,.. All rights reserved.
 */

/*
 * Created on 2004/08/26
 */
package jp.kitec.lib.kwt;

import java.io.Serializable;

/**
 * @author kawae
 */
public class KColor implements KFill, Serializable {
	private static final long serialVersionUID = 1L;

	/**
	 * 明るい黄色
	 */
	public static final KColor LIGHT_YELLOW = new KColor(255, 255, 192);
	/**
	 * 濃い黄色
	 */
	public static final KColor DARK_YELLOW = new KColor(192, 192, 0);
	/**
	 * 空色
	 */
	public static final KColor SKY_BLUE = new KColor(128, 128, 255);
	/**
	 * 水色
	 */
	public static final KColor WATER = new KColor(147, 221, 221);
	/**
	 * 濃い青
	 */
	public static final KColor DARK_BLUE = new KColor(0, 0, 128);
	/**
	 * 明るい緑
	 */
	public static final KColor LIGHT_GREEN = new KColor(128, 255, 128);
	/**
	 * 濃い緑
	 */
	public static final KColor DARK_GREEN = new KColor(0, 192, 0);
	/**
	 * かなり明るい灰色
	 */
	public static final KColor MORELIGHT_GRAY = new KColor(220, 220, 220);
	/**
	 * さらに明るい灰色
	 */
	public static final KColor SUPERLIGHT_GRAY = new KColor(240, 240, 240);
	/**
	 * 明るい灰色
	 */
	public static final KColor LIGHT_GRAY2 = new KColor(212, 208, 200);


	public static final KColor WHITE = new KColor(255, 255, 255);
	public static final KColor LIGHT_GRAY = new KColor(192, 192, 192);
	public static final KColor GRAY = new KColor(128, 128, 128);
	public static final KColor DARK_GRAY = new KColor(64, 64, 64);
	public static final KColor BLACK = new KColor(0, 0, 0);
	public static final KColor RED = new KColor(255, 0, 0);
	public static final KColor PINK = new KColor(255, 175, 175);
	public static final KColor ORANGE = new KColor(255, 200, 0);
	public static final KColor YELLOW = new KColor(255, 255, 0);
	public static final KColor GREEN = new KColor(0, 255, 0);
	public static final KColor MAGENTA = new KColor(255, 0, 255);
	public static final KColor CYAN = new KColor(0, 255, 255);
	public static final KColor BLUE = new KColor(0, 0, 255);

	private static final double SCALE_FACTOR = 0.7;
	private static final int MIN_SCALABLE = 3;

	final int argb;

	/**
	 * コンストラクタ
	 *
	 * @param rgb
	 */
	public KColor(int rgb) {
		argb = rgb | (0xff << 24);
	}

	/**
	 * コンストラクタ
	 *
	 * @param argb
	 * @param hasAlpha
	 */
	public KColor(int argb, boolean hasAlpha) {
		if (hasAlpha)
			this.argb = argb;
		else
			this.argb = argb | (0xff << 24);
	}

	/**
	 * コンストラクタ
	 *
	 * @param r
	 * @param g
	 * @param b
	 */
	public KColor(int r, int g, int b) {
		argb = b | (g << 8) | (r << 16) | (0xff << 24);
	}

	/**
	 * コンストラクタ
	 *
	 * @param r
	 * @param g
	 * @param b
	 * @param alpha
	 */
	public KColor(int r, int g, int b, int alpha) {
		argb = b | (g << 8) | (r << 16) | (alpha << 24);
	}

	public KColor(double r, double g, double b, double a) {
		this((int)(r*255+0.5), (int)(g*255+0.5), (int)(b*255+0.5), (int)(a*255+0.5));
	}

	public int getRed() {
		return (argb >> 16) & 0xFF;
	}

	public int getRGB() {
		return argb;
	}

	public int getGreen() {
		return (argb >> 8) & 0xFF;
	}

	public int getBlue() {
		return argb & 0xFF;
	}

	public int getAlpha() {
		return (argb >> 24) & 0xFF;
	}

	/**
	 * ARGBを16進数で返す
	 * @return
	 */
	@Override
	public String encode() {
		 String fill = "0";
		 StringBuilder sb = new StringBuilder();
		 sb.append("#");
		 String s;
		 s = Integer.toHexString(getAlpha());
		 if (s.length() == 1)
			 sb.append(fill);
		 sb.append(s);

		 s = Integer.toHexString(getRed());
		 if (s.length() == 1)
			 sb.append(fill);
		 sb.append(s);

		 s = Integer.toHexString(getGreen());
		 if (s.length() == 1)
			 sb.append(fill);
		 sb.append(s);

		 s = Integer.toHexString(getBlue());
		 if (s.length() == 1)
			 sb.append(fill);
		 sb.append(s);
		 return sb.toString();
	}

	public KColor darker() {
		return new KColor(
				(int)(getRed()*SCALE_FACTOR),
				(int)(getGreen()*SCALE_FACTOR),
				(int)(getBlue()*SCALE_FACTOR));
	}

	public KColor brighter() {

		int r = getRed();
		int b = getBlue();
		int g = getGreen();

		if(r == 0 && b == 0 && g == 0) {
			return new KColor(MIN_SCALABLE, MIN_SCALABLE, MIN_SCALABLE);
		}

		if(r < MIN_SCALABLE && r != 0) {
			r = MIN_SCALABLE;
		} else {
			r = (int) (r/SCALE_FACTOR);
			r = (r > 255) ? 255 : r;
		}

		if(b < MIN_SCALABLE && b != 0) {
			b = MIN_SCALABLE;
		} else {
			b = (int) (b/SCALE_FACTOR);
			b = (b > 255) ? 255 : b;
		}

		if(g < MIN_SCALABLE && g != 0) {
			g = MIN_SCALABLE;
		} else {
			g = (int) (g/SCALE_FACTOR);
			g = (g > 255) ? 255 : g;
		}

		return new KColor(r, g, b);
	}

	@Override
	public int hashCode() {
		return argb;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		KColor other = (KColor) obj;
		if (argb != other.argb)
			return false;
		return true;
	}

	/**
	 * カンマ区切りのRGBからColorを作成
	 *
	 * @param s
	 * @return
	 * @author kawae
	 * @since 2004/10/12
	 */
	public static KColor getColorFromRGBString(String s) {
		try {
			String[] st = s.split(",");
			if (st.length != 3)
				return null;
			int r = Integer.valueOf(st[0]).intValue();
			int g = Integer.valueOf(st[1]).intValue();
			int b = Integer.valueOf(st[2]).intValue();
			return new KColor(r, g, b);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/**
	 * カンマ区切りのRGB文字列の取得
	 *
	 * @param c
	 * @return
	 * @author kawae
	 * @since 2004/11/05
	 */
	public static String getColorRGBString(KColor c) {
		if (c != null)
			return c.getRed() + "," + c.getGreen() + "," + c.getBlue();
		return null;
	}

	/**
	 * nullを許容する比較
	 * @param c1
	 * @param c2
	 * @return
	 */
	public static boolean isDiff(KColor c1, KColor c2) {
		if ((c1 == null) != (c2 == null))
			return true;
		if (c1 == null)
			return false;
		return c1.getRGB() != c2.getRGB();
	}

	public static boolean equals(KColor c1, KColor c2) {
		return !isDiff(c1, c2);
	}

	/**
	 * アルファ値を使ってKColorオブジェクトを作成する
	 * @param val
	 * @return
	 */
	public static KColor decode(String val) {
		Long n = Long.decode(val);
		return new KColor(n.intValue(), true);
	}

	@Override
	public String toString() {
		return "KColor[" + getAlpha() + ", " + getRed() + ", " + getGreen() + ", " + getBlue() + "]";
	}
}
