package jp.kitec.lib.util.xml;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.StringTokenizer;

import jp.kitec.lib.io.AbstFile;
import jp.kitec.lib.io.Base64Decoder;
import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;


/**
 * XML 形式データの読み込みを行う。
 *
 * @since		2004/05/06
 * @author		fujita
 * @version	2004/05/06
 *
 * Copyright (c) 2004 KITec Inc,.. All rights reserved.
 */
public class XMLReader {

	/** コメントの開始／終了文字群 */
	private static final String[][] COMMENT = {
		{ "<?",   "?>"  },
		{ "<!--", "-->" }
	};

	/** データの読み込み元オブジェクト */
	protected AbstFile mInput;
	/** データ読み込みできるかどうかを表すフラグ */
	protected boolean mCanRead;
	/** データ（ファイル）の終端かどうかを表すフラグ */
	protected boolean mEOF;
	/** 読み込み途中の残データ */
	private String   mRemain = "";
	/** 読み込み用のリーダー */
	private BufferedReader mBr;

	
	/**
	 * 指定されたファイル（名）を、データの読み込み元とする
	 * 新たな XMLReader を作成する。
	 * 
	 * @param		file	ファイル（名）
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 * @throws	FileNotFoundException	ファイルが見つからないか、開けなかった場合
	 */
	public XMLReader(String file) {
		try {
			mCanRead = setFile(new FileInputStream(file), file);
		} catch (FileNotFoundException e) {
			throw new RuntimeException("File not found.", e);
		}
	}
	
	/**
	 * 指定されたストリームを、データの読み込み元とする
	 * 新たな XMLReader を作成する。
	 *
	 * @param		input	入力ストリーム
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	public XMLReader(InputStream input) {
		mCanRead = setFile(input, "");
	}

	/**
	 * 指定されたオブジェクトを、データの読み込み元とする
	 * 新たな XMLReader を作成する。
	 *
	 * @param		input	入力ストリーム読み込み用オブジェクト
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	public XMLReader(AbstFile input) {
		if (input != null) {
			mInput   = input;
			mCanRead = true;
		}
	}

	/**
	 * 指定されたストリーム、およびファイル名をもとに
	 * データ読み込みオブジェクトを設定する。<BR>
	 *
	 * @param		input	入力ストリーム
	 * @param		file	ファイル名
	 * @return		データ読み込みオブジェクトが正常に
	 * 				設定された場合、true。
	 * 				そうでない場合は、false
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected boolean setFile(InputStream input, String file) {
		if (input == null) return false;

		mInput = new AbstFile();
		mInput.read(input, file);
		return true;
	}

	/**
	 *
	 * @param code
	 * @return
	 * @author kawae
	 * @since 2005/12/09
	 */
	public ObjectFolder read(String code) throws IOException, UnsupportedEncodingException {
		if (code == null)
			return read();

		if (!open(code)) {
			return null;
		}
		ObjectFolder of = null;
		of = parseXML(code);
		close();

		if (of == null || of.getChildren().size() < 1) {
			return null;
		}
		return of;
	}

	/**
	 * XML 形式データ読み込み、そのデータを表現するツリー構造を
	 * 持つオブジェクトを作成し、返す。
	 *
	 * @return		データの読み込み、およびその内容が正常な場合、
	 * 				ツリー構造を持つデータオブジェクト。
	 * 				そうでない場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	public ObjectFolder read() throws IOException {
		if (!open()) {
			return null;
		}
		ObjectFolder of = null;
		try {
			of = parseXML(null);
		} catch (UnsupportedEncodingException e) {
		}
		close();

		if (of == null || of.getChildren().size() < 1) {
			return null;
		}
		return of;
	}



	/**
	 *
	 * @param code
	 * @return
	 * @author kawae
	 * @throws UnsupportedEncodingException
	 * @since 2005/12/09
	 */
	protected boolean open(String code) throws UnsupportedEncodingException {
		if (code == null)
			return open();

		if (mCanRead) {
			mBr = mInput.openBufferedReader(code);
			mEOF = false;
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * データ読み込みの前処理として、読み込み先ストリームをオープンする。
	 *
	 * @return		正しくオープンできた場合、true。
	 * 				そうでない場合は、false
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected boolean open() {
		if (mCanRead) {
			mBr = mInput.openBufferedReader();
			mEOF = false;
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * データ読み込みの後処理として、読み込み先ストリームをクローズする。
	 *
	 * @return		正しくクローズできた場合、true。
	 * 				そうでない場合は、false
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected boolean close() {
		try {
			mBr.close();
		} catch (IOException e) {
			return false;
		}
		return true;
	}

	/**
	 * XML データを読み込み解析する。そして、そのデータを
	 * 表現するツリー構造を持つオブジェクトを作成し、返す。
	 *
	 * @return		データの読み込み、およびその内容が正常な場合、
	 * 				ツリー構造を持つデータオブジェクト。
	 * 				そうでない場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 * @throws UnsupportedEncodingException
	 */
	protected ObjectFolder parseXML(String code) throws IOException, UnsupportedEncodingException {
		ObjectFolder    of = new ObjectFolder("");
		ObjectFolder    topf = of;
		String          line;
		int             pos;

		while (true) {
			line = readXML();


			if (line == null) {
				break;
			}

			for (int i = 0; i < line.length() - 1; i++) {
				pos = indexOfDelimiter(line, i);
				of  = parseElement(line.substring(i, pos).trim(), of, code);
				i   = pos - 1;
			}
		}
		return topf;
	}

	/**
	 * 指定されたデータにおいて、指定されたインデックスから
	 * 検索を開始し、引用符内を除き、区切り文字が最初に出現
	 * したインデックスを返す。
	 *
	 * @param		line	検索を行うデータ
	 * @param		st		検索を開始するインデックス
	 * @return		区切り文字が、指定されたインデックスと同じか
	 * 				これより大きいインデックス位置にある場合は、
	 * 				最初に出現した位置のインデックス。
	 * 				区切り文字がない場合は、line の文字列の長さ
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected int indexOfDelimiter(String line, int st) {
		boolean equal = false;
		short   quote = 0;
		int i;
		char c;
		for (i = st + 1; i < line.length(); i++) {
			c = line.charAt(i);
			if (quote == 0 && c == '=') {
				equal = true;
			}
			else if (equal && c == '\"') {
				quote++;
			}
			else if (c == ' ') {
				if (quote != 1) {
					break;
				}
			}
		}
		return i;
	}

	/**
	 * 指定された要素を解析し、指定されたツリーオブジェクトへ
	 * その要素が定義するデータを追加する。
	 *
	 * @param		elem	解析する要素
	 * @param		of		ツリーオブジェクト
	 * @return		データ追加後の新しいツリーオブジェクト
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 * @throws UnsupportedEncodingException
	 */
	protected ObjectFolder parseElement(String elem, ObjectFolder of, String code) throws UnsupportedEncodingException {
		String       tag = null;
		String       val = null;
		int          q1, q2;

		if (elem != null && elem.length() > 0) {
			int deli = elem.indexOf('=');
			if (deli >= 0) {
				tag = elem.substring(0, deli);
				val = elem.substring(deli + 1, elem.length());
				q1  = val.indexOf('\"');
				q2  = val.lastIndexOf('\"');
				if (q1 >= 0 && q2 >= 0 && q2 > q1) {
					val = val.substring(q1 + 1, q2);
				}
				if (val != null)
					val = getValue(val, code);
			}
			else {
				tag = elem;
			}
		}

		if (tag != null && tag.length() > 0) {
			StringTokenizer st  = new StringTokenizer(tag, "<> ");
			tag = (String)st.nextElement();
			if (val == null) {
				if (tag.charAt(0) == '/' || tag.equalsIgnoreCase("/")) {
					of = of.getParent();
				}
				else {
					ObjectFolder w = new ObjectFolder(tag);
					of.addChild(w, true);
					of = w;
				}
			}
			else {
				of.addChild(new ObjectNode(tag, val), true);
			}
		}

		return of;
	}

	/**
	 * Base64タグが存在するか確認し、存在すればタグを外した文字列を得る
	 * @param val
	 * @return
	 * @author kawae
	 * @since 2005/08/11
	 */
	protected String getValue(String val, String code) throws UnsupportedEncodingException {
		int indexStart = val.indexOf(XMLManager.BASE64_TAG_BRGIN);
		int indexEnd = val.indexOf(XMLManager.BASE64_TAG_END);
		if (
				indexStart == 0 &&
				indexEnd > 0
				) {
			String str = val.substring(XMLManager.BASE64_TAG_BRGIN.length(), indexEnd);
			byte[] bytes = Base64Decoder.decode(str);
			if (code == null) {
				val = new String(bytes);
			} else {
				val = new String(bytes, code);
			}
		}
		return val;
	}

	/**
	 * 1 件分のXML データ（"<" から ">" まで）を読み込む。<BR>
	 * #parseXML() で解析しやすいようデータの整形を主に行う。
	 *
	 * @return		読み込まれた XML データ。
	 * 				ストリームの終わりに達した場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected String readXML() throws IOException {
		if (mEOF) {
			return null;
		}

//		StringBuffer sb = new StringBuffer(mRemain);
		StringBuffer sb = new StringBuffer(128);
		sb.append(mRemain);

		mRemain = "";
		String line = readLineWithoutComment();
		if (line == null) {
			return null;
		}

		int pos, len;
		while (line != null) {
			pos = line.lastIndexOf("/>");
			len = line.length();
			if (pos < 0) {
				pos = line.lastIndexOf(">");
				if (pos < 0) {
					//TODO:川会 クォーテーションを考慮し、その内部であれば改行する
					sb.append(line + "\n");
					line = readLineWithoutComment();
					continue;
				}
				else {
					sb.append(line.substring(0, pos).trim());
					sb.append(">");
					if (pos + 1 != len) {
						mRemain = line.substring(pos + 1, len);
					}
					break;
				}
			}
			else {
				if (pos + 2 != len) {
					mRemain = line.substring(pos + 2, len);
				}
				sb.append(line.substring(0, pos));
				sb.append(" />");
				break;
			}
		}


		if (line == null) {
			mEOF = true;
		}
		if (sb.length() > 0) {
			return sb.toString();
		}
		else {
			return null;
		}
	}

	/**
	 * XML 形式データでのコメントを除く、1 行分のデータを読み込み。<BR>
	 * 次のデータはコメントとして取り扱う。<BR>
	 * 	(1)<!-- xxx --><BR>
	 * 	(2)<? xxxxxx ?><BR>
	 *
	 * @return		読み込まれたデータ。
	 * 				ストリームの終わりに達した場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected String readLineWithoutComment() throws IOException {
		String line = readLine();
		boolean trimed;
		while (line != null) {
			if (line.length() == 0) {
				//川会変更
				line = "";
//				line = readLine();
			}
//			if (line.startsWith("<?")) {
//				line = trimComment(line, "<?", "?>");
//			}
//			else if (line.startsWith("<!--")) {
//				line = trimComment(line, "<!--", "-->");
//			}
//			else {
//				break;
//			}

			trimed = false;
			for (String[] element : COMMENT) {
				if (line.startsWith(element[0])) {
					line = trimComment(line, element[0], element[1]);
					trimed = true;
				}
			}
			if (!trimed) {
				break;
			}
		}
		return line;
	}

	/**
	 * 指定された行データを解析し、コメントを除くデータを返す。
	 *
	 * @param		line	解析を開始する行（データ）
	 * @param		start	コメントの開始文字
	 * @param		end		コメントの終了文字
	 * @return		コメントを除くデータ。
	 * 				ストリームの終わりに達した場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected String trimComment(String line,
								  final String start,
								  final String end) throws IOException {
		if (line == null) {
			return null;
		}

//		line = line.trim();
		if (!line.startsWith(start)) {
			return line;
		}

		// 終了文字が発見できるまで、繰り返しデータを読み込む。
		int pos;
		final int sz = end.length();
		while (true) {
			pos = line.lastIndexOf(end);
			if (pos >= 0) {
				break;
			}

			line = readLine();
			if (line == null) {
				return null;
			}
		}

		// コメントが行の途中で終了し、その後データが続くことを考慮する。
		final int len = line.length();
		if (pos + sz != len) {
			line = line.substring(pos + sz, len).trim();
		}
		else {
			line = readLine();
		}

		return line;
	}

	/**
	 * 1 行分のデータを読み込む。
	 *
	 * @return		前後の空白を除いた読み込んだデータ。
	 * 				ストリームの終わりに達した場合は、null
	 * @since		2004/05/06
	 * @author		fujita
	 * @version	2004/05/06
	 */
	protected String readLine() throws IOException {
		String line = mBr.readLine();
		if (line != null) {
			return line.trim();
		} else {
			return null;
		}
	}
}