/*
 * Copyright (c) 2009 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



/**
 * オブジェクトに関するユーティリティー
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class ObjectUtil {

	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	private static final Log mLog = LogFactory.getLog(ObjectUtil.class);



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ
	 */
	protected ObjectUtil() {
	}



	//------------------------------------------------------------------
	//- methods
	//------------------------------------------------------------------

	/**
	 * 言語型(primitive/wrapper/enum)であるかチェックする。
	 *
	 * @param clazz 対象
	 * @return 言語型である場合true。
	 */
	public static boolean isLangType(Object obj) {
		return isLangType(obj.getClass());
	}

	/**
	 * 言語型(primitive/wrapper/enum)であるかチェックする。
	 *
	 * @param clazz 対象
	 * @return 言語型である場合true。
	 */
	public static boolean isLangType(Class<?> clazz) {
		if (clazz == null) return false;

		boolean result = false;
		if (clazz == boolean.class) {
			result = true;
		} else if (clazz == byte.class) {
			result = true;
		} else if (clazz == int.class) {
			result = true;
		} else if (clazz == long.class) {
			result = true;
		} else if (clazz == float.class) {
			result = true;
		} else if (clazz == double.class) {
			result = true;
		} else if (Boolean.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (Byte.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (Integer.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (Long.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (Float.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (Double.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (String.class.isAssignableFrom(clazz)) {
			result = true;
		} else if (clazz.isEnum()) {
			result = true;
		}
		return result;
	}

	/**
	 * オブジェクトのShallowCopyを行う。<br/>
	 *
	 * @param src コピー元オブジェクト
	 * @return 内容がシャローコピーされた新たなオブジェクト
	 */
	@SuppressWarnings("unchecked")
	public static <T> T shallowCopy(T src) {
		T dst = (T)ReflectUtil.newInstance(src.getClass(), new Object[]{});
		shallowCopy(src, dst);
		return dst;
	}

	/**
	 * オブジェクトのShallowCopyを行う。<br/>
	 *
	 * @param src コピー元オブジェクト
	 * @param dst コピー先オブジェクト
	 */
	public static <T> void shallowCopy(T src, T dst) {
		for (Field srcField: ReflectUtil.getFields(src)) {
			try {
				Field dstField = ReflectUtil.getField(dst, srcField.getName());
				ReflectUtil.setFieldValue(dst, dstField, srcField.get(src));
			} catch (Exception e) {
			}
		}
	}

	/**
	 * コレクションのShallowCopyを行う。<br/>
	 *
	 * @param srcs コピー元オブジェクトコレクション
	 * @return 内容がシャローコピーされた新たなオブジェクトのコレクション
	 */
	@SuppressWarnings("unchecked")
	public static <T> Collection<T> shallowCopy(Collection<T> srcs) {
		Collection<T> dsts = ReflectUtil.newInstance(srcs.getClass(), new Object[]{});
		return shallowCopy(srcs, dsts);
	}

	/**
	 * コレクションのShallowCopyを行う。<br/>
	 *
	 * @param srcs コピー元オブジェクトコレクション
	 * @param dsts コピー先オブジェクトコレクション
	 */
	@SuppressWarnings("unchecked")
	public static <T> Collection<T> shallowCopy(Collection<T> srcs, Collection<T> dsts) {
		for (T src: srcs) {
			T dst = (T)ReflectUtil.newInstance(src.getClass(), new Object[]{});
			shallowCopy(srcs, dst);
			dsts.add(dst);
		}
		return dsts;
	}

	/**
	 * 配列のShallowCopyを行う。<br/>
	 *
	 * @param srcs コピー元オブジェクト配列
	 * @return 内容がシャローコピーされた新たなオブジェクトの配列
	 */
	@SuppressWarnings("unchecked")
	public static <T> void shallowCopy(T[] srcs, T[] dsts) {
		int dstIdx = 0;
		for (T src: srcs) {
			T dst = (T)ReflectUtil.newInstance(src.getClass());
			shallowCopy(srcs, dst);
			dsts[dstIdx++] = dst;
		}
	}

	/**
	 * ディープコピーを行う。<br/>
	 * ただし、オブジェクトがSerializableをimplementしている必要がある。
	 *
	 * @param src コピー元オブジェクト
	 * @return 内容がディープコピーされた新たなオブジェクト
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Serializable> T deepCopy(T src) {
		if (src == null) return null;

		T dst = null;
		try {
			ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
			ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
			objOut.writeObject(src);
			ByteArrayInputStream byteIn = new ByteArrayInputStream(byteOut.toByteArray());
			ObjectInputStream objIn = new ObjectInputStream(byteIn);
			dst = (T)objIn.readObject();
		} catch (Exception e) {
			mLog.debug("", e);
		}
		return dst;
	}

	/**
	 * Objectの文字列表現を生成する。<br/>
	 * Object配下のフィールドを文字列表現に加える。<br/>
	 *
	 * @param obj Object
	 * @return Objectの文字列表現
	 */
	public static String toString(Object obj) {
		if (obj == null) return "null";
		ToStringBuilder buf = new ToStringBuilder(obj);
		for (Field field: ReflectUtil.getFields(obj)) {
			buf.append(ReflectUtil.getFieldValue(obj, field));
		}
		return buf.toString();
	}

	/**
	 * Objectの文字列表現を生成する。<br/>
	 * Object配下のフィールドを文字列表現に加える。<br/>
	 *
	 * @param obj Object
	 * @return Objectの文字列表現
	 */
	public static String[] toStringArray(Object obj) {
		List<String> strList = new ArrayList<String>();
		Collection<Field> fields = ReflectUtil.getFields(obj);
		for (Field field: fields) {
			Object value = ReflectUtil.getFieldValue(obj, field);
			if (value == null) continue;

			strList.add(value.toString());
		}
		return strList.toArray(new String[strList.size()]);
	}

	/**
	 * オブジェクト比較を行う。<br/>
	 * オブジェクトのequalsでの比較を行う前にnullによる比較を行う。<br/>
	 *
	 * @param obj1 比較元オブジェクト
	 * @param obj2 比較対象オブジェクト
	 * @return ob1とobj2が同様と見なされればtrue。
	 */
	public static boolean equals(Object obj1, Object obj2) {
		if (obj1 == null && obj2 == null) return true;
		if (obj1 != null && obj2 == null) return false;
		if (obj1 == null && obj2 != null) return false;
		return obj1.equals(obj2);
	}

	/**
	 * オブジェクトのフィールド比較を行う。<br/>
	 * 現時点では浅い参照のみで比較を行う。<br/>
	 * ※obj1が保持しているフィールドの比較のみを行う。<br/>
	 *
	 * @param obj1 比較元オブジェクト
	 * @param obj2 比較対象オブジェクト
	 * @return obj2のフィールドがobj1のフィールドと同じであるならtrueを返す。
	 */
	public static boolean equalFields(Object obj1, Object obj2) {
		for (Field obj1Field: ReflectUtil.getFields(obj1)) {
			String obj1Name = obj1Field.getName();
			Object obj1Value = ReflectUtil.getFieldValue(obj1, obj1Name);
			Object obj2Value = ReflectUtil.getFieldValue(obj2, obj1Name);

			if (obj1Value == null && obj2Value == null) continue;
			if (obj1Value == null && obj2Value != null) return false;
			if (obj1Value != null && obj2Value == null) return false;
			if (!obj1Value.equals(obj2Value)) return false;
		}
		return true;
	}



} // end-class
