package jp.kitec.lib.util;

/**
 * 幾何計算クラス
 */
public class MathUtil {

	public static final double EPS = 1.0e-3f;
	public static final double D2R = Math.PI / 180;
	public static final double R2D = 180 / Math.PI;

	/**
	 * 小数点s桁以降を切り捨てる。
	 *
	 * @param f
	 * @param s
	 * @return
	 */
	public static double kirisute(double f, int s) {
		double kei = Math.pow(10, s - 1);
		return (Math.floor(f * kei) / kei);
	}

	/**
	 * 外積(のZ値)を求める
	 */
	public static double crossProduct(double p0x, double p0y, double p1x, double p1y, double p2x, double p2y) {
		double dx1 = p1x - p0x, dy1 = p1y - p0y;
		double dx2 = p2x - p0x, dy2 = p2y - p0y;
		return dx1 * dy2 - dx2 * dy1;
	}

	/**
	 * ２つの角度の差(360度内で）
	 * getAnglGapRad()とは挙動が違うので注意
	 */
	public static double getAnglGapDeg(double a, double b) {
		b -= a;
		return getAngl360(b);
	}

	/**
	 * ２つの角度の差(2PI度内で）
	 * getAnglGapDeg()とは挙動が違うので注意
	 */
	public static double getAnglGapRad(double a, double b) {
		a = getAngl2PI(a);
		b = getAngl2PI(b);
		return Math.abs(a - b);
	}

	/**
	 * 360度内で開始角と終了角の間にあるか(ラジアン)
	 */
	public static boolean isInnerAngl2PI(double s, double e, double a) {
		return isInnerAngl360(getAngl360(s),  getAngl360(e),  getAngl360(a));
	}

	/**
	 * 360度内で開始角と終了角の間にあるか(度)
	 */
	public static boolean isInnerAngl360(double s, double e, double a) {
		double ss = s;
		double ee = e;
		double aa = a;

		while(ss > ee)
			ee += 360;

		ss -= s;
		ee -= s;
		aa -= s;
		while(aa < 0)
			aa += 360;
		if (ss < aa && aa < ee)
			return true;
		return false;
	}

	/**
	 * 円弧の始点・終点と点の距離を調べる
	 * @param x
	 * @param y
	 * @param r
	 * @param s
	 * @param e
	 * @param x
	 * @param y
	 * @param dir
	 * @return
	 */
	public static double getLengthToArcStartEnd(double cx, double cy, double r, double s, double x, double y) {
		double xx = cx + r * Math.cos(s * D2R);
		double yy = cy + r * Math.sin(s * D2R);
		return getLength2D(x, y, xx, yy);
	}

	public static double rachet(double angl, int rat) {
		angl += Math.PI * 2 / rat;
		while(angl >= (Math.PI * 2))
		angl -= (Math.PI * 2);

		for (int i = 0; i < rat + 1; i++) {
			if (Math.abs(angl - i * Math.PI * 2 / rat) < (Math.PI / rat)) {
				angl = (Math.PI * 2 / rat * i);
				break;
			}
		}
		return angl;
	}

	/**
	 * 2直線の角度の取得（時計回り方向）
	 * @param ax1
	 * @param ay1
	 * @param ax2
	 * @param ay2
	 * @param bx1
	 * @param by1
	 * @param bx2
	 * @param by2
	 * @return
	 */
	public static double getAngle2Lines(double x1, double y1, double x2, double y2, double x3, double y3) {
		double angl1 = getAngle2D(x2, y2, x1, y1);
		double angl2 = getAngle2D(x2, y2, x3, y3);
		angl2 -= angl1;
		return getAngl2PI(angl2);
	}

	/**
	 * ２Dでの角度の計算
	 * @param x1 点１のX座標
	 * @param y1 点１のY座標
	 * @param x2 点２のX座標
	 * @param y2 点２のY座標
	 * @return 角度（ラジアン）
	 */
	public static double getAngle2D(double x1, double y1, double x2, double y2) {
		double dx, dy, t, a;
		int area;

		dx = x2 - x1;
		dy = y2 - y1;

		if (dx >= 0.0) {
			if (dy >= 0.0)
				area = 0;
			else {
				area = 3;
				t = dx;
				dx = -dy;
				dy = t;
			}
		} else {
			if (dy >= 0.0) {
				area = 1;
				t = dx;
				dx = dy;
				dy = -t;
			} else {
				area = 2;
				dx = -dx;
				dy = -dy;
			}
		}

		if (dy > dx)
			a = Math.PI / 2.0f - Math.atan(dx / dy);
		else
			a = Math.atan(dy / dx);
		return a + area * (Math.PI / 2.0);
	}



	/**
	 * 直線をローカル座標X軸とした場合の点のXの値を取得
	 * @param x		検証する点X
	 * @param y		検証する点Y
	 * @param x1	原点側X
	 * @param y1	原点側Y
	 * @param x2	X軸側X
	 * @param y2	X軸側Y
	 * @return		ローカル座標系でのX座標
	 */
	public static double getX(double x, double y, double x1, double y1, double x2, double y2) {
		double cc, ss;
		double rot = getAngle2D(x1, y1, x2, y2);
		cc = Math.cos(rot);
		ss = Math.sin(rot);

		x -= x1;
		y -= y1;
		return norm(x * cc + y * ss);
	}

	/**
	 * 線分p1-p2が線分p3-p4に含まれるか
	 */
	public static boolean inIncludeLine(double p1x, double p1y, double p2x, double p2y, double p3x, double p3y, double p4x, double p4y) {
		return (isOnLimitLinePoint2D(p1x, p1y, p3x, p3y, p4x, p4y) && isOnLimitLinePoint2D(p2x, p2y, p3x, p3y, p4x, p4y));
	}

	/**
	 * 直線をローカル座標X軸とした場合の点のYの値を取得
	 * @param x
	 * @param y
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 * @return
	 */
	public static double getY(double x, double y, double x1, double y1, double x2, double y2) {
		double cc, ss;
		double rot = getAngle2D(x1, y1, x2, y2);
		cc = Math.cos(rot);
		ss = Math.sin(rot);

		x -= x1;
		y -= y1;
		return norm(-x * ss + y * cc);
	}

	/**
	 * 象限を調べる。
	 * @param px 検査する点のX座標
	 * @param py 検査する点のY座標
	 * @param x1 点１のX座標
	 * @param y1 点１のY座標
	 * @param x2 点２のX座標
	 * @param y2 点２のY座標
	 */
	public static int getQuadrant(double px, double py, double x1, double y1, double x2, double y2) {
		double baseangl = getAngle2D(x1, y1, x2, y2);
		double angl = getAngle2D(x1, y1, px, py);
		double res = angl - baseangl < 0.0f ? angl - baseangl + (Math.PI * 2) : angl - baseangl;

		if (res >= 0.0f && res < (Math.PI / 2))
			return 1;
		else if (res >= (Math.PI / 2) && res < Math.PI)
			return 2;
		else if (res >= (Math.PI) && res < (Math.PI / 2 * 3))
			return 3;
		else
			return 4;
	}

	/**
	 * 角度（ラジアン）を２＊PIの範疇に変更する
	 * @param d
	 * @return
	 */
	public static double getAngl2PI(double d) {
		if (Math.abs(d) < EPS)
			d = 0;
		while (d < 0)
			d += (Math.PI * 2);
		while (d > (Math.PI * 2))
			d -= (Math.PI * 2);
		return d;
	}

	/**
	 * 角度（度）360度をの範疇に変更する
	 * @param d
	 * @return
	 */
	public static double getAngl360(double d) {
		while (d < 0)
			d += 360;
		while (d > 360)
			d -= 360;
		return d;
	}

	/**
	 * ２Dでの点の回転
	 * @param xc 回転中心のX座標
	 * @param yc 回転中心のX座標
	 * @param rot 回転角度
	 * @param xsrc 対象の点のX座標
	 * @param ysrc 対象の点のY座標
	 * @param res 結果座標
	 */
	public static void rotation2D(double xc, double yc, double rot, double xsrc, double ysrc, double res[]) {
		double cc, ss;

		cc = Math.cos(-rot);
		ss = Math.sin(-rot);

		xsrc -= xc;
		ysrc -= yc;
		if (res != null) {
			res[0] = norm(xsrc * cc + ysrc * ss + xc);
			res[1] = norm(-xsrc * ss + ysrc * cc + yc);
		}
	}

	/**
	 * 2次元の2点の間の距離を調べる
	 * @param x1 点１のX座標
	 * @param y1 点１のY座標
	 * @param x2 点２のX座標
	 * @param y2 点２のY座標
	 */
	public static double getLength2D(double x1, double y1, double x2, double y2) {
		return norm(Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)));
	}

	/**
	 * 2点が同じ位置にあるか検査
	 * @param p1x 点1のX座標
	 * @param p1y 点1のY座標
	 * @param p2x 点2のX座標
	 * @param p2y 点2のY座標
	 */
	public static boolean isSame2D(double p1x, double p1y, double p2x, double p2y) {
		if (Math.abs(p2x - p1x) < EPS && Math.abs(p2y - p1y) < EPS)
			return true;
		return false;
	}

	/**
	 * 2直線が同じか調べる
	 * @param pa1x 点１
	 *
	 */
	public static boolean isSameLine(double pa1x, double pa1y, double pa2x, double pa2y, double pb1x, double pb1y, double pb2x, double pb2y) {
		if ((isSame2D(pa1x, pa1y, pb1x, pb1y) && isSame2D(pa2x, pa2y, pb2x, pb2y)) || (isSame2D(pa1x, pa1y, pb2x, pb2y) && isSame2D(pa2x, pa2y, pb1x, pb1y)))
			return true;
		return false;
	}
	/**
	 * aはbに含まれるか
	 */
	public static boolean isIncludeLine(double pa1x, double pa1y, double pa2x, double pa2y, double pb1x, double pb1y, double pb2x, double pb2y) {
		if (isOnLimitLinePoint2D(pa1x, pa1y, pb1x, pb1y, pb2x, pb2y) && isOnLimitLinePoint2D(pa2x, pa2y, pb1x, pb1y, pb2x, pb2y))
			return true;
		return false;
	}

	/**
	 * 2線分が平行に関係を持つか
	 * (平行かつ重なるか)
	 */
	public static boolean isRelatePara2Line(double pa1x, double pa1y, double pa2x, double pa2y, double pb1x, double pb1y, double pb2x, double pb2y) {
		if (!isOnLine(pa1x, pa1y, pa2x, pa2y, pb1x, pb1y, pb2x, pb2y))
			return false;
		if (isOnLimitLinePoint2D(pa1x, pa1y, pb1x, pb1y, pb2x, pb2y) || isOnLimitLinePoint2D(pa2x, pa2y, pb1x, pb1y, pb2x, pb2y) ||
		    isOnLimitLinePoint2D(pb1x, pb1y, pa1x, pa1y, pa2x, pa2y) || isOnLimitLinePoint2D(pb2x, pb2y, pa1x, pa1y, pa2x, pa2y))
		    return true;
		return false;
	}



	public static  boolean isVerticalLimitLineToPoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y, double[] tmp) {
		double xp1, yp1, x21, y21;
		double denom;
		xp1 = p1x - px;
		yp1 = p1y - py;
		x21 = p2x - p1x;
		y21 = p2y - p1y;
		denom = x21 * x21 + y21 * y21;

		double t = - (xp1 * x21 + yp1 * y21) / denom;

		if (t >= 0.0f && t <= 1.0f) {
			if (tmp != null) {
				double x, y;
				x = xp1 + t * x21;
				y = yp1 + t * y21;
				tmp[0] = (px + x);
				tmp[1] = (py + y);
			}
			return true;
		}
		return false;
	}

	/**
	 * 点から直線(無限)に垂直に降ろした交点を調べる
	 * @param	px	検査する点のX座標
	 * @param	py	検査する点のY座標
	 * @param	p1x	線の点１のX座標
	 * @param	p1y	線の点１のY座標
	 * @param	p2x	線の点２のX座標
	 * @param	p2y	線の点２のY座標
	 */
	public static boolean isCrossPointVertLimitlessLineToPoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y, double[] res) {
		//線１の方程式
		double a1 = getLineEquation2Da(p1x, p1y, p2x, p2y);
		double b1 = getLineEquation2Db(p1x, p1y, p2x, p2y);
		double c1 = getLineEquation2Dc(p1x, p1y, p2x, p2y);
		//線２の方程式
		double a2 = b1;
		double b2 = -a1;
		double c2 = a1 * py - b1 * px;

		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return false;
		if (res != null) {
			res[0] = norm((b1 * c2 - b2 * c1) / det);
			res[1] = norm((a2 * c1 - a1 * c2) / det);
		}
		return true;
	}

	/**
	 * 点と線分の距離を調べる
	 * 点から直線へ下ろした垂線が線分の外の場合は-1を返す。
	 * 
	 * @param px 検査する点のX座標
	 * @param py 検査する点のY座標
	 * @param p1x 線分の点1のx座標
	 * @param p1y 線分の点1のY座標
	 * @param p2x 線分の点2のx座標
	 * @param p2y 線分の点2のY座標
	 */
	public static double getLengthLimitLineToPoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y) {
		double xp1, yp1, x21, y21;
		double denom;
		xp1 = p1x - px;
		yp1 = p1y - py;
		x21 = p2x - p1x;
		y21 = p2y - p1y;
		denom = x21 * x21 + y21 * y21;

		double t = - (xp1 * x21 + yp1 * y21) / denom;

		if (t >= 0.0f && t <= 1.0f) {
			double x, y;
			x = xp1 + t * x21;
			y = yp1 + t * y21;
			return norm(Math.sqrt(x * x + y * y));
		}
		return -1.0;
	}

	/**
	 * 点と線分の距離を調べる
	 * @param px 検査する点のX座標
	 * @param py 検査する点のY座標
	 * @param p1x 線分の点1のx座標
	 * @param p1y 線分の点1のY座標
	 * @param p2x 線分の点2のx座標
	 * @param p2y 線分の点2のY座標
	 */
	public static double getLengthLineSegmentToPoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y) {
		double xp1, yp1, x21, y21;
		double denom;
		xp1 = p1x - px;
		yp1 = p1y - py;
		x21 = p2x - p1x;
		y21 = p2y - p1y;
		denom = x21 * x21 + y21 * y21;

		double t = - (xp1 * x21 + yp1 * y21) / denom;

		if (t >= 0.0f && t <= 1.0f) {
			double x, y;
			x = xp1 + t * x21;
			y = yp1 + t * y21;
			return norm(Math.sqrt(x * x + y * y));
		} else {
			double a = getLength2D(px, py, p1x, p1y);
			double b = getLength2D(px, py, p2x, p2y);
			return norm(a < b ? a : b);
		}
	}

	/**
	 * 点と直線(無限)の距離を調べる
	 *
	 * @param	px	検査する点のX座標
	 * @param	py	検査する点のY座標
	 * @param	p1x	線分の点1のx座標
	 * @param	p1y	線分の点1のY座標
	 * @param	p2x	線分の点2のx座標
	 * @param	p2y	線分の点2のY座標
	 *
	 * @return	点と線分の距離。計算できなければ-1を返す。
	 */
	public static double getLengthLimitlessLineToPoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y) {
		if (isSame2D(p1x, p1y, p2x, p2y))
			return -1f;

		//直線の方程式を算出
		double a = getLineEquation2Da(p1x, p1y, p2x, p2y);
		double b = getLineEquation2Db(p1x, p1y, p2x, p2y);
		double c = getLineEquation2Dc(p1x, p1y, p2x, p2y);

		//距離を調べる
		double absq = a * a + b * b;
		if (absq < EPS)
			return -1f;
		double sr = a * px + b * py + c;
		return norm(Math.sqrt(sr * sr / absq));
	}

	/**
	 * 点と直線(無限)の位置を調べる
	 *
	 * @param	px	検査する点のX座標
	 * @param	py	検査する点のY座標
	 * @param	p1x	線分の点1のx座標
	 * @param	p1y	線分の点1のY座標
	 * @param	p2x	線分の点2のx座標
	 * @param	p2y	線分の点2のY座標
	 *
	 * @return	上にあればtrue.下ならfalse。
	 */
	public static boolean isPointPickUporDown(double px, double py, double p1x, double p1y, double p2x, double p2y) {
		//直線の方程式を算出
		double a = getLineEquation2Da(p1x, p1y, p2x, p2y);
		double b = getLineEquation2Db(p1x, p1y, p2x, p2y);
		double c = getLineEquation2Dc(p1x, p1y, p2x, p2y);

		//距離を調べる
		return a * px + b * py + c >= 0.0f ? true : false;
	}

	/**
	 * 点１から点２の方向にofsx,
	 * そこから垂直にofsy（左が正）の位置を算出
	 *
	 * @param	x1		点１のX座標
	 * @param	y1		点１のY座標
	 * @param	x2		点２のX座標
	 * @param	y2		点２のY座標
	 * @param	ofsx	xオフセット(線方向)
	 * @param	ofsy	yオフセット(法線方向)
	 * @param	res		結果座標
	 */
	public static void getOffsetPoint2D(double x1, double y1, double x2, double y2, double ofsx, double ofsy, double[] res) {
		double angl = getAngle2D(x1, y1, x2, y2);
		if (res != null) {
			res[0] = norm(x1 + (ofsx * Math.cos(angl)) + (ofsy * Math.cos(angl + Math.PI / 2.0)));
			res[1] = norm(y1 + (ofsx * Math.sin(angl)) + (ofsy * Math.sin(angl + Math.PI / 2.0)));
		}
	}

	public static void getOffsetLength2D(double x1, double y1, double x2, double y2, double ofsx, double ofsy, double[] res) {
		double angl = getAngle2D(x1, y1, x2, y2);
		if (res != null) {
			res[0] = ((ofsx * Math.cos(angl)) + (ofsy * Math.cos(angl + Math.PI / 2.0)));
			res[1] = ((ofsx * Math.sin(angl)) + (ofsy * Math.sin(angl + Math.PI / 2.0)));
		}
	}

	/**
	 * 3点を通る円の半径と中心点の取得
	 */
	public static double get3PointCircle(double x1, double y1, double x2, double y2, double x3, double y3, double[] res) {
		double xc11 = (x2 + x1) / 2;
		double yc11 = (y2 + y1) / 2;
		double xc21 = (x3 + x1) / 2;
		double yc21 = (y3 + y1) / 2;

		double xc12 = xc11 - (y2 - y1);
		double yc12 = yc11 + (x2 - x1);
		double xc22 = xc21 - (y3 - y1);
		double yc22 = yc21 + (x3 - x1);

		if (isCrossInfinity(xc11, yc11, xc12, yc12, xc21, yc21, xc22, yc22, res)) {
			return getLength2D(res[0], res[1], x1, y1);
		}
		return -1;
	}

	/**
	 * 3点を通る円の半径と中心点の取得
	 */
	public static double get2LineArc(double x1, double y1, double x2, double y2, double x3, double y3, double[] res) {
		double xc1 = x1 - (y2 - y1);
		double yc1 = y1 + (x2 - x1);
		double xc2 = x3 - (y3 - y2);
		double yc2 = y3 + (x3 - x2);

		if (isCrossInfinity(x1, y1, xc1, yc1, x3, y3, xc2, yc2, res))
			return getLength2D(res[0], res[1], x1, y1);
		return -1;

	}

	/**
	 * 2線分が交点を持つか調べる
	 *
	 * @param	ap1x	線１の点１のX座標
	 * @param	ap1y	線１の点１のY座標
	 * @param	bp1x	線２の点１のX座標
	 * @param	bp1y	線２の点１のY座標
	 * @param	res		結果座標
	 */
	public static boolean isCrossSegment(double ap1x, double ap1y, double ap2x, double ap2y, double bp1x, double bp1y, double bp2x, double bp2y, double[] res) {
		double x21 = ap2x - ap1x;
		double y21 = ap2y - ap1y;
		double x43 = bp2x - bp1x;
		double y43 = bp2y - bp1y;
		double x31 = bp1x - ap1x;
		double y31 = bp1y - ap1y;

		double det = x43 * y21 - y43 * x21;
		if (Math.abs(det) < EPS)
			return false;

		double s = (x43 * y31 - y43 * x31) / det;
		double t = (x21 * y31 - y21 * x31) / det;
		if (s < -EPS || s > (1.0 + EPS) || t < -EPS || t > (1.0 + EPS))
			return false;
//		if (s < 0.0 || s > 1.0 || t < 0.0 || t > 1.0)
//			return false;
		if (res != null) {
			res[0] = norm(ap1x + x21 * s);
			res[1] = norm(ap1y + y21 * s);
		}
		return true;
	}

	/**
	 * 2直線(無限)が交点を持つか調べる
	 *
	 * @param	ap1x	線１の点１のX座標
	 * @param	ap1y	線１の点１のY座標
	 * @param	ap2x	線１の点２のX座標
	 * @param	ap2y	線１の点２のY座標
	 * @param	bp1x	線２の点１のX座標
	 * @param	bp1y	線２の点１のY座標
	 * @param	bp2x	線２の点２のX座標
	 * @param	bp2y	線２の点２のY座標
	 * @param	res		結果座標
	 */
	public static boolean isCrossInfinity(double ap1x, double ap1y, double ap2x, double ap2y, double bp1x, double bp1y, double bp2x, double bp2y, double[] res) {
		//線１の方程式
		double a1 = getLineEquation2Da(ap1x, ap1y, ap2x, ap2y);
		double b1 = getLineEquation2Db(ap1x, ap1y, ap2x, ap2y);
		double c1 = getLineEquation2Dc(ap1x, ap1y, ap2x, ap2y);
		//線２の方程式
		double a2 = getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
		double b2 = getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
		double c2 = getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);

		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return false;
		if (res != null) {
			res[0] = norm((b1 * c2 - b2 * c1) / det);
			res[1] = norm((a2 * c1 - a1 * c2) / det);
		}
		return true;
	}


	public static boolean isCrossInfinity(double a1, double b1, double c1, double a2, double b2, double c2, double[] res) {
		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return false;
		if (res != null) {
			res[0] = norm((b1 * c2 - b2 * c1) / det);
			res[1] = norm((a2 * c1 - a1 * c2) / det);
		}
		return true;
	}

	public static boolean isCrossInfinityPointAndAngle(double x1, double y1, double ang1, double xa, double ya, double anga, double[] tmpf) {
		double x2 = x1 + 100 * Math.cos(ang1);
		double y2 = y1 + 100 * Math.sin(ang1);
		double xb = xa + 100 * Math.cos(anga);
		double yb = ya + 100 * Math.sin(anga);
		return isCrossInfinity(x1, y1, x2, y2, xa, ya, xb, yb, tmpf);
	}

	public static boolean isCrossXandLimitLine(double x, double bp1x, double bp1y, double bp2x, double bp2y, double[] res) {
		//線１の方程式
		double a1 = getLineEquation2Da(x, 0, x, 100);
		double b1 = getLineEquation2Db(x, 0, x, 100);
		double c1 = getLineEquation2Dc(x, 0, x, 100);
		//線２の方程式
		double a2 = getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
		double b2 = getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
		double c2 = getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);

		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return false;
		if (res != null) {
			double xx  = norm((b1 * c2 - b2 * c1) / det);
			double yy  = norm((a2 * c1 - a1 * c2) / det);
			if (isOnLimitLinePoint2D(xx, yy, bp1x, bp1y, bp2x, bp2y)) {
				res[0] = xx;
				res[1] = yy;
				return true;
			}
		}
		return false;
	}

	public static boolean isCrossYandLimitLine(double y, double bp1x, double bp1y, double bp2x, double bp2y, double[] res) {
		//線１の方程式
		double a1 = getLineEquation2Da(0, y, 100, y);
		double b1 = getLineEquation2Db(0, y, 100, y);
		double c1 = getLineEquation2Dc(0, y, 100, y);
		//線２の方程式
		double a2 = getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
		double b2 = getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);
		double c2 = getLineEquation2Dc(bp1x, bp1y, bp2x, bp2y);

		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return false;
		if (res != null) {
			double xx  = norm((b1 * c2 - b2 * c1) / det);
			double yy  = norm((a2 * c1 - a1 * c2) / det);
			if (isOnLimitLinePoint2D(xx, yy, bp1x, bp1y, bp2x, bp2y)) {
				res[0] = xx;
				res[1] = yy;
				return true;
			}
		}
		return false;
	}

	/**
	 * 2直線が同一線上に存在するか
	 *
	 * @param	ap1x	線１の点１のX座標
	 * @param	ap1y	線１の点１のY座標
	 * @param	ap2x	線１の点２のX座標
	 * @param	ap2y	線１の点２のY座標
	 * @param	bp1x	線２の点１のX座標
	 * @param	bp1y	線２の点１のY座標
	 * @param	bp2x	線２の点２のX座標
	 * @param	bp2y	線２の点２のY座標
	 */
	public static boolean isOnLine(double ap1x, double ap1y, double ap2x, double ap2y, double bp1x, double bp1y, double bp2x, double bp2y) {
		double d1 = getLengthLimitlessLineToPoint2D(bp1x, bp1y, ap1x, ap1y, ap2x, ap2y);
		double d2 = getLengthLimitlessLineToPoint2D(bp2x, bp2y, ap1x, ap1y, ap2x, ap2y);
		return (d1 >= 0f && d1 < EPS) && (d2 >= 0f && d2 < EPS);
	}

	/**
	 * 点が線分上にあるか調べる
	 *
	 * @param	px	検査する点のX座標
	 * @param	py	検査する点のY座標
	 * @param	p1x	線分の点１のX座標
	 * @param	p1y	線分の点１のY座標
	 * @param	p2x	線分の点２のX座標
	 * @param	p2y	線分の点２のY座標
	 */
	public static boolean isOnLimitLinePoint2D(double px, double py, double p1x, double p1y, double p2x, double p2y) {
		if (isSame2D(px, py, p1x, p1y) || isSame2D(px, py, p2x, p2y))
			return true;
		double r = getLengthLimitLineToPoint2D(px, py, p1x, p1y, p2x, p2y);
		if (r < EPS && r >= 0.0f)
			return true;
		return false;
	}

	/**
	 *  2直線が平行か調べる
	 * @param	ap1x	線１の点１のX座標
	 * @param	ap1y	線１の点１のY座標
	 * @param	ap2x	線１の点２のX座標
	 * @param	ap2y	線１の点２のY座標
	 * @param	bp1x	線２の点１のX座標
	 * @param	bp1y	線２の点１のY座標
	 * @param	bp2x	線２の点２のX座標
	 * @param	bp2y	線２の点２のY座標
	 */
	public static boolean isParallel2D(double ap1x, double ap1y, double ap2x, double ap2y, double bp1x, double bp1y, double bp2x, double bp2y) {
		//線１の方程式
		double a1 = getLineEquation2Da(ap1x, ap1y, ap2x, ap2y);
		double b1 = getLineEquation2Db(ap1x, ap1y, ap2x, ap2y);
		//線２の方程式
		double a2 = getLineEquation2Da(bp1x, bp1y, bp2x, bp2y);
		double b2 = getLineEquation2Db(bp1x, bp1y, bp2x, bp2y);

		double det = a1 * b2 - a2 * b1;
		if (Math.abs(det) < EPS)
			return true;
		return false;
	}

	/**
	 * 二直線から垂直にオフセットした交点を計算
	 */
	public static boolean get2LineOffsetCross(double x1, double y1, double x2, double y2, double x3, double y3, double ofs1, double ofs2, double[] tmp) {
		if (isParallel2D(x1, y1, x2, y2, x2, y2, x3, y3)) {
			getOffsetPoint2D(x2, y2, x3, y3, 0, ofs1, tmp);
			return true;
		}

		double p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y;
		getOffsetPoint2D(x1, y1, x2, y2, 0, ofs1, tmp);
		p1x = norm(tmp[0]);
		p1y = norm(tmp[1]);
		getOffsetPoint2D(x2, y2, x1, y1, 0, -ofs1, tmp);
		p2x = norm(tmp[0]);
		p2y = norm(tmp[1]);
		getOffsetPoint2D(x2, y2, x3, y3, 0, ofs2, tmp);
		p3x = norm(tmp[0]);
		p3y = norm(tmp[1]);
		getOffsetPoint2D(x3, y3, x2, y2, 0, -ofs2, tmp);
		p4x = norm(tmp[0]);
		p4y = norm(tmp[1]);
		return isCrossInfinity(p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y, tmp);
	}

   /**
     * 楕円と楕円中心から点へ繋がる線の交点を算出
     * 線は無限。
     *
     * @param x		楕円中心と繋がる点のX座標
     * @param y		楕円中心と繋がる点のY座標
     * @param cpx	楕円中心X
     * @param cpy	楕円中心Y
     * @param rh	楕円幅半径
     * @param rv	楕円高半径
     * @param tmp	交点
     */
	public static void getCrossPointEllipse(double x, double y, double cpx, double cpy, double rh, double rv, double[] tmp) {
		double ex,ey;
		if ((x - cpx) != 0) {
			double c = (y - cpy) / (x - cpx);
			ex = rv*rh / Math.sqrt(Math.abs(rh * rh + rv * c * rv * c));
			// 負の解
			if ((x - cpx) < 0)
				ex *= -1;
			ey = ex*c;
		} else {
			ex = 0;
			ey = ((y - cpy) >= 0) ? rh : -rh;
		}
//		tmp[0] = (int)(ex + cpx);
//		tmp[1] = (int)(ey + cpy);
		tmp[0] = (ex + cpx);
		tmp[1] = (ey + cpy);
	}

	/**
	 * 楕円度の取得
	 * @param rh
	 * @param rv
	 * @param angl
	 * @return
	 */
	public static double getEllipseAngl(double rh, double rv, double angl) {
       	double angle = Math.atan((rh / rv * Math.tan(angl)));
    	int quad = getQuadrant(Math.cos(angl), Math.sin(angl), 0, 0, 100, 0);
    	switch (quad) {
    		case 1:
    			break;
    		case 2:
    			if (angle > 0)
    				angle *= -1;
    			angle = (Math.PI + angle);
    			break;
    		case 3:
    			if (angle < 0)
    				angle *= -1;
      			angle = (Math.PI + angle);
      			break;
    		case 4:
      			if (angle > 0)
    				angle *= -1;
      			angle = (Math.PI * 2 + angle);
      			break;
    	}
    	return angle;
	}

	/**
	 * 直線の方程式の切片Aを算出
	 *
	 * @param	p1x	線分の点1のx座標
	 * @param	p1y	線分の点1のY座標
	 * @param	p2x	線分の点2のx座標
	 * @param	p2y	線分の点2のY座標
	 *
	 * @return
	 */
	public static double getLineEquation2Da(double p1x, double p1y, double p2x, double p2y) {
		double xlk = p2x - p1x;
		double ylk = p2y - p1y;
		double rsq = xlk * xlk + ylk * ylk;

		double rinv = 1.f / Math.sqrt(rsq);
		return norm(-ylk * rinv);
	}

	/**
	 * 直線の方程式の切片Bを算出
	 *
	 * @param	p1x	線分の点1のx座標
	 * @param	p1y	線分の点1のY座標
	 * @param	p2x	線分の点2のx座標
	 * @param	p2y	線分の点2のY座標
	 *
	 * @return
	 */
	public static double getLineEquation2Db(double p1x, double p1y, double p2x, double p2y) {
		double xlk = p2x - p1x;
		double ylk = p2y - p1y;
		double rsq = xlk * xlk + ylk * ylk;

		double rinv = 1.f / Math.sqrt(rsq);
		return norm(xlk * rinv);
	}

	/**
	 * 直線の方程式の切片Cを算出
	 *
	 * @param	p1x	線分の点1のx座標
	 * @param	p1y	線分の点1のY座標
	 * @param	p2x	線分の点2のx座標
	 * @param	p2y	線分の点2のY座標
	 *
	 * @return
	 */
	public static double getLineEquation2Dc(double p1x, double p1y, double p2x, double p2y) {
		double xlk = p2x - p1x;
		double ylk = p2y - p1y;
		double rsq = xlk * xlk + ylk * ylk;

		double rinv = 1.f / Math.sqrt(rsq);
		return norm((p1x * p2y - p2x * p1y) * rinv);
	}

	/**
	 * 丸める
	 */
	protected static double norm(double f) {
		return (Math.round(f * 1e8) / 1e8);
	}



	//---> 20030910 add start Fujita
	//     V2 -> V3 での PostMath 廃止に伴い、以下３メソッドを PostMath から移行
	/**
	 * メソッド名　 : 配列の平均化？<BR>
	 * 処理内容　　 : <BR>
	 * 事前条件 　　: <BR>
	 * 事後条件　　 : <BR>
	 *
	 * @param         vt 配列
	 * @return        標準偏差？
	 * @exception     なし
	 * @since         2003/09/10
	 * @author        fujita
	 * @version       2003/09/10
	 */
	public static double  normalizeVector(double vt[]) {
		double EPS = 1.0e-6f;
		double vector_length;

		vector_length =  Math.sqrt( vt[0] * vt[0] + vt[1] * vt[1] + vt[2] * vt[2] );
		if( vector_length > EPS )
		{
		   double inv_vector_length = (1.0/vector_length);
		   vt[0] *= inv_vector_length;
		   vt[1] *= inv_vector_length;
		   vt[2] *= inv_vector_length;
		 }
		 return vector_length;
	}

	/**
	 * メソッド名　 : ３次元空間における２点間の距離の取得<BR>
	 * 処理内容　　 : ３次元空間において、a 点と b 点間の距離を算出し返す。<BR>
	 * 事前条件 　　: <BR>
	 * 事後条件　　 : <BR>
	 *
	 * @param         a ３次元空間での点
	 * @param         b ３次元空間での点
	 * @return        ２点間の距離
	 * @exception     なし
	 * @since         2003/09/10
	 * @author        fujita
	 * @version       2003/09/10
	 */
	public static double getDistanceOn3D(double a[], double b[]) {
	  double x, y, z;
	  double aq;

	  x = a[0] - b[0];
	  y = a[1] - b[1];
	  z = a[2] - b[2];

	  aq = x * x + y * y + z * z;

	  return (Math.sqrt(aq));
	}

	/**
	 * メソッド名　 : <BR>
	 * 処理内容　　 : <BR>
	 * 事前条件 　　: <BR>
	 * 事後条件　　 : <BR>
	 *
	 * @param         なし
	 * @return        なし
	 * @exception     なし
	 * @since         2003/09/10
	 * @author        fujita
	 * @version       2003/09/10
	 */
	public static void calcOuterProduct(double a[], double b[], double c[]) {
		c[0] = a[1] * b[2] - a[2] *b[1];
		c[1] = a[2] * b[0] - a[0] *b[2];
		c[2] = a[0] * b[1] - a[1] *b[0];
	}
	//<--- add end
}
