/*
 * Copyright (c) 2008-2011 KITec Inc,.. All rights reserved.
 */
package jp.kitec.lib.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Stack;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.kitec.lib.annotation.Reflectable;



/**
 * クラス情報を表現するクラス
 *
 * @author $Author$
 * @version $Revision$ $Date::                           $
 */
public class ClassInfo {

	//------------------------------------------------------------------
	//- types
	//------------------------------------------------------------------

	private static class TypeInfo {
		public String type;
		public String name;
		public String reflectName;
		public Class<?>[] paramTypes;
	}



	//------------------------------------------------------------------
	//- fields
	//------------------------------------------------------------------

	/** ログ */
	@SuppressWarnings("unused")
	private static final Log _log = LogFactory.getLog(ClassInfo.class);

	/** クラス型 */
	private Class<?> mClazz = null;

	/** スーパークラス型 */
	private Class<?> mSuperclass = null;

	/** インターフェースセット */
	private Set<Class<?>> mInterfaceSet = new LinkedHashSet<Class<?>>();

	/** アノテーションセット */
	private Set<Annotation> mAnnotationSet = new HashSet<Annotation>();

	/** コンストラクタマップ */
	private Map<TypeInfo, Constructor<?>> mConstructorMap = new LinkedHashMap<TypeInfo, Constructor<?>>();

	/** メソッドマップ */
	private Map<TypeInfo, Method> mMethodMap = new LinkedHashMap<TypeInfo, Method>();

	/** フィールドマップ */
	private Map<TypeInfo, Field> mFieldMap = new LinkedHashMap<TypeInfo, Field>();

	/** ジェネリックスマップ */
	private Map<TypeInfo, Class<?>> mGenericsClassMap = new LinkedHashMap<TypeInfo, Class<?>>();

	/** ジェネリックス型リスト */
	private List<Class<?>> mGenericParameterTypeList = new ArrayList<Class<?>>();



	//------------------------------------------------------------------
	//- constructors
	//------------------------------------------------------------------

	/**
	 * コンストラクタ（外部new禁止）
	 */
	public ClassInfo(Class<?> clazz) {
		this.mClazz = clazz;
	}



	//-----------------------------------------------------------------------
	//- methods
	//-----------------------------------------------------------------------

	/**
	 * クラスの型を取得する。
	 *
	 * @return 型
	 */
	public Class<?> getType() {
		return this.mClazz;
	}

	/**
	 * 型が配列の場合、配列の型を取得する。
	 *
	 * @return 配列の型
	 */
	public Class<?> getArrayType() {
		if (!mClazz.isArray()) return null;

		Class<?> clazz = null;
		try {
			String arrayClazzName = mClazz.getName();
			String clazzName = arrayClazzName.substring(2, arrayClazzName.length() - 1);
			clazz = Class.forName(clazzName);
		} catch (ClassNotFoundException e) {
		}
		return clazz;
	}

	/**
	 * ジェネリックス型情報を取得する。
	 *
	 * @return ジェネリックス型コレクション
	 */
	public Collection<Class<?>> getGenericParameterTypes() {
		if (this.mGenericParameterTypeList.isEmpty()) this.initGenericParameterTypeList();
		return this.mGenericParameterTypeList;
	}

	/**
	 * スーパークラス型を取得する。
	 *
	 * @return スーパークラス型
	 */
	public Class<?> getSuperclass() {
		if (this.mSuperclass == null) this.initSuperclass();
		return this.mSuperclass;
	}

	/**
	 * インターフェース型情報を取得する。<br/>
	 *
	 * @return インターフェース型コレクション
	 */
	public Collection<Class<?>> getInterfaces() {
		if (this.mInterfaceSet.isEmpty()) this.initInterfaceSet();
		return this.mInterfaceSet;
	}

	/**
	 * アノテーション情報を取得する。<br/>
	 *
	 * @return アノテーションコレクション
	 */
	public Collection<Annotation> getAnnotations() {
		if (this.mAnnotationSet.isEmpty()) this.initAnnotationSet();
		return this.mAnnotationSet;
	}

	/**
	 * 指定のコンストラクタを取得します。<br/>
	 * private/protected/packageのコンストラクタも取得できます。<br/>
	 *
	 * @param paramTypes コンストラクタのパラメータ型
	 * @return コンストラクタ
	 */
	public Constructor<?> getConstructor(Class<?>... paramTypes) {
		Constructor<?> result = null;
		for (TypeInfo typeInfo: this.getConstructorMap().keySet()) {
			if (!matchTypeInfo(typeInfo, "constructor", null, paramTypes)) continue;
			result = this.getConstructorMap().get(typeInfo);
		}
		return result;
	}

	/**
	 * 全コンストラクタを取得する。
	 *
	 * @return コンストラクタコレクション
	 */
	public Collection<Constructor<?>> getConstructors() {
		return this.getConstructorMap().values();
	}

	/**
	 * 指定のメソッドを取得します。<br/>
	 * private/protected/package/staticのメソッドも取得できます。<br/>
	 *
	 * @param methodName メソッド名
	 * @param paramTypes メソッドのパラメータ型
	 * @return メソッド
	 */
	public Method getMethod(String methodName, Class<?>... paramTypes) {
		if (StringUtil.isEmpty(methodName)) return null;

		Method result = null;
		for (TypeInfo typeInfo: this.getMethodMap().keySet()) {
			if (!matchTypeInfo(typeInfo, "method", methodName, paramTypes)) continue;
			result = this.getMethodMap().get(typeInfo);
		}
		return result;
	}

	/**
	 * 全メソッドを取得する。
	 *
	 * @return メソッドコレクション
	 */
	public Collection<Method> getMethods() {
		return this.getMethodMap().values();
	}

	/**
	 * 指定のフィールドを取得する。<br/>
	 * private/protected/package/staticのフィールドも取得できます。<br/>
	 *
	 * @param fieldName フィールド名
	 * @return フィールド
	 */
	public Field getField(String fieldName) {
		if (StringUtil.isEmpty(fieldName)) return null;

		Field result = null;
		for (TypeInfo typeInfo: this.getFieldMap().keySet()) {
			if (!matchTypeInfo(typeInfo, "field", fieldName)) continue;
			result = this.getFieldMap().get(typeInfo);
		}
		return result;
	}

	/**
	 * 全フィールドを取得する。
	 *
	 * @return フィールドコレクション
	 */
	public Collection<Field> getFields() {
		return this.getFieldMap().values();
	}

	/**
	 * 指定のフィールドを取得する。<br/>
	 * private/protected/package/staticのフィールドも取得できます。<br/>
	 *
	 * @param name フィールド名
	 * @return フィールド
	 */
	public Class<?> getGenericsClass(String name) {
		if (StringUtil.isEmpty(name)) return null;

		Class<?> result = null;
		for (TypeInfo typeInfo: this.getGenericsClassMap().keySet()) {
			if (!matchTypeInfo(typeInfo, "generics_class", name)) continue;
			result = this.getGenericsClassMap().get(typeInfo);
		}
		return result;
	}

	/**
	 * 全フィールドを取得する。
	 *
	 * @return フィールドコレクション
	 */
	public Collection<Class<?>> getGenericsClasses() {
		return this.getGenericsClassMap().values();
	}

	/**
	 * コンストラクタが存在するかチェックする。<br/>
	 *
	 * @param paramTypes パラメータ型配列
	 * @return 存在する場合true。
	 */
	public boolean existConstructor(final Class<?>... paramTypes) {
		return this.getConstructor(paramTypes) != null;
	}

	/**
	 * メソッドが存在するかチェックする。<br/>
	 *
	 * @param methodName メソッド名
	 * @param paramTypes パラメータ型配列
	 * @return 存在する場合true。
	 */
	public boolean existMethod(final String methodName, final Class<?>... paramTypes) {
		return this.getMethod(methodName, paramTypes) != null;
	}

	/**
	 * フィールドが存在するかチェックする。<br/>
	 *
	 * @param fieldName フィールド名
	 * @return 存在する場合true。
	 */
	public boolean existField(final String fieldName) {
		return this.getField(fieldName) != null;
	}

	/**
	 * コンストラクタマップを取得する。<br/>
	 * 未初期化の場合初期化する。<br/>
	 */
	private Map<TypeInfo, Constructor<?>> getConstructorMap() {
		if (this.mConstructorMap.isEmpty()) this.initConstructorMap();
		return this.mConstructorMap;
	}

	/**
	 * メソッドマップを取得する。<br/>
	 * 未初期化の場合初期化する。<br/>
	 */
	private Map<TypeInfo, Method> getMethodMap() {
		if (this.mMethodMap.isEmpty()) this.initMethodMap();
		return this.mMethodMap;
	}

	/**
	 * フィールドマップを取得する。<br/>
	 * 未初期化の場合初期化する。<br/>
	 */
	private Map<TypeInfo, Field> getFieldMap() {
		if (this.mFieldMap.isEmpty()) this.initFieldMap();
		return this.mFieldMap;
	}

	/**
	 * フィールドマップを取得する。<br/>
	 * 未初期化の場合初期化する。<br/>
	 */
	private Map<TypeInfo, Class<?>> getGenericsClassMap() {
		if (this.mGenericsClassMap.isEmpty()) this.initGenericsClassMap();
		return this.mGenericsClassMap;
	}

	/**
	 * スーパークラス型を初期化する。
	 */
	private void initSuperclass() {
		this.mSuperclass = this.mClazz.getSuperclass();
	}

	/**
	 * インターフェースセットを初期化する。
	 */
	private void initInterfaceSet() {
		for (Class<?> iface: this.mClazz.getInterfaces()) {
			this.mInterfaceSet.add(iface);
		}
	}

	/**
	 * アノテーションセットを初期化する。
	 */
	private void initAnnotationSet() {
		this.mAnnotationSet.addAll(this.initAnnotationSet(this.mClazz));
	}

	/**
	 * アノテーションセットを初期化する。
	 */
	private Set<Annotation> initAnnotationSet(Class<?> clazz) {
		Set<Annotation> result = new HashSet<Annotation>();
		if (clazz == null) return result;

		Class<?> superClazz = clazz.getSuperclass();
		if (superClazz != null) {
			result.addAll(this.initAnnotationSet(clazz.getSuperclass()));
		}
		for (Annotation annotation: clazz.getAnnotations()) {
			result.add(annotation);
		}
		return result;
	}

	/**
	 * コンストラクタマップを初期化する。
	 */
	private void initConstructorMap() {
		for (Constructor<?> constructor: this.mClazz.getDeclaredConstructors()) {
			TypeInfo typeInfo = createTypeInfo("constructor", null, null, constructor.getParameterTypes());
			this.mConstructorMap.put(typeInfo, constructor);
		}
	}

	/**
	 * メソッドマップを初期化する。
	 */
	private void initMethodMap() {
		this.mMethodMap.putAll(this.initMethodMap(this.mClazz));
	}

	/**
	 * メソッドマップを初期化する。
	 */
	private Map<TypeInfo, Method> initMethodMap(Class<?> clazz) {
		Map<TypeInfo, Method> result = new LinkedHashMap<TypeInfo, Method>();
		if (clazz == null) return result;

		Class<?> superClazz = clazz.getSuperclass();
		if (superClazz != null) {
			result.putAll(this.initMethodMap(clazz.getSuperclass()));
		}
		for (Method method: clazz.getDeclaredMethods()) {
			Reflectable annoReflectable = method.getAnnotation(Reflectable.class);
			String reflectName = null;
			if (annoReflectable != null) reflectName = annoReflectable.name();

			TypeInfo typeInfo = createTypeInfo("method", method.getName(), reflectName, method.getParameterTypes());
			result.put(typeInfo, method);
		}
		return result;
	}

	/**
	 * フィールドマップを初期化する。
	 */
	private void initFieldMap() {
		this.mFieldMap.putAll(this.initFieldMap(this.mClazz));
	}

	/**
	 * フィールドマップを初期化する。
	 */
	private Map<TypeInfo, Field> initFieldMap(Class<?> clazz) {
		Map<TypeInfo, Field> result = new LinkedHashMap<TypeInfo, Field>();
		if (clazz == null) return result;

		Class<?> superClazz = clazz.getSuperclass();
		if (superClazz != null) {
			result.putAll(this.initFieldMap(clazz.getSuperclass()));
		}
		for (Field field: clazz.getDeclaredFields()) {
			Reflectable annoReflectable = field.getAnnotation(Reflectable.class);
			String reflectName = null;
			if (annoReflectable != null) reflectName = annoReflectable.name();

			TypeInfo typeInfo = createTypeInfo("field", field.getName(), reflectName);
			result.put(typeInfo, field);
		}
		return result;
	}

	/**
	 * ジェネリックスマップを初期化する。
	 */
	private void initGenericsClassMap() {
		initFieldMap();
		this.mGenericsClassMap.putAll(this.initGenericsClassMap(this.mClazz));
	}

	/**
	 * ジェネリックスマップを初期化する。
	 */
	private Map<TypeInfo, Class<?>> initGenericsClassMap(Class<?> clazz) {
		Map<TypeInfo, Class<?>> result = new LinkedHashMap<TypeInfo, Class<?>>();
		if (clazz == null) return result;

		for (Entry<TypeInfo, Field> fieldEntry: getFieldMap().entrySet()) {
			Field field = fieldEntry.getValue();
			Reflectable annoReflectable = field.getAnnotation(Reflectable.class);
			String reflectName = null;
			if (annoReflectable != null) reflectName = annoReflectable.name();

			Type genType = field.getGenericType();
			if (genType instanceof TypeVariable) {
				TypeVariable<?> typev = (TypeVariable<?>) genType;
				TypeInfo typeInfo = createTypeInfo("generics_class", field.getName(), reflectName);
				Class<?> genClazz = findGenericType(clazz, typev.getName());
				result.put(typeInfo, genClazz);
			}
		}
		return result;
	}

	/**
	 * パラメータ型リストを初期化する。
	 */
	private void initGenericParameterTypeList() {
		for (Type type: this.mClazz.getGenericInterfaces()) {
			if (!(type instanceof ParameterizedType)) continue;

			ParameterizedType paramType = (ParameterizedType)type;
			for (Type actualType: paramType.getActualTypeArguments()) {
				this.mGenericParameterTypeList.add((Class<?>)actualType);
			}
		}
	}

	/**
	 * 型情報を生成する。
	 *
	 * @param type 種別名
	 * @param name 名称
	 * @param reflectName リフレクション名称
	 * @param paramTypes パラメータ型
	 * @return 型情報
	 */
	private TypeInfo createTypeInfo(String type, String name, String reflectName, Class<?>... paramTypes) {
		TypeInfo typeInfo = new TypeInfo();
		typeInfo.type = type;
		typeInfo.name = name;
		typeInfo.reflectName = reflectName;
		typeInfo.paramTypes = paramTypes;
		return typeInfo;
	}

	/**
	 * 型情報が一致しているかチェックする。
	 *
	 * @param typeInfo 型情報
	 * @param type 種別名
	 * @param name 名称
	 * @return 一致していればtrue。
	 */
	private boolean matchTypeInfo(TypeInfo typeInfo, String type, String name) {
		type = StringUtil.emptyToZeroString(type);
		name = StringUtil.emptyToZeroString(name);
		String typeInfoType = StringUtil.emptyToZeroString(typeInfo.type);
		String typeInfoName = StringUtil.emptyToZeroString(typeInfo.name);
		String typeInfoReflectName = StringUtil.emptyToZeroString(typeInfo.reflectName);
		if (!typeInfoType.equals(type)) return false;
		if (!typeInfoName.equals(name) && !typeInfoReflectName.equals(name)) return false;
		return true;
	}

	/**
	 * 型情報が一致しているかチェックする。
	 *
	 * @param typeInfo 型情報
	 * @param type 種別名
	 * @param name 名称
	 * @param paramTypes パラメータ型
	 * @return 一致していればtrue。
	 */
	private boolean matchTypeInfo(TypeInfo typeInfo, String type, String name, Class<?>[] types) {
		if (!matchTypeInfo(typeInfo, type, name)) return false;
		if (!matchParameterTypes(typeInfo.paramTypes, types)) return false;
		return true;
	}

	/**
	 * ２つのパラメータ配列が一致しているかチェックする。
	 *
	 * @param types1 パラメータ配列1
	 * @param types2 パラメータ配列2
	 * @return 一致していればtrue。
	 */
	private boolean matchParameterTypes(Class<?>[] types1, Class<?>[] types2) {
		if (types1 == null && types2 == null) return true;
		if (types1 == null && types2 != null) return false;
		if (types1 != null && types2 == null) return false;
		if (types1.length != types2.length) return false;
		if (types1.length == 0 && types2.length == 0) return true;

		for (int i = 0; i < types1.length; i++) {
			if (!types1[i].isAssignableFrom(types2[i])) return false;
		}
		return true;
	}

	/**
	 * 渡された型から継承階層を登って、
	 * 指定の親の型の指定の名前のジェネリクス型パラメータが
	 * 継承の過程で何型で具現化されているかを走査して返す。
	 *
	 * @param clazz 走査開始する型
	 * @param targetTypeName 何型で具現化されたを確認したい型パラメータのプレースホルダ名
	 * @return 具現化された型
	 */
	private <T> Class<T> findGenericType(Class<?> clazz, String targetTypeName) {
		if (clazz.getSuperclass() == null) return null;

		Stack<Class<?>> stack = new Stack<Class<?>>();
		stack.push(clazz);
		return findGenericType(clazz.getSuperclass(), targetTypeName, stack);
	}

	/**
	 * 型パラメータの具象型取得の実装。再帰処理される。
	 *
	 * @param clazz 現在の走査対象型
	 * @param targetTypeName 現在の走査対象のジェネリクス型パラメータ名
	 * @param stack 現在の走査対象型以下の継承階層が積まれたStack
	 * @return 該当型パラメータの具現化された型
	 */
	@SuppressWarnings("unchecked")
	private <T> Class<T> findGenericType(Class<?> clazz, String targetTypeName, Stack<Class<?>> stack) {
		if (clazz == null) return null;
		Class<?> superClazz = clazz.getSuperclass();
		if (superClazz == null) return null;

		TypeVariable<? extends Class<?>>[] superGenTypeArray = superClazz.getTypeParameters();
		if (superGenTypeArray.length == 0) return null;

		// 走査対象の型パラメータの名称(Tなど)から宣言のインデックスを取得
		int index = 0;
		boolean matchGenType = false;
		for (TypeVariable<? extends Class<?>> type : superGenTypeArray) {
			if (targetTypeName.equals(type.getName())) {
				matchGenType = true;
				break;
			}
			index++;
		}
		if (!matchGenType) return null;

		// 走査対象の型パラメータが何型とされているのかを取得
		ParameterizedType type = (ParameterizedType) clazz.getGenericSuperclass();
		Type y = type.getActualTypeArguments()[index];

		// 具象型で継承されている場合
		if (y instanceof Class) {
			return (Class<T>) y;
		}
		// ジェネリックパラメータの場合
		if (y instanceof TypeVariable) {
			TypeVariable<Class<?>> tv = (TypeVariable<Class<?>>) y;
			// 再帰して同名の型パラメータを継承階層を下りながら解決を試みる
			Class<?> sub = stack.pop();
			return findGenericType(sub, tv.getName(), stack);
		}
		// ジェネリック型パラメータを持つ型の場合
		if (y instanceof ParameterizedType) {
			ParameterizedType pt = (ParameterizedType) y;
			return (Class<T>) pt.getRawType();
		}
		throw new IllegalArgumentException("予期せぬ型 : " + y.toString() + " (" + y.getClass() + ")");
	}



} // end-class
