/*
 * Created on 2004/08/02
 */
package jp.kitec.lib.kwt;

import java.io.Serializable;

import jp.kitec.lib.util.tree.ObjectFolder;
import jp.kitec.lib.util.tree.ObjectNode;


/**
 * 線種指定クラス
 * @author kawae
 */
public class KLineType implements Serializable {
	private static final long serialVersionUID = 1L;
	public static final String GEOMNAME = "LINETYPE";

	/* 描画タイプ */
	/** 実座標系での幅指定 */
	public static final short TYPE_DRAWING = 0;

	/** デバイス座標系での幅指定 */
	public static final short TYPE_DEVICE = 1;

	public static enum Cap {
		BUTT, ROUND, SQUARE
	}

	public static enum Join {
		BEVEL, MITER, ROUND
	}

	/** 名称 */
	private String mName;

	/** ストローク */
	private float[] mStroke;

	/** 幅 */
	private float mWidth = 0;

	/** 幅指定オプション */
	private short mWidthType;

	/** 両端部オプション */
	private Cap mCap;

	/** 接合部オプション */
	private Join mJoin;

	/** 接合トリミングの制限 */
	private double mMiterLimit;

	@Deprecated
	public KLineType() {
	}

	/**
	 * コンストラクタ
	 * @param name
	 * @param str
	 * @param w
	 */
	public KLineType(String name, float[] str, float w) {
		this(name, str, w, TYPE_DRAWING);
	}

	/**
	 * コンストラクタ
	 * @param name
	 * @param str
	 * @param w
	 * @param widType
	 */
	public KLineType(String name, float[] str, float w, short widType) {
		this(name, str, w, widType, Cap.BUTT, Join.MITER, 1);
	}

	/**
	 * コンストラクタ
	 * @param name
	 * @param str
	 * @param w
	 * @param widType
	 */
	public KLineType(String name, float[] str, float w, short widType, Cap cap, Join join, double miterLimit) {
		if (cap == null || join == null)
			throw new NullPointerException();
		mName = name;
		mStroke = str;
		mWidth = w;
		mWidthType = widType;
		mCap = cap;
		mJoin = join;
		mMiterLimit = miterLimit;
	}

	/**
	 * 名称を返す
	 * @return
	 */
	public String getName() {
		return mName;
	}

	/**
	 * ストロークを返す
	 * @return
	 */
	public float[] getStroke() {
		return mStroke;
	}

	/**
	 * 幅を返す
	 * @return
	 */
	public float getWidth() {
		return mWidth;
	}

	/**
	 * 幅指定オプションを返す
	 * @return
	 */
	public short getWidthType() {
		return mWidthType;
	}

	/**
	 * 両端部オプションを返す
	 * @return
	 */
	public Cap getCap() {
		return mCap;
	}

	/**
	 * 接合部オプションを返す
	 * @return
	 */
	public Join getJoin() {
		return mJoin;
	}

	/**
	 * 接合トリミングの制限を返す
	 * @return
	 */
	public double getMiterLimit() {
		return mMiterLimit;
	}

	/**
	 * 現在のKLineTypeに新しいストロークを適用した複製を返す
	 * @param stroke
	 * @return
	 */
	public KLineType deriveLineType(float[] stroke) {
		return new KLineType(mName, stroke, mWidth, mWidthType, mCap, mJoin, mMiterLimit);
	}

	/**
	 * 現在のKLineTypeに新しい幅を適用した複製を返す
	 * @param width
	 * @return
	 */
	public KLineType deriveLineType(float width) {
		return new KLineType(mName, mStroke, width, mWidthType, mCap, mJoin, mMiterLimit);
	}

	/**
	 * 現在のKLineTypeに新しい幅指定オプションを適用した複製を返す
	 * @param widthType
	 * @return
	 */
	public KLineType deriveLineType(short widthType) {
		return new KLineType(mName, mStroke, mWidth, widthType, mCap, mJoin, mMiterLimit);
	}

	/**
	 * 現在のKLineTypeに新しい両端部オプションを適用した複製を返す
	 * @param cap
	 * @return
	 */
	public KLineType deriveLineType(Cap cap) {
		return new KLineType(mName, mStroke, mWidth, mWidthType, cap, mJoin, mMiterLimit);
	}

	/**
	 * 現在のKLineTypeに新しい接合部オプションを適用した複製を返す
	 * @param join
	 * @return
	 */
	public KLineType deriveLineType(Join join) {
		return new KLineType(mName, mStroke, mWidth, mWidthType, mCap, join, mMiterLimit);
	}

	/**
	 * 現在のKLineTypeに新しい接合トリミング制限を適用した複製を返す
	 * @param miterLimit
	 * @return
	 */
	public KLineType deriveLineType(double miterLimit) {
		return new KLineType(mName, mStroke, mWidth, mWidthType, mCap, mJoin, miterLimit);
	}

	public void save(ObjectFolder savenode) {
		savenode.setName(GEOMNAME);
		//名前
		if (mName != null)
			savenode.addNode("name", mName);
		//ストローク
		if (mStroke != null) {
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < mStroke.length; i++) {
				sb.append(Float.toString(mStroke[i]));
				if (i != mStroke.length - 1)
					sb.append(",");
			}
			String s = sb.toString();
			if (s.length() > 0)
				savenode.addNode("stroke", s);
		}
		//線幅
		savenode.addNode("width", mWidth);

		savenode.addNode("cap", mCap.name());
		savenode.addNode("join", mJoin.name());
		savenode.addNode("miterlimit", mMiterLimit);
	}

	@Deprecated
	public static void readData(ObjectFolder of, KLineType l) {
		ObjectNode on;
		if ((on = of.getNode("name")) != null)
			l.mName = on.getString();
		if ((on = of.getNode("stroke")) != null) {
			String str = on.getString();
			String[] st = str.split(",");
			l.mStroke = new float[st.length];
			for (int i = 0; i < l.mStroke.length; i++)
				l.mStroke[i] = Float.parseFloat(st[i]);
		}
		if ((on = of.getNode("width")) != null)
			l.mWidth = on.getFloat();
		if ((on = of.getNode("cap")) != null)
			l.mCap = Cap.valueOf(on.getString());
		if ((on = of.getNode("join")) != null)
			l.mJoin = Join.valueOf(on.getString());
		if ((on = of.getNode("miterlimit")) != null)
			l.mMiterLimit = on.getFloat();
	}

	public static KLineType readData(ObjectFolder of) {
		KLineType l = new KLineType(null, null, 0);
		readData(of, l);
		return l;
	}
}
